package cn.quantgroup.xyqb.service.session.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.session.SessionValue;
import cn.quantgroup.xyqb.service.session.ISessionService;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * Created by 11 on 2016/12/28.
 */
@Slf4j
@Service
public class SessionServiceImpl implements ISessionService {

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Override
  public AuthBean createSession(Long channelId, Long createdFrom, String appChannel, User user, Merchant merchant) {
    AuthBean authBean = new AuthBean();
    LoginProperties properties = new LoginProperties();
    properties.setAppChannel(appChannel);
    properties.setMerchantName(merchant.getName());
    properties.setChannelId(channelId);
    properties.setCreatedFrom(createdFrom);
    //找到用户
    String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties);
    // String sessionId = sessionService.findSessionIdByUserIdAndMerchant(user.getId(), merchant);
    if (org.apache.commons.lang.StringUtils.isNotEmpty(sessionId)) {
      SessionStruct sessionStruct = findSessionBySessionId(sessionId);
      sessionStruct.setAttribute("channelId", String.valueOf(channelId));
      sessionStruct.setAttribute("createdFrom", String.valueOf(createdFrom));
      sessionStruct.setAttribute("appChannel", String.valueOf(appChannel));
      authBean.setPhoneNo(user.getPhoneNo());
      authBean.setToken(sessionStruct.getSid());
      persistSession(sessionStruct.getSid(), sessionStruct.getValues());
      log.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", createdFrom, user.getPhoneNo(), appChannel);
      return authBean;
    }
    SessionStruct sessionStruct = createSessionAndPersist(user, properties);
    authBean.setPhoneNo(user.getPhoneNo());
    authBean.setToken(sessionStruct.getSid());
    log.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", createdFrom, user.getPhoneNo(), appChannel);
    return authBean;
  }

  @Override
  public SessionStruct createSessionAndPersist(User user, LoginProperties properties) {
    SessionStruct sessionStruct;
    //获取sessionid
    String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties);
    if (StringUtils.length(sessionId) == 36) {
      sessionStruct = findSessionBySessionId(sessionId);
      if (sessionStruct == null) {
        sessionStruct = newSession(user, properties);
      } else {
        sessionStruct.getValues().setLoginProperties(properties);
      }
      persistSession(sessionStruct.getSid(), sessionStruct.getValues());
    } else {
      sessionStruct = newSession(user, properties);
      persistSession(sessionStruct.getSid(), sessionStruct.getValues());
    }
    return sessionStruct;
  }

  @Override
  public String findSessionIdByUserIdLoginProperties(Long userId, LoginProperties properties) {
    return stringRedisTemplate.opsForValue().get(generateLoginPropertiesKey(userId, properties));
  }

  private String generateLoginPropertiesKey(Long userId, LoginProperties properties) {
    if ("baitiao".equals(properties.getMerchantName())) {
      return Constants.Session.USER_SESSION_ID_CACHE + ":" + String.valueOf(userId) + ":" + properties.getMerchantName() + ":" + properties.getCreatedFrom();
    } else {
      return Constants.Session.USER_SESSION_ID_CACHE + ":" + String.valueOf(userId) + ":" + properties.getMerchantName();
    }
  }

  @Override
  public String findSessionValueBySessionId(String sessionId) {
    String result = stringRedisTemplate.opsForValue().get(Constants.Session.USER_SESSION_CACHE + sessionId);
    return StringUtils.defaultString(result, "");
  }

  @Override
  public SessionStruct newSession(User user, LoginProperties loginProperties) {
    Timestamp now = new Timestamp(System.currentTimeMillis());
    SessionStruct sessionStruct = new SessionStruct();
    SessionValue sessionValue = new SessionValue();
    sessionStruct.setSid(UUID.randomUUID().toString());
    sessionValue.setCreatedAt(now);
    sessionValue.setLastAccessTime(now);
    sessionValue.setUser(user);
    sessionValue.setLoginProperties(loginProperties);
    Map<String, String> values = new HashMap<>();
    sessionValue.setValues(values);
    sessionStruct.setValues(sessionValue);
    return sessionStruct;
  }

  @Override
  public void persistSession(String token, SessionValue sessionValue) {
    Timestamp current = new Timestamp(System.currentTimeMillis());
    sessionValue.setLastAccessTime(current);
    String json = JSON.toJSONString(sessionValue);
    stringRedisTemplate.opsForValue().set(Constants.Session.USER_SESSION_CACHE + token, json,
        Constants.Session.ONE_DAY, TimeUnit.SECONDS);
    String key = generateLoginPropertiesKey(sessionValue.getUser().getId(), sessionValue.getLoginProperties());
    stringRedisTemplate.opsForValue().set(key, token, Constants.Session.ONE_DAY, TimeUnit.SECONDS);
  }

  @Override
  public SessionStruct findSessionBySessionId(String sessionId) {
    String sessionValue = findSessionValueBySessionId(sessionId);
    if (StringUtils.isEmpty(sessionValue)) {
      return null;
    }
    try {
      SessionValue value = JSON.parseObject(sessionValue, SessionValue.class);
      if (null == value) {
        return null;
      }
      SessionStruct struct = new SessionStruct();
      struct.setSid(sessionId);
      struct.setValues(value);
      return struct;
    } catch (Exception ex) {
      return null;
    }

  }

  @Override
  public void deleteByUserId(long userId) {
    String pattern = Constants.Session.USER_SESSION_ID_CACHE + ":" + String.valueOf(userId) + ":*";
    Set<String> keys = stringRedisTemplate.keys(pattern);
    if (!CollectionUtils.isEmpty(keys)) {
      log.info("删除用户userId={}的缓存信息,个数:{},keys={}", userId,
          keys.size(),
          Joiner.on(",").join(keys));
    }
    stringRedisTemplate.delete(keys);
  }
}
