package cn.quantgroup.xyqb.controller.external.user.center;

import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.user.*;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by 11 on 2017/3/22.
 */
@RestController
@RequestMapping("/user/center")
public class UserCenterController {

    @Autowired
    private UserCenterService userCenterService;
    @Autowired
    private IUserDetailService userDetailService;
    @Autowired
    private IUserExtInfoService userExtInfoService;
    @Autowired
    private IContactService contactService;
    @Autowired
    private IAddressService addressService;
    @Autowired
    private IUserService userService;

    private static final Logger LOGGER = LoggerFactory.getLogger(UserCenterController.class);

    /**
     * 用户中心首页，显示用户头像、昵称、姓名
     * @param phoneNo
     * @return
     */
    @RequestMapping("/index")
    public JsonResult userCenterIndex(String phoneNo) {
        Long userId = queryUserId(phoneNo);
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        Map<String, String> result = new HashMap<>();
        if(userAttached != null) {
            result.put("avatar", userAttached.getAvatar());
            result.put("nick", userAttached.getNick());
        }
        UserDetail userDetail = userDetailService.findByUserId(userId);
        if(userDetail != null) {
            result.put("name", userDetail.getName());
        }
        return JsonResult.buildSuccessResult(null, result);
    }

    /**
     * 上传后调用该接口保存用户头像
     * @param phoneNo 用户手机号
     * @param avatarUrl 上传后生成的头像地址
     * @return
     */
    @RequestMapping("/save/avatar")
    public JsonResult SaveUserAvatarAddr(String phoneNo, String avatarUrl) {
        /*if(null == phoneNo || userId == 0) {
            LOGGER.error("用户修改头像、昵称,用户id未获取到:{}", userId);
            return JsonResult.buildErrorStateResult("参数不完整,请校验参数.", null);
        }*/
        if(StringUtils.isEmpty(avatarUrl) || StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("参数不合法:avatarUrl:{}, phoneNo:{}", avatarUrl, phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        //查询到数据,直接更新头像和更新时间
        if(null != userAttached) {
            userAttached.setAvatar(avatarUrl);
            userAttached.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        } else {
            userAttached = createUserAttached(userId, avatarUrl, "");
        }
        UserAttached result = userCenterService.saveUserAttached(userAttached);
        if(null == userAttached) {
            LOGGER.error("保存用户头像昵称失败.");
            return JsonResult.buildErrorStateResult("信息保存失败,请稍后再试.", null);
        }
        return JsonResult.buildSuccessResult("保存成功", result);
    }

    /**
     * 根据用户手机号查询昵称.
     * @param phoneNo
     * @return
     */
    @RequestMapping("/queryNick")
    public JsonResult queryUserNick(String phoneNo) {
        if(StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        if(null != userAttached) {
            LOGGER.info("获取用户昵称:{}", userAttached.getNick());
            return JsonResult.buildSuccessResult(null, userAttached.getNick());
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 修改用户的昵称
     * @param phoneNo
     * @param nick
     * @return
     */
    @RequestMapping("/saveNick")
    public JsonResult saveUserNick(String phoneNo, String nick) {
        if(StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        if(null == userAttached) {
            userAttached = createUserAttached(userId, "", nick);
        } else {
            userAttached.setNick(nick);
            userAttached.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        }
        UserAttached result = userCenterService.saveUserAttached(userAttached);
        return JsonResult.buildSuccessResult(null, result.getNick());
    }

    /**
     * 查询用户是否实名认证.
     * @param phoneNo
     * @return
     */
    @RequestMapping("/query/verified")
    public JsonResult queryVerified(String phoneNo) {
        if(StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        UserDetail userDetail = userDetailService.findByUserId(userId);
        if(null != userDetail) {
            return JsonResult.buildSuccessResult(null, userDetail);
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 个人资料信息
     * @param phoneNo
     * @return
     */
    @RequestMapping("/personalData")
    public JsonResult personalData(String phoneNo) {
        if(StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
        List<Contact> contacts = contactService.findByUserId(userId);
        Map<String, Object> result = new HashMap<>();
        if(null != userExtInfo) {
            result.put("contacts", null != contacts && contacts.size() > 0 ? true : false);
            result.put("occupation", userExtInfo.getOccupationEnum().ordinal());
            result.put("education", userExtInfo.getEducationEnum().ordinal());
            result.put("income", userExtInfo.getIncomeRangeEnum().ordinal());
            //查询用户居住地信息
            result.put("marryStatus", userExtInfo.getMarryStatus().ordinal());
        }
        Address address = addressService.findByUserId(userId);
        if(null != address) {
            //用户住址信息,返回二级信息:
            result.put("address", address.getProvince() + " " + address.getCity());
        }
        return JsonResult.buildSuccessResult(null, result);
    }

    private UserAttached createUserAttached(Long userId, String avatar, String nick) {
        UserAttached userAttached = new UserAttached();
        Timestamp now = new Timestamp(System.currentTimeMillis());
        userAttached.setUserId(userId);
        userAttached.setAvatar(avatar);
        userAttached.setNick(nick);
        userAttached.setCreatedAt(now);
        userAttached.setUpdatedAt(now);
        return userAttached;
    }

    /**
     * 根据手机号查询用户id, nodejs调用接口无法获取到userId.
     * 所以增加该接口来查询用户id
     * @param phoneNo
     * @return
     */
    private Long queryUserId(String phoneNo) {
        User user = userService.findByPhoneInDb(phoneNo);
        return user.getId();
    }
}
