package cn.quantgroup.xyqb.service.user.impl;

import java.util.*;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.persistence.criteria.Predicate;

import org.apache.commons.lang.StringUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;

import cn.quantgroup.acolyte.buddhistscriptures.pojo.UserRealInfo;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.event.StatisticsEvent;
import cn.quantgroup.xyqb.model.Gender;
import cn.quantgroup.xyqb.model.IdCardInfo;
import cn.quantgroup.xyqb.repository.IUserDetailRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.vo.UserDetailVO;
import cn.quantgroup.xyqb.util.ValidationUtil;

/**
 * Created by 11 on 2016/12/29.
 */
@Slf4j
@Service
public class UserDetailServiceImpl implements IUserDetailService {
    @Autowired
    private IUserDetailRepository userDetailRepository;
    @Autowired
    private IUserRepository userRepository;
    @Autowired
    private IIdCardService idCardService;
    @Resource
    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public UserDetail findByUserId(Long userId) {
        return userDetailRepository.findByUserId(userId);
    }

    @Override
    public UserDetail saveUserDetail(UserDetail userDetail) throws DataIntegrityViolationException {
        UserDetail userDetail1 = userDetailRepository.save(userDetail);
        if(Objects.nonNull(userDetail)){
            // 发送实名登记统计消息
            UserRealInfo userRealInfo = new UserRealInfo(userDetail.getUserId(), userDetail.getName(), userDetail.getIdNo(), userDetail.getIdType().ordinal());
            applicationEventPublisher.publishEvent(new StatisticsEvent(this, userRealInfo));
        }
        return userDetail1;
    }

    @Override
    public UserDetail findByPhoneNo(String phoneNo) {
        return userDetailRepository.findByPhoneNo(phoneNo);
    }

    @Override
    public void updateUserQQ(Long userId, String qq) {
        userDetailRepository.updateUserQQ(qq, userId);
    }

    @Override
    public void updateUserEmail(Long userId, String email) {
        userDetailRepository.updateUserEmail(email, userId);
    }

    @Override
    public List<UserDetailVO> searchUserDetailList(String name, String phoneNo, String idNo) {
        List<UserDetail> details = userDetailRepository.findAll(getSpecification(name, phoneNo, idNo));
        Map<Long, User> userMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(details)) {
            List<Long> userIds = details.stream().map(UserDetail::getUserId).collect(Collectors.toList());
            List<User> users = userRepository.findAll((root, query, cb) -> {
                query.where(root.get("id").in(userIds));
                return query.getRestriction();
            });
            userMap = users.stream().collect(Collectors.toMap(User::getId, o -> o));
        }
        Map<Long, User> finalUserMap = userMap;
        List<UserDetailVO> userDetailVOS = details.stream().map(o -> fromUserDetailAndUserMap(o, finalUserMap)).collect(Collectors.toList());
        return userDetailVOS;
    }

    @Override
    public List<UserDetail> findByUserIdIn(List<Long> userIds) {

        return userDetailRepository.findAll((root, query, cb) -> {
            query.where(root.get("userId").in(userIds));
            return query.getRestriction();
        });

    }

    @Override
    public List<UserDetail> findByPhoneNos(List<String> phoneNos) {
        return userDetailRepository.findAll((root, query, cb) -> {
            query.where(root.get(Constants.PHONE_NO).in(phoneNos));
            return query.getRestriction();
        });
    }

    @Override
    public List<UserDetail> findByIdnos(List<String> idnos) {
        return userDetailRepository.findAll((root, query, cb) -> {
            query.where(root.get("idNo").in(idnos));
            return query.getRestriction();
        });
    }

    private Specification<UserDetail> getSpecification(String name, String phoneNo, String idNo) {
        List<Predicate> list = new ArrayList<>();
        Specification<UserDetail> specification = (root, criteriaQuery, criteriaBuilder) -> {
            if(ValidationUtil.validatePhoneNo(phoneNo)){
                list.add(criteriaBuilder.equal(root.get(Constants.PHONE_NO).as(String.class), phoneNo));
            }else if (StringUtils.isNotBlank(phoneNo)) {
                list.add(criteriaBuilder.like(root.get(Constants.PHONE_NO).as(String.class), phoneNo.concat("%")));
            }
            if (StringUtils.isNotBlank(idNo)) {
                if(Objects.equals(Constants.IDNO_LENGTH, idNo.length())){
                    list.add(criteriaBuilder.equal(root.get("idNo").as(String.class), idNo));
                }else{
                    list.add(criteriaBuilder.like(root.get("idNo").as(String.class), idNo.concat("%")));
                }
            }
            if (StringUtils.isNotBlank(name)) {
                list.add(criteriaBuilder.equal(root.get("name").as(String.class), name));
            }
            Predicate[] p = new Predicate[list.size()];
            return criteriaBuilder.and(list.toArray(p));
        };
        return specification;
    }

    private UserDetailVO fromUserDetailAndUserMap(UserDetail userDetail, Map<Long, User> userMap) {
        UserDetailVO userDetailVO = UserDetailVO.fromUserDetail(userDetail);
        User user = userMap.get(userDetail.getUserId());
        if (user != null) {
            userDetailVO.setEnable(user.getEnable());
        }
        return userDetailVO;
    }

    private Specification<UserDetail> valueInSpecification(List<Long> userId, List<String> phoneNo, List<String> idNo) {
        List<Predicate> list = new ArrayList<>();
        Specification<UserDetail> specification = (root, criteriaQuery, criteriaBuilder) -> {
            if (userId != null && userId.size() > 0) {
                criteriaQuery.where(root.get("userId").in(userId));
                list.add(criteriaQuery.getRestriction());
            }
            if (phoneNo != null && phoneNo.size() > 0) {
                criteriaQuery.where(root.get(Constants.PHONE_NO).in(phoneNo));
                list.add(criteriaQuery.getRestriction());
            }
            if (idNo != null && idNo.size() > 0) {
                criteriaQuery.where(root.get("idNo").in(idNo));
                list.add(criteriaQuery.getRestriction());
            }
            Predicate[] p = new Predicate[list.size()];
            return criteriaBuilder.and(list.toArray(p));
        };
        return specification;
    }

    private PageRequest buildPageRequest(int pageNumber, int pagzSize, String sortType) {
        Sort sort = null;
        if ("auto".equals(sortType)) {
            sort = new Sort(Sort.Direction.ASC, "userId");
        }

        return new PageRequest(pageNumber - 1, pagzSize, sort);
    }

    @Override
    public Page<UserDetail> getUserDetailsPage(List<Long> userId, List<String> phoneNos, List<String> idNos, int pageNumber, int pageSize,
                                               String sortType) {
        PageRequest pageRequest = buildPageRequest(pageNumber, pageSize, sortType);
        Specification<UserDetail> spec = valueInSpecification(userId, phoneNos, idNos);
        return userDetailRepository.findAll(spec, pageRequest);
    }

    @Override
    public int updateIdCard(String name, String idNo, String phoneNo) {
        UserDetail userDetail = userDetailRepository.findByPhoneNo(phoneNo);
        if (null != userDetail) {
            if (StringUtils.isNotBlank(idNo)) {
                IdCardInfo idCardInfo = idCardService.getIdCardInfo(idNo);
                if (idCardInfo == null || !idCardInfo.isValid()) {
                    log.error("用户的身份证错误，phoneNo:{},idNo:{}", phoneNo, idNo);
                    return 0;
                }
                return userDetailRepository.updateIdNoByPhoneNo(idCardInfo.getIdNo(), Optional.ofNullable(idCardInfo.getGender()).orElse(Gender.UNKNOWN).ordinal(), phoneNo);
            }
            if (StringUtils.isNotBlank(name) && ValidationUtil.validateChinese(name)) {
                return userDetailRepository.updateNameByPhoneNo(name, phoneNo);
            }
        }
        return 0;
    }

    @Override
    public List<UserDetail> findByPhones(List<String> phoneNos) {
        List<UserDetail> userDetails = userDetailRepository.findAll((root, query, cb) -> {
            query.where(root.get("phoneNo").as(String.class).in(phoneNos));
            return query.getRestriction();
        });
        return userDetails;
    }

    @Override
    public List<UserDetail> fuzzyQueryByPhoneNoAndIdNo(String phoneNo, String idNo){
        return userDetailRepository.fuzzyQueryByPhoneNoAndIdNo(phoneNo.concat("%"), idNo.concat("%"));
    }

}
