package cn.quantgroup.xyqb.controller.external.user;

import javax.servlet.http.HttpServletRequest;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.quantgroup.xyqb.aspect.accessable.IpValidator;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.UserRet;
import cn.quantgroup.xyqb.model.session.LoginInfo;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.IPUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;

import static cn.quantgroup.xyqb.constant.UserConstant.USER_ERROR_OR_PASSWORD_ERROR;

/**
 * 第三方用户免密登录
 *
 * @author mengfan.feng
 * @time 2015-10-27 11:41
 */
@RestController
@RequestMapping("/app")
public class AppController implements IBaseController {

    private static final Logger LOGGER = LoggerFactory.getLogger(AppController.class);
    @Autowired
    private ISessionService sessionService;
    @Autowired
    private IUserService userService;
    @Autowired
    private IMerchantService merchantService;
    @Autowired
    private IUserRegisterService userRegisterService;

    /**
     * 第三方用户免密登录
     * 发现新手机号，自动执行注册
     *
     * @param phoneNo             - 手机号
     * @param idNo                - 用户身份证号
     * @param name                - 用户姓名
     * @param key                 - Merchant表name属性
     * @param btRegisterChannelId - 白条注册渠道id
     * @param registerFrom        - 注册渠道
     * @param channelId           -
     * @param appChannel          -
     * @param request
     * @return Token和phoneNo
     */
    @IpValidator
    @RequestMapping("/login")
    @ApiOperation(value = "App登陆", notes = "App登陆", httpMethod = "POST")
    public JsonResult login(
            String phoneNo,
            String idNo, String name, String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            HttpServletRequest request) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        LOGGER.info("第三方用户登录 [AppController] login --> loginFrom:{},channelId：{},btRegisterChannelId:{} requestIp:{},idNo:{},name:{}", registerFrom, channelId, btRegisterChannelId, IPUtil.getRemoteIP(request), idNo, name);
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            user = userRegisterService.register(registerFrom, phoneNo, idNo, name, channelId, btRegisterChannelId);
        }
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (!user.getEnable()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("无效的商户", null);
        }
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, merchant.getId(), merchant.getName());
        SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);
        AuthBean bean = new AuthBean();
        bean.setToken(sessionStruct.getSid());
        bean.setPhoneNo(phoneNo);
        LOGGER.info("第三方用户登录成功 [AppController] login --> loginFrom:{}, phoneNo:{},appChannel:{}", registerFrom, phoneNo, appChannel);
        return new JsonResult(bean);
    }

    /**
     * 第三方用户免密登录
     * 发现新手机号，自动执行注册
     *
     * @param phoneNo             - 手机号
     * @param idNo                - 用户身份证号
     * @param name                - 用户姓名
     * @param key                 - Merchant表name属性
     * @param btRegisterChannelId - 白条注册渠道id
     * @param registerFrom        - 注册渠道
     * @param channelId           -
     * @param appChannel          -
     * @param request
     * @return 用户信息
     */

    @IpValidator
    @RequestMapping("/login_super")
    @ApiOperation(value = "免密登陆, 新手机号还自动注册", notes = "免密登陆, 新手机号还自动注册", httpMethod = "POST")
    public JsonResult loginSuper(
            String phoneNo,
            String idNo, String name, String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            HttpServletRequest request) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        LOGGER.info("第三方用户登录 [AppController] loginSuper --> loginFrom:{},phoneNo:{},appChannel:{},channelId：{},btRegisterChannelId:{} requestIp:{},idNo:{},name:{}", registerFrom, phoneNo, appChannel, channelId, btRegisterChannelId, IPUtil.getRemoteIP(request), idNo, name);
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            user = userRegisterService.register(registerFrom, phoneNo, idNo, name, channelId, btRegisterChannelId);
        }
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (!user.getEnable()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("无效的商户", null);
        }
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, merchant.getId(), merchant.getName());
        SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);
        LoginInfo loginInfo = new LoginInfo();
        loginInfo.setUser(new UserRet(user));
        loginInfo.setToken(sessionStruct.getSid());
        LoginInfo.LoginContext context = new LoginInfo.LoginContext();
        context.setChannelId(sessionStruct.getValues().getLoginProperties().getChannelId());
        context.setCreatedFrom(sessionStruct.getValues().getLoginProperties().getCreatedFrom());
        context.setAppChannel(appChannel);
        loginInfo.setLoginContext(context);
        LOGGER.info("第三方用户获取信息登录成功 [AppController] loginSuper --> loginFrom:{}, phoneNo:{},appChannel:{},channelId:{}", registerFrom, phoneNo, appChannel, channelId);

        return JsonResult.buildSuccessResult("", loginInfo);
    }

    /**
     * 第三方用户免密登录
     * 发现新手机号不会执行注册
     *
     * @param phoneNo      - 手机号
     * @param registerFrom - 注册渠道
     * @param channelId    -
     * @param appChannel   -
     * @param request
     * @return Token和phoneNo
     */

    @IpValidator
    @RequestMapping("/login2")
    @ApiOperation(value = "免密登陆, 新手机号不自动注册", notes = "免密登陆, 新手机号不自动注册", httpMethod = "POST")
    public JsonResult login2(
            String phoneNo,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            HttpServletRequest request) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        LOGGER.info("第三方用户登录 [AppController] login2 --> loginFrom:{},channelId:{}, requestIp:{}", registerFrom, channelId, IPUtil.getRemoteIP(request));
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (!user.getEnable()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, null, "");
        SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);
        AuthBean bean = new AuthBean();
        bean.setToken(sessionStruct.getSid());
        bean.setPhoneNo(phoneNo);
        LOGGER.info("第三方用户登录成功 [AppController] login2 --> loginFrom:{}, phoneNo:{},appChannel:{}", registerFrom, phoneNo, appChannel);
        return new JsonResult(bean);
    }

}
