package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.user.enums.UserStatusEnum;
import cn.quantgroup.xyqb.aspect.accessable.IpValidator;
import cn.quantgroup.xyqb.aspect.captcha.LoginInterceptor;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.UserRet;
import cn.quantgroup.xyqb.model.session.LoginInfo;
import cn.quantgroup.xyqb.security.AuthorityManager;
import cn.quantgroup.xyqb.security.AuthorizationPoint;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.user.IOauthLoginInfoService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.IpUtil;
import cn.quantgroup.xyqb.util.TenantUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.sensorsdata.analytics.javasdk.ISensorsAnalytics;
import com.sensorsdata.analytics.javasdk.bean.EventRecord;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.persistence.PersistenceException;
import javax.servlet.http.HttpServletRequest;

import static cn.quantgroup.xyqb.constant.UserConstant.USER_ERROR_OR_PASSWORD_ERROR;

/**
 * 第三方用户免密登录
 *
 * @author mengfan.feng
 * @time 2015-10-27 11:41
 */
@Api(tags = "第三方用户免密登录 API")
@Slf4j
@RestController
@RequestMapping("/app")
public class AppController implements IBaseController {

    @Autowired
    private ISessionService sessionService;
    @Autowired
    private IUserService userService;
    @Autowired
    private IOauthLoginInfoService oauthLoginInfoService;
    @Autowired
    private IMerchantService merchantService;
    @Autowired
    private IUserRegisterService userRegisterService;

    @ApiOperation(value = "app 认证登录", httpMethod = "POST", notes="app 认证登录")
    @IpValidator
    @AuthorizationPoint
    @RequestMapping("/auth/v1/login")
    public JsonResult authLoginV1(
            String idNo
            , String name
            , @RequestParam(required = true) String key,
            @RequestParam(required = true) Long createdFrom,
            @RequestParam(required = false) Long registerFrom,
            @RequestParam(required = true) Long channelId,
            @RequestParam(required = true) String appChannel,
            @RequestParam(required = false) Integer tenantId,
            HttpServletRequest request) {

        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        JsonResult result = null;
        Authentication authentication = AuthorityManager.authentication();
        if(authentication.isAuthenticated()){
            String phoneNumber = ObjectUtils.getDisplayString(authentication.getCredentials());
            if (!ValidationUtil.validatePhoneNo(phoneNumber)) {
                result = JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
            }else{
                User user = userService.findByPhoneInDb(phoneNumber);
                // 默认为已注册
                Boolean isRegister = false;
                if (user == null) {
                    // 未注册过
                    isRegister = true;
                    user = userRegisterService.register(registerFrom, phoneNumber, idNo, name, channelId, createdFrom, tenantId);
                }
                // 无论是否注册过都添加映射
                oauthLoginInfoService.addLoginInfo(user, tenantId);

                if (user == null) {
                    result = JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
                } else if (UserStatusEnum.isDisable(user).getKey()) {
                    result = JsonResult.buildErrorStateResult("登录失败", null);
                } else {
                    Merchant merchant = merchantService.findMerchantByName(key);
                    if (merchant == null) {
                        result = JsonResult.buildErrorStateResult("无效的商户", null);
                    }else{
                        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, merchant.getId(), merchant.getName(), null);
                        AuthBean bean = sessionService.createSession(user, loginProperties);
                        LoginInfo.LoginContext context = new LoginInfo.LoginContext();
                        context.setChannelId(channelId);
                        context.setCreatedFrom(registerFrom);
                        context.setAppChannel(appChannel);
                        LoginInfo loginInfo = new LoginInfo();
                        loginInfo.setUser(new UserRet(user));
                        loginInfo.setHasPassword(user.getHasPassword());
                        loginInfo.setToken(bean.getToken());
                        loginInfo.setLoginContext(context);
                        loginInfo.setRegister(isRegister);
                        result = JsonResult.buildSuccessResult("登陆成功", loginInfo);
                    }
                }
            }
        }else{
            result = JsonResult.buildErrorStateResult("登录失败", null);
        }
        return result;
    }

    /**
     * 第三方用户免密登录
     * 发现新手机号，自动执行注册
     *
     * @param phoneNo             - 手机号
     * @param idNo                - 用户身份证号
     * @param name                - 用户姓名
     * @param key                 - Merchant表name属性
     * @param btRegisterChannelId - 白条注册渠道ID
     * @param registerFrom        - 注册渠道（第一次用户来源channelId）
     * @param channelId           - 渠道ID
     * @param appChannel          - 应用平台（应用商店、AppStore...）
     * @param request
     * @return Token和phoneNo
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/169
     * @Deprecated 20210318
     */
    @Deprecated
    @IpValidator
    @RequestMapping("/login")
    @ApiOperation(value = "App登陆", notes = "App登陆", httpMethod = "POST")
    public JsonResult login(
            String phoneNo,
            String idNo, String name, String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            HttpServletRequest request) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        log.info("第三方用户登录 [AppController] login --> loginFrom:{},channelId：{},btRegisterChannelId:{} requestIp:{},idNo:{},name:{}", registerFrom, channelId, btRegisterChannelId, IpUtil.getRemoteIP(request), idNo, name);
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            user = userRegisterService.register(registerFrom, phoneNo, idNo, name, channelId, btRegisterChannelId);
        }
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (UserStatusEnum.isDisable(user).getKey()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("无效的商户", null);
        }

        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, merchant.getId(), merchant.getName(), null);
        AuthBean bean = sessionService.createSession(user, loginProperties);
        log.info("第三方用户登录成功 [AppController] login --> loginFrom:{}, phoneNo:{},appChannel:{}", registerFrom, phoneNo, appChannel);
        return JsonResult.buildSuccessResult("登录成功", bean);
    }

    @Autowired
    private ISensorsAnalytics iSensorsAnalytics;

    /**
     * 第三方用户免密登录
     * 发现新手机号，自动执行注册
     *
     * @param phoneNo             - 手机号
     * @param idNo                - 用户身份证号
     * @param name                - 用户姓名
     * @param key                 - merchant表的name属性（跳转相关）
     * @param btRegisterChannelId - 白条注册渠道id
     * @param registerFrom        - 注册渠道（第一次用户来源channelId）
     * @param channelId           - 渠道
     * @param appChannel          - 应用平台（应用商店、AppStore...）
     * @param request
     * @return 用户信息
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/173
     */
    @IpValidator
    @RequestMapping("/login_super")
    @ApiOperation(value = "免密登陆, 新手机号还自动注册", notes = "免密登陆, 新手机号还自动注册", httpMethod = "POST")
    public JsonResult loginSuper(
            String phoneNo,
            String idNo, String name, String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            @RequestParam(required = false) Integer tenantId,
            HttpServletRequest request) {
        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        log.info("第三方用户登录 [AppController] loginSuper --> loginFrom:{},phoneNo:{},appChannel:{},channelId：{},btRegisterChannelId:{} requestIp:{},idNo:{},name:{}", registerFrom, phoneNo, appChannel, channelId, btRegisterChannelId, IpUtil.getRemoteIP(request), idNo, name);
        User user = userService.findByPhoneInDb(phoneNo);
        boolean isRegister=false;
        if (user == null) {
            try {
                user = userRegisterService.register(registerFrom, phoneNo, idNo, name, channelId, btRegisterChannelId);
                isRegister=true;
            } catch (PersistenceException e) {
                user = userService.findByPhoneInDb(phoneNo);
            }
        }
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (UserStatusEnum.isDisable(user).getKey()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("无效的商户", null);
        }

        //租户下新增用户
        oauthLoginInfoService.addLoginInfo(user, tenantId);

        log.info("=addLoginInfo end");
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, merchant.getId(), merchant.getName(), tenantId);
        AuthBean bean = sessionService.createSession(user, loginProperties);
        log.info("=createSession end");
        LoginInfo loginInfo = new LoginInfo();
        loginInfo.setUser(new UserRet(user));
        loginInfo.setHasPassword(user.getHasPassword());
        loginInfo.setToken(bean.getToken());
        if(isRegister){
            loginInfo.setRegister(true);
        }else {
            loginInfo.setRegister(false);
        }
        LoginInfo.LoginContext context = new LoginInfo.LoginContext();
        context.setChannelId(channelId);
        context.setCreatedFrom(registerFrom);
        context.setAppChannel(appChannel);
        loginInfo.setLoginContext(context);
        log.info("=setLoginInfo end");
        log.info("第三方用户获取信息登录成功 [AppController] loginSuper --> loginFrom:{}, phoneNo:{},appChannel:{},channelId:{}", registerFrom, phoneNo, appChannel, channelId);

        try {
            String scDeviceId=request.getHeader("scDeviceId");
            if(!StringUtils.isEmpty(scDeviceId)){
                iSensorsAnalytics.trackSignUp( user.getUuid(),scDeviceId);
            }
            String terminal=request.getHeader("terminal");
            String channel=request.getHeader("channel");
            log.info("--------手机号{}-------scDeviceId{},terminal{},channel{},是否注册{}",phoneNo,scDeviceId,terminal,channel,isRegister);
            if(!StringUtils.isEmpty(terminal)&&"APP".equals(terminal)||!StringUtils.isEmpty(channel)&&("214".equals(channel)||"217".equals(channel))){
                if(isRegister){
                    EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
                            .setEventName("App_RegisterEvent")
                            .build();
                    iSensorsAnalytics.track(userRecord);
                }else {
                    EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
                            .setEventName("App_LoginEvent")
                            .build();
                    iSensorsAnalytics.track(userRecord);
                }
                iSensorsAnalytics.flush();
            }else {
                EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
                        .setEventName("PD_WUXIEC_UserLoginVccorCash")
                        .addProperty("son_channel_id", user.getRegisteredFrom())
                        .addProperty("parent_channel_id", -1L)
                        .addProperty("vcccash_uuid", user.getUuid())
                        .build();
                iSensorsAnalytics.track(userRecord);
                iSensorsAnalytics.flush();
            }
        } catch (Exception e) {
            log.info("神策埋点出现问题", e);
        }
        return JsonResult.buildSuccessResult("登陆成功", loginInfo);
    }

    /**
     * 第三方用户免密登录
     * 发现新手机号不会执行注册
     *
     * @param phoneNo      - 手机号
     * @param registerFrom - 注册渠道（第一次用户来源channelId）
     * @param channelId    - 渠道
     * @param appChannel   - 应用平台（应用商店、AppStore...）
     * @param request
     * @return Token和phoneNo
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/171
     */
    @LoginInterceptor
    @IpValidator
    @RequestMapping("/login2")
    @ApiOperation(value = "免密登陆, 新手机号不自动注册", notes = "免密登陆, 新手机号不自动注册", httpMethod = "POST")
    public JsonResult login2(
            String phoneNo,
            @RequestParam(required = false, defaultValue = "1") Long registerFrom,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "") String appChannel,
            @RequestParam(required = false) Integer tenantId,
            HttpServletRequest request) {
        //默认羊小咩租户
        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        log.info("第三方用户登录 [AppController] login2 --> loginFrom:{},channelId:{}, requestIp:{}", registerFrom, channelId, IpUtil.getRemoteIP(request));
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_PASSWORD_ERROR, null);
        }
        if (UserStatusEnum.isDisable(user).getKey()) {
            return JsonResult.buildErrorStateResult("登录失败", null);
        }
        //校验租户ID tenantId
        oauthLoginInfoService.addLoginInfo(user, tenantId);
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, registerFrom, appChannel, null, "", tenantId);
        AuthBean bean = sessionService.createSession(user, loginProperties);
        log.info("第三方用户登录成功 [AppController] login2 --> loginFrom:{}, phoneNo:{},appChannel:{}", registerFrom, phoneNo, appChannel);
        return JsonResult.buildSuccessResult("登录成功", bean);
    }

//    @RequestMapping("/login33")
//    public JsonResult login233() {
//        User user = new User();
//        user.setUuid("3213213321");
//        user.setRegisteredFrom(221L);
//        try {
//            EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
//                    .setEventName("PD_WUXIEC_UserLoginVccCash")
//                    .addProperty("son_channel_id", user.getRegisteredFrom())
//                    .addProperty("parent_channel_id",-1L)
//                    .addProperty("vcc_uuid", user.getUuid())
//                    .build();
//            iSensorsAnalytics.track(userRecord);
//            iSensorsAnalytics.flush();
//            log.info("神策上报成功");
//        } catch (Exception e) {
//            log.info("神策埋点出现问题", e);
//        }
//        return JsonResult.buildSuccessResult("登录成功", null);
//
//    }

}
