package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserAttached;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.repository.IUserAttachedRepository;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.user.UserCenterService;
import cn.quantgroup.xyqb.util.EmojiUtil;
import cn.quantgroup.xyqb.util.TenantUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.exception.ConstraintViolationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Created by 11 on 2017/3/22.
 */
@Slf4j
@Service
public class UserCenterServiceImpl implements UserCenterService {

    @Autowired
    private IUserAttachedRepository userAttachedRepository;
    @Autowired
    private IUserInfoRepository userInfoRepository;

    @Autowired
    private IUserRepository userRepository;

    @Override
//    @Cacheable(value = "userAttachedCache", key = "'xyqbUserAttached' + #userId", unless = "#result == null", cacheManager = "cacheManager")
    public UserAttached searchUserAttachedByUserId(long userId) {
        return userAttachedRepository.findByUserId(userId);
    }

    @Override
    public List<UserAttached> searchUserAttachedListByUserId(List<Long> userIds, Integer tenantId) {
        List<UserInfoEntity> userInfoEntityList = userInfoRepository.findByTenantIdAndUserIdIn(tenantId, userIds);
        if (CollectionUtils.isEmpty(userInfoEntityList)) {
            return Lists.newArrayList();
        }
        List<UserAttached> list = Lists.newArrayList();
        userInfoEntityList.stream().forEach(e -> {
            UserAttached userAttached = new UserAttached();
            userAttached.setUserId(e.getUserId());
            userAttached.setNick(e.getName());
            userAttached.setAvatar(e.getPhoto());

            list.add(userAttached);
        });
        return list;
    }

    @Override
    @CacheEvict(value = "userAttachedCache", key = "'xyqbUserAttached' + #userId +'-'+#tenantId+'-'+#tenantId", cacheManager = "cacheManager")
    public UserAttached saveUserAvatar(long userId, String avatar, Integer tenantId) {
        UserInfoEntity userAttached = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        if (Objects.isNull(userAttached)) {
            userAttached = createUserAttached(tenantId, userId, avatar, null);
        }
        if (!Objects.equals(avatar, userAttached.getPhoto())) {
            userAttached.setPhoto(avatar);
            userAttached = userInfoRepository.save(userAttached);
        }
        UserAttached attached = new UserAttached();
        attached.setUserId(userAttached.getUserId());
        attached.setAvatar(userAttached.getPhoto());
        attached.setNick(userAttached.getName());
        return attached;
    }

    @Override
    @CacheEvict(value = "userAttachedCache", key = "'xyqbUserAttached' + #userId +'-'+#tenantId", cacheManager = "cacheManager")
    public UserAttached saveUserNick(long userId, String nick,Integer tenantId) {
        UserInfoEntity userAttached = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        // 替换所有,UTF-8编码时4字节的Emoji表情字符
        nick = EmojiUtil.filterUnicode4(nick);
        if (Objects.isNull(userAttached)) {
            userAttached = createUserAttached(tenantId,userId, null, nick);
        }

        UserAttached attached = null;
        if (!Objects.equals(nick, userAttached.getName())) {
            userAttached.setName(nick);
            try {
                userAttached = userInfoRepository.save(userAttached);
                attached = new UserAttached();
                attached.setUserId(userAttached.getUserId());
                attached.setAvatar(userAttached.getPhoto());
                attached.setNick(userAttached.getName());
            } catch (ConstraintViolationException e) {
                // Sql唯一约束异常（诱因是Form重复提交，因为该操作是幂等的，故此不需额外处理，可返回成功）
                log.error("Sql约束异常[uni_idx_user_id]重复提交Form是幂等操作，不影响处理结果", e);
            }
        }

        return attached;
    }

    @Override
    public List<UserAttached> queryUserAttachListLimit(Long id) {
        return userAttachedRepository.findALlAttached(id);
    }

    /**
     * 创建用户附加信息实体
     *
     * @param userId - 用户主键
     * @param avatar - 头像url地址
     * @param nick   - 昵称
     * @return
     */
    private synchronized UserInfoEntity createUserAttached(Integer tenantId, Long userId, String avatar, String nick) {
        //UserAttached userAttached = userAttachedRepository.findByUserId(userId);
        UserInfoEntity userAttached = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        // 更新实例
        userAttached = Optional.ofNullable(userAttached).orElse(new UserInfoEntity());
        userAttached.setUserId(userId);
        if (StringUtils.isBlank(userAttached.getPhoto())) {
            userAttached.setPhoto(avatar);
        }
        if (StringUtils.isBlank(userAttached.getName())) {
            userAttached.setName(nick);
        }
        return userInfoRepository.save(userAttached);
    }
}
