package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaFineteValidator;
import cn.quantgroup.xyqb.aspect.lock.PasswordFineteValidator;
import cn.quantgroup.xyqb.aspect.logcaller.LogHttpCaller;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.PasswordErrorLimitException;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.exception.VerificationCodeErrorException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserModel;
import cn.quantgroup.xyqb.model.UserStatistics;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.*;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Http服务接口：用户注册、登录、重置密码
 * Created by FrankChow on 15/7/5.
 */
@RestController
@RequestMapping("/user")
public class UserController implements IBaseController {


  private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);
  private final String pwdSalt = "_lkb";

  @Autowired
  private IUserService userService;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private ISmsService smsService;

  @Autowired
  private ISessionService sessionService;

  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IMerchantService merchantService;
  @Autowired
  private IWechatService wechatService;

  @RequestMapping("/test")
  public JsonResult test() {
    HttpServletRequest request = getRequest();
    String remoteAddr = request.getRemoteAddr();
    String xRealIp = request.getHeader("x-real-ip");
    String xOriginalClientIp = request.getHeader("x-original-client-ip");
    LOGGER.info("Test ips:[client={}, old={}, new={}]", remoteAddr, xRealIp, xOriginalClientIp);
    return JsonResult.buildSuccessResult("", getCurrentUserFromRedis());
  }

  /**
   * 解锁特定IP
   * @param ip - 目标IP
   * @param act - 操作标记，true-lock，false-unlock
   * @param key - 密令
   * @param request
   * @return
   */
  @RequestMapping("/lock_ipv4")
  public JsonResult clearOrLockIpv4(@RequestParam(required = true)String ip,
                                     @RequestParam(required = false)String act,
                                     @RequestParam(required = true)String key,
                                     HttpServletRequest request) {
    if(!ValidationUtil.validateIpv4(ip) || StringUtils.isBlank(act) || StringUtils.isBlank(key)){
      LOGGER.info("Fail to clear_or_lock ip:{}", ip);
      return JsonResult.buildErrorStateResult("Are you a robot monkey？（^_^）",null);
    }
    // 操作标记
    boolean lock = Objects.equals(Boolean.TRUE.toString(), act);
    String header_key = request.getHeader(Constants.IPV4_LOCK.replace(":", ""));
    if(Objects.equals(Constants.CLEAR_LOCK_FOR_IPV4, header_key)){
      return JsonResult.buildErrorStateResult("Are you a robot monkey？（^_^）", ValidationUtil.getMd5Key(lock));
    }
    boolean valid = ValidationUtil.isValid(key, lock);
    if(!valid){
      Calendar now = Calendar.getInstance();
      int hour = now.get(Calendar.HOUR_OF_DAY);
      valid = Objects.equals(Constants.CLEAR_LOCK_FOR_IPV4_KEY+hour, key);
    }
    if(valid){
      String lockIpv4Key = getLockIpv4Key(ip);
      if(lock){
        stringRedisTemplate.opsForValue().set(lockIpv4Key, Boolean.TRUE.toString(), Constants.IPV4_LOCK_MINUTES, TimeUnit.MINUTES);
        LOGGER.info("Locked ip access:{}, error overstep {} times in {} minutes, do lock {} minutes", ip, Constants.IPV4_LOCK_ON_COUNTS, Constants.IPV4_COUNT_MINUTES, Constants.IPV4_LOCK_MINUTES);
      }else{
        stringRedisTemplate.delete(lockIpv4Key);
        LOGGER.info("Clear_or_lock ip Success:{}", ip);
      }
      return JsonResult.buildSuccessResult("Success",null);
    }
    LOGGER.info("Fail to clear_or_lock ip:{}", ip);
    return JsonResult.buildErrorStateResult("Are you a robot monkey？（^_^）",null);
  }

  /**
   * 登录（账号 + 密码）
   * 密码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param userId
   * @param key
   * @param request
   * @param openId
   * @param dimension
   * @return
   */
  @CaptchaFineteValidator
  @RequestMapping("/loginV1")
  public JsonResult loginV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          HttpServletRequest request, String openId,
          @RequestParam(required = false) String dimension) {
    LOGGER.info("loginV1 -> channelId:{},appChennel:{},createdFrom:{},userId:{},key:{},openId:{},dimension:{}",channelId, appChannel, createdFrom, userId, key, openId, dimension);
    return login(channelId, appChannel, createdFrom, userId, key, request, openId, dimension);
  }

  /**
   * 快速登录（手机号 + 短信验证码），H5专用入口
   * 短信验证码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param key
   * @param btRegisterChannelId
   * @param dimension
   * @param request
   * @return
   */
  // Todo @OneTimeTokenValidator
  @RequestMapping("/login/fastV1")
  public JsonResult loginFastV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false)Long btRegisterChannelId,
          @RequestParam(required = false) String dimension ,HttpServletRequest request) {
    LOGGER.info("login/fastV1 -> channelId:{},appChennel:{},createdFrom:{},key:{},btRegisterChannelId:{},dimension:{}",channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension);
    return loginFast(channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, request);
  }

  @LogHttpCaller
  @PasswordFineteValidator
  @RequestMapping("/login")
  public JsonResult login(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          HttpServletRequest request, String openId,
          @RequestParam(required = false) String dimension) {
    LOGGER.info("user/login,请求参数channelId:{},appChannel:{},createdFrom:{},userId:{},key:{},openId:{},dimension:{},",channelId,appChannel,createdFrom,userId,key,openId,dimension);
    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    if (!StringUtils.isEmpty(userId) && userId.length() > 10) {

      return loginWithUserId(channelId, appChannel, createdFrom, userId, merchant,dimension);
    } else {
      return loginWithHttpBasic(channelId, appChannel, createdFrom, merchant, request, openId,dimension);
    }
  }

  @RequestMapping("/login/fast")
  public JsonResult loginFast(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false)Long btRegisterChannelId,
          @RequestParam(required = false) String dimension ,HttpServletRequest request) {
    Map<String, JsonResult> validMap = getHeaderParam(request);
    LOGGER.info("user/login/fast,请求参数channelId:{},appChannel:{},createdFrom:{},btRegisterChannelId:{},key:{},dimension:{},",channelId,appChannel,createdFrom,btRegisterChannelId,key,dimension);
    if (null != validMap.get("fail")) {
      return validMap.get("fail");
    }
    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    JsonResult successResult = validMap.get("success");
    String phoneNo = successResult.getData().toString();
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户快速注册失败，手机号错误, createdFrom:{},phoneNo:{}", createdFrom, phoneNo);
      throw new UserNotExistException("手机号错误");
    }
    String verificationCode =  successResult.getMsg();
    // 执行短信验证码检查
    smsValidForFastLogin(phoneNo, verificationCode);
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user != null && !user.getEnable()) {
      LOGGER.error("用户不存在，或者已经注销，phoneNo:{}",phoneNo);
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
    if (user == null) {
      user = registerFastWhenLogin(phoneNo, channelId, createdFrom, appChannel,btRegisterChannelId,dimension);
      if (user == null) {
        throw new UserNotExistException("用户未找到");
      }
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,3,channelId);
    MqUtils.sendLoanVest(statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
    // return createSession(channelId, createdFrom, appChannel, user);
  }

  private User registerFastWhenLogin(String phoneNo, Long channelId, Long registerFrom, String appChannel, Long btRegisterChannelId,String dimension) {
    String password = PasswordUtil.generateRandomPwd(15);
    LOGGER.info("用户快速注册, phoneNo:{}, channelId:{}, registerFrom:{},appChannel:{},btRegisterChannelId", phoneNo, channelId, registerFrom, appChannel,btRegisterChannelId);
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    if (channelId == 222L) {
      registerFrom=222L;
    }
    User user=userService.registerAndReturn(phoneNo, password, registerFrom,btRegisterChannelId);
    LOGGER.info("用户快速注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
     MqUtils.sendRegisterMessage(channelId,dimension, user);
    return user;
  }

  /**
   * 快速登录验证
   *
   * @param request
   * @return
   */
  private Map<String, JsonResult> getHeaderParam(HttpServletRequest request) {
    Map<String, JsonResult> result = new HashMap<String, JsonResult>();
    String verificationHeader = "Verification ";
    String credential = request.getHeader("authorization");
    if (StringUtils.isBlank(credential)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    if (!credential.startsWith(verificationHeader)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    credential = credential.substring(verificationHeader.length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    credential = new String(buf, Charset.forName("UTF-8"));
    String[] credentialArr = credential.split(":");
    if (credentialArr.length != 2) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    String phoneNo = credentialArr[0];
    String verificationCode = credentialArr[1];
    LOGGER.info("用户快速登录,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
    if (!ValidationUtil.validatePhoneNo(phoneNo) || StringUtils.isBlank(verificationCode)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    result.put("success", JsonResult.buildSuccessResult(verificationCode, phoneNo));
    return result;
  }

  /**
   * 用户注册
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @param channelId
   * @return
   */
  @RequestMapping("/register")
  public JsonResult register(@RequestParam String phoneNo, @RequestParam String password,
                             @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                             @RequestParam(required = false) Long registerFrom,
                             @RequestParam(required = false)Long btRegisterChannelId,@RequestParam(required = false)String dimension) {
    LOGGER.info("[/user/register]用户注册, phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},btRegisterChannelId:{},dimension:{}", phoneNo, verificationCode, channelId, registerFrom,btRegisterChannelId,dimension);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      LOGGER.info("用户注册失败，密码不能为空, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      LOGGER.info("用户注册失败，密码长度须在6位至12位之间, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    smsValidForRegister(phoneNo, verificationCode);
    if (userService.exist(phoneNo)) {
      LOGGER.info("用户注册失败，该手机号已经被注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
    }
    if (!userService.register(phoneNo, password, registerFrom, getIp(), channelId,btRegisterChannelId,dimension)) {
      LOGGER.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
    }
    LOGGER.info("用户注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist")
  public JsonResult exist(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    if (userService.exist(phoneNo)) {
      LOGGER.info("该手机号已经注册, phoneNo:{}", phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经注册", null);
    }
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist_check")
  public JsonResult existForResetPwd(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    return JsonResult.buildSuccessResult(null, userService.exist(phoneNo));
  }

  /**
   * 重置密码
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/reset_password")
  public JsonResult resetPassword(@RequestParam String phoneNo,
                                  @RequestParam String password,
                                  @RequestParam(required = false) String registerFrom,
                                  @RequestParam String verificationCode) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    smsValidForRegister(phoneNo, verificationCode);
    if (!userService.exist(phoneNo)) {
      LOGGER.info("修改密码失败，该手机号尚未注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号尚未注册", null);
    }
    if (!userService.resetPassword(phoneNo, password)) {
      return JsonResult.buildErrorStateResult("修改密码失败", null);
    }

    // TODO  加渠道号
    LOGGER.info("修改密码成功, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查token是否已经过期不存在了
   *
   * @param token
   * @return
   */
  @RequestMapping("/exists_token")
  public JsonResult checkToken(@RequestParam String token) {
    if(StringUtils.isEmpty(token)){
      return JsonResult.buildSuccessResult(null,false);
    }
    if(token.contains("*")){
      return JsonResult.buildSuccessResult(null,false);
    }

    String tokenKey = Constants.SESSION_PREFIX + token;
    String tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
    return JsonResult.buildSuccessResult(null, stringRedisTemplate.hasKey(tokenKey)||stringRedisTemplate.hasKey(tokenKey2));
  }

  @RequestMapping("/syncUserInfo")
  public JsonResult syncUserInfo() {
    User user = getCurrentUserFromRedis();
    if (null == user) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    UserDetail detail = userDetailService.findByUserId(user.getId());
    //UserDetail detail = userDetailRepository.findByUserId(user.getId());
    UserModel userModel = new UserModel(user, detail);
    return JsonResult.buildSuccessResult("token校验成功", userModel);
  }

  private JsonResult loginWithHttpBasic(Long channelId, String appChannel, Long createdFrom, Merchant merchant, HttpServletRequest request, String openId,String dimension) {
    User user = verificateUserNameAndPassword(request, openId);
    if (user == null) {
      return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,1,channelId);
    MqUtils.sendLoanVest(statistics);
    LOGGER.info("用户登陆成功phonNo:{},当前ip:{}，用户登陆信息：{}",user.getPhoneNo(), IPUtil.getRemoteIP(request),statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
  }

  private User verificateUserNameAndPassword(HttpServletRequest request, String openId) {
    String credential = request.getHeader("authorization");
    if (StringUtils.isBlank(credential) || !credential.startsWith("Basic ")) {
      return null;
    }
    credential = credential.substring("Basic ".length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    String bufStr = "";
    try {
      bufStr = new String(buf, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("不支持的编码: ", e);
    }
    String[] credentialArr = bufStr.split(":");
    if (credentialArr.length != 2) {
      LOGGER.info("用户登录失败：{}", bufStr);
      // 向该ipv4添加错误计数器
      countErrorByIpv4();
      return null;
    }
    LOGGER.info("用户正在登录... [{}]", credentialArr);
    String phoneNo = credentialArr[0];
    String pass = credentialArr[1];
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user == null || !user.getEnable()) {
      // 向该phoneNo添加错误计数器
      countErrorByPhoneNo(phoneNo);
      // 向该ipv4添加错误计数器
      countErrorByIpv4();
      return null;
    }
    //验证密码
    if (!validatePassword(pass, user.getPassword())) {
      // 向该phoneNo添加错误计数器
      countErrorByPhoneNo(phoneNo);
      // 向该ipv4添加错误计数器
      countErrorByIpv4();
      return null;
    }
    return user;
  }

  /**
   * 向该phoneNo添加错误计数器
   * @param phoneNo
   */
  private void countErrorByPhoneNo(String phoneNo) {
    // 密码错误时，给该账号添加计数器
    String key = Constants.REDIS_PASSWORD_ERROR_COUNT + phoneNo;
    if (!stringRedisTemplate.hasKey(key)) {
      LOGGER.info("添加错误计数器，key={}", key);
      stringRedisTemplate.opsForValue().set(key, String.valueOf(0), DateUtils.getSeconds(), TimeUnit.SECONDS);
    }
    // 密码错误计数
    Long errorCount = stringRedisTemplate.opsForValue().increment(key, 1L);
    if (errorCount > Constants.Image_Need_Count) {
      throw new PasswordErrorLimitException("用户名或密码不正确");
    } else if (Objects.equals(errorCount, Constants.Image_Need_Count)) {
      throw new PasswordErrorLimitException("请输入图形验证码");
    }
  }

  /**
   * 向该ipv4添加错误计数器
   */
  private void countErrorByIpv4() {
    // Todo -- 全天候开放监控
    /*if(!ValidationUtil.isAtDangerousTime()){
      return;
    }*/
    String ipv4 = getIp();
    if (StringUtils.isNotBlank(ipv4) && !ValidationUtil.validateLocalIpv4(ipv4)) {
      String ipv4Key = getIpKey(ipv4);
      if(!stringRedisTemplate.hasKey(ipv4Key)){
        // 计数周期1分钟
        stringRedisTemplate.opsForValue().set(ipv4Key, String.valueOf(0), Constants.IPV4_COUNT_MINUTES, TimeUnit.MINUTES);
      }
      Long count = stringRedisTemplate.opsForValue().increment(ipv4Key, 1L);
      LOGGER.info("Lock_ipv4: count ip access: ip={}, count={}", ipv4, count);
      lockIpv4(ipv4, count);
    }
  }

  /**
   * 锁定IPV4
   * @param ip - 目标ip
   * @param count - 错误计数
   */
  private void lockIpv4(String ip, long count){
    // 每分钟计数阈值
    long counts = Constants.IPV4_LOCK_ON_COUNTS;
    String redisCounts = stringRedisTemplate.opsForValue().get(Constants.IPV4_LOCK_ON_COUNTS_REDIS);
    if(StringUtils.isNumeric(redisCounts) && Integer.valueOf(redisCounts) > 0){
      counts = Integer.valueOf(redisCounts);
    }
    if(count < counts){
      return;
    }
    // 锁定时长
    long minutes = Constants.IPV4_LOCK_MINUTES;
    String redisMinutes = stringRedisTemplate.opsForValue().get(Constants.IPV4_LOCK_ON_COUNTS_REDIS);
    if(StringUtils.isNumeric(redisMinutes) && Integer.valueOf(redisMinutes) > 0){
      minutes = Integer.valueOf(redisMinutes);
    }
    String lockIpv4Key = getLockIpv4Key(ip);
    stringRedisTemplate.opsForValue().set(lockIpv4Key, Boolean.TRUE.toString(), minutes, TimeUnit.MINUTES);
    LOGGER.info("Lock_ipv4: locked ip access:{}, error overstep {} times in {} minutes, do lock {} minutes", ip, counts, Constants.IPV4_COUNT_MINUTES, minutes);
  }

  private final static String getIpKey(String ipv4){
    return Constants.REDIS_PASSWORD_ERROR_COUNT_FOR_IPV4 + ipv4;
  }

  private final static String getLockIpv4Key(String ipv4){
    return Constants.IPV4_LOCK + ipv4;
  }

  private boolean validatePassword(String paramPass, String targetPassword) {
    return StringUtils.defaultString(targetPassword, "").equals(PasswordUtil.MD5(paramPass.toLowerCase() + pwdSalt));
  }

  private JsonResult loginWithUserId(Long channelId, String appChannel, Long createdFrom, String userId, Merchant merchant,String dimension) {
    //查询用户,存在则保存用户session信息,userId为uuid
    User user = userService.findByUuidInDb(userId);
    //用户信息存在,更新session中的最后访问时间,重新写入缓存.
    if (null != user || !user.getEnable()) {
      //增加登陆统计发送
      UserStatistics statistics=new UserStatistics(user,dimension,1,channelId);
      MqUtils.sendLoanVest(statistics);
      return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
    } else {
      LOGGER.error("用户不存在，或者已经注销，userId:{}",userId);
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
  }

  @RequestMapping("/associate_wechat")
  public JsonResult associateWithWechat(String openId) {
    User user = getCurrentUserFromRedis();
    Long userId = user.getId();
    WechatUserInfo userInfo = wechatService.findWechatUserInfoFromDb(openId);
    LOGGER.info("微信关联openId,user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo);

    // 已经绑定过了
    if (userInfo != null && StringUtils.isNotEmpty(userInfo.getPhoneNo())) {
      return JsonResult.buildSuccessResult(null, null);
    }
    // 前置绑定微信出错
    if (userInfo == null) {
      return JsonResult.buildSuccessResult(null, null);
    }
    // 未绑定信用钱包用户
    if (userInfo.getUserId() == null) {
      userInfo.setUserId(userId);
      userInfo.setPhoneNo(user.getPhoneNo());
      try {
        wechatService.saveWechatUserInfo(userInfo);
      } catch (Exception ex) {
        // 不做绑定
        return JsonResult.buildErrorStateResult("该手机号已绑定其他微信号码", null);
      }
      return JsonResult.buildSuccessResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 注册时校验短信验证码
   * @param phoneNo
   * @param verificationCode
   */
  private void smsValidForRegister(String phoneNo, String verificationCode) {
    if (!smsService.validRegisterOrResetPasswdVerificationCode(phoneNo, verificationCode)) {
      smsReSendOrNot(phoneNo);
      LOGGER.info("用户快速注册，验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
      throw new VerificationCodeErrorException("短信验证码错误");
    }
  }

  /**
   * 登录时校验短信验证码
   * @param phoneNo
   * @param verificationCode
   */
  private void smsValidForFastLogin(String phoneNo, String verificationCode) {
    if (!smsService.validateFastLoginVerificationCode(phoneNo, verificationCode)) {
      smsReSendOrNot(phoneNo);
      LOGGER.info("用户快速登录，验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
      throw new VerificationCodeErrorException("短信验证码错误");
    }
  }

  /**
   * 是否需要重新获取短信验证码
   * @param phoneNo
   */
  private void smsReSendOrNot(String phoneNo) {
    if(needRetSendCode(phoneNo)){
      String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
      stringRedisTemplate.delete(key);
      String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
      stringRedisTemplate.opsForHash().delete(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT);
      throw new VerificationCodeErrorException("验证码失效，请重新获取");
    }
  }

  /**
   * 是否需要重新发送短信验证码
   * @param phoneNo
   * @return
   */
  private boolean needRetSendCode(String phoneNo) {
    String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
    Long getVerificationCount = stringRedisTemplate.opsForHash().increment(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT, 1);
    return (getVerificationCount >= Constants.VERIFICATION_CODE_FINITE_COUNT);
  }

}
