package cn.quantgroup.xyqb.aspect.token;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.model.JsonResult;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * 一次性令牌校验切面
 *
 * @author 任文超
 * @version 1.0.0
 * @since 2017-10-31
 */
@Aspect
@Component
public class OneTimeTokenValidateAdvisor {

    private static final Logger LOGGER = LoggerFactory.getLogger(OneTimeTokenValidateAdvisor.class);

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    /**
     * 一次性令牌校验切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.token.OneTimeTokenValidator)")
    private void needOneTimeToken() {}

    /**
     * 在受一次性令牌保护的接口方法执行前, 执行一次性令牌校验
     *
     * @throws Throwable
     */
    @Around("needOneTimeToken()")
    private Object checkOneTimeToken(ProceedingJoinPoint pjp) throws Throwable {
        boolean valid = oneTimeTokenValid();
        if (valid) {
            return pjp.proceed();
        }
        return JsonResult.buildSuccessResult("令牌已失效", "", 2L);
    }

    /**
     * 校验一次性令牌
     * @return True or False
     */
    private boolean oneTimeTokenValid() {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        // 当前请求的OneTimeToken
        String oneTimeToken = request.getParameter(Constants.ONE_TIME_TOKEN);
        if (StringUtils.isBlank(oneTimeToken)){
            return false;
        }
        String oneTimeToken_value = redisTemplate.opsForValue().get(oneTimeToken);
        // OneTimeToken再Redis中值不应为空值（空白、空格、null）
        if (StringUtils.isBlank(oneTimeToken_value)) {
            return false;
        }
        boolean valid = Objects.equals(Boolean.TRUE.toString(), oneTimeToken_value);
        // OneTimeToken校验正确时删除key
        if(valid) {
            redisTemplate.delete(oneTimeToken);
        }else {
            LOGGER.info("令牌已失效，请重新请求, oneTimeToken={}, clientIp={}", oneTimeToken, request.getRemoteAddr());
        }
        return valid;
    }

}
