package cn.quantgroup.xyqb.service.session.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.constant.enums.RecordType;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserTag;
import cn.quantgroup.xyqb.event.UserLoginEvent;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.UserStatistics;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.session.SessionValue;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.session.aspect.UserBtRegisterFill;
import cn.quantgroup.xyqb.service.user.ILoginRecordService;
import cn.quantgroup.xyqb.util.MqUtils;
import cn.quantgroup.xyqb.util.TenantUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Caching;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.sql.Timestamp;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * session创建和续期
 */
@Slf4j
@Service
public class SessionServiceImpl implements ISessionService {

    @Resource
    private RedisTemplate<String, String> stringRedisTemplate;

    @Resource
    private ILoginRecordService loginRecordService;

    @Value("${token.prefix}")
    private String prefix;

    @Autowired
    private ApplicationEventPublisher applicationEventPublisher;


    /**
     * 更新session
     * 用户信息存在,更新session中的最后访问时间,重新写入缓存.
     * 存在则保存用户session信息,userId为uuid
     *
     * @param user
     * @return
     */
    @Override
    public AuthBean createSession(User user, LoginProperties properties, int loginType, Integer tenantId) {
        return this.createSession(user, properties, loginType, tenantId, true);
    }

    public AuthBean createSession(User user, LoginProperties properties, int loginType, Integer tenantId, boolean send) {
        //找到用户
        //TODO: 使用userId
        String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties, tenantId);
        SessionStruct sessionStruct = null;
        if (StringUtils.isNotEmpty(sessionId)) {
            sessionStruct = findSessionBySessionId(sessionId, tenantId);
        }
        if (Objects.nonNull(sessionStruct) && Objects.nonNull(properties)) {
            sessionStruct.setAttribute("channelId", String.valueOf(properties.getChannelId()));
            sessionStruct.setAttribute("createdFrom", String.valueOf(properties.getCreatedFrom()));
            sessionStruct.setAttribute("appChannel", String.valueOf(properties.getAppChannel()));
            sessionStruct.setTenantId(tenantId);
            sessionStruct.getValues().setLoginProperties(properties);
            persistSession(sessionStruct.getSid(), sessionStruct.getValues(), tenantId);
        } else {
            sessionStruct = createSessionAndPersist(user, properties, tenantId);
        }
        String uuid = user.getUuid();
        AuthBean authBean = new AuthBean();
        authBean.setPhoneNo(user.getPhoneNo());
        authBean.setToken(sessionStruct.getSid());
        authBean.setUuid(uuid);
        authBean.setHasPassword(user.getHasPassword());
        log.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{},channelId:{}", properties.getCreatedFrom(), user.getPhoneNo(), properties.getAppChannel(), properties.getChannelId());

        // 添加登陆日志
        loginRecordService.saveLoginRecord(user.getId(), RecordType.LOGINRECORD.getName(), loginType);

        if (send) {
            //更新user_tag记录
            UserLoginEvent userLoginEvent = new UserLoginEvent(this, UserTag.builder()
                    .userId(user.getId()).registeredFrom(properties.getCreatedFrom()).tenantId(user.getTenantId()).build());
            applicationEventPublisher.publishEvent(userLoginEvent);
        }

        return authBean;
    }


    @Override
    public SessionStruct createSessionAndPersist(User user, LoginProperties properties, Integer tenantId) {
        SessionStruct sessionStruct;
        //获取sessionid
        String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties, tenantId);
        if (StringUtils.length(sessionId) > 0) {
            sessionStruct = findSessionBySessionId(sessionId, tenantId);
            if (sessionStruct == null) {
                sessionStruct = newSession(user, properties);
            } else {
                sessionStruct.getValues().setLoginProperties(properties);
            }
            persistSession(sessionStruct.getSid(), sessionStruct.getValues(), tenantId);
        } else {
            sessionStruct = newSession(user, properties);
            persistSession(sessionStruct.getSid(), sessionStruct.getValues(), tenantId);
        }
        return sessionStruct;
    }

    private String findSessionIdByUserIdLoginProperties(Long userId, LoginProperties properties, Integer tenantId) {
        return stringRedisTemplate.opsForValue().get(generateLoginPropertiesKey(userId, properties, tenantId));
    }

    /**
     * 每个渠道都有不同的 session
     *
     * @param userId     用户 ID
     * @param properties baitiao/xyqb/vcc ... + 用户注册来源
     * @return redisKey. 用来标识这个渠道的用户 Session 是否存在
     */
    private String generateLoginPropertiesKey(Long userId, LoginProperties properties, Integer tenantId) {
        if (ObjectUtils.isEmpty(properties.getTenantId())) {
            return Constants.Session.USER_SESSION_ID_CACHE + ":" + userId + ":" + properties.getMerchantName() + ":" + properties.getCreatedFrom();
        } else if (properties.getTenantId().equals(0) || TenantUtil.TENANT_DEFAULT.equals(properties.getTenantId())) {
            return Constants.Session.USER_SESSION_ID_CACHE + ":" + userId + ":" + properties.getMerchantName() + ":" + properties.getCreatedFrom();
        } else {
            Integer key;
            if (tenantId != null) {
                key = tenantId;
            } else {
                key = properties.getTenantId();
            }
            return Constants.Session.USER_SESSION_ID_CACHE + ":" + userId + ":" + properties.getMerchantName() + ":" + properties.getCreatedFrom() + ":" + key;
        }
    }

    private String findSessionValueBySessionId(String sessionId, Integer tenantId) {
        String tokenKey2;
        if (tenantId == null || UserConstant.defaultTenantId.equals(tenantId)) {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + sessionId;
        } else {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + sessionId;
        }

        String result = stringRedisTemplate.opsForValue().get(tokenKey2);
        return StringUtils.defaultString(result, "");
    }

    private SessionStruct newSession(User user, LoginProperties properties) {
        Timestamp now = new Timestamp(System.currentTimeMillis());
        SessionStruct sessionStruct = new SessionStruct();
        SessionValue sessionValue = new SessionValue();
        sessionStruct.setSid(prefix + UUID.randomUUID().toString());
        sessionValue.setCreatedAt(now);
        sessionValue.setLastAccessTime(now);
        sessionValue.setUser(user);
        sessionValue.setLoginProperties(properties);
        Map<String, String> values = new HashMap<>();
        sessionValue.setValues(values);
        sessionStruct.setValues(sessionValue);
        // 发送登陆成功统计消息
        /*
         * 部分免密渠道登录统计，用户中心不需识别，由统计平台来过滤
         * 贷款导航(84660)；壹账通H5(159384)
         */
        User userRereset = new User();
        BeanUtils.copyProperties(user, userRereset);
        userRereset.setRegisteredFrom(properties.getCreatedFrom());
        UserStatistics statistics = new UserStatistics(userRereset, null, properties.getAction(), properties.getChannelId());
        // 推送老的登陆统计信息
        MqUtils.sendLoanVest(statistics);
        return sessionStruct;
    }

    @Override
    @UserBtRegisterFill
    public void persistSession(String token, SessionValue sessionValue, Integer tenantId) {
        persistSession(token, sessionValue, Constants.Session.SESSION_VALID_TIME, tenantId);
    }

    @Override
    public void persistSession(String token, SessionValue sessionValue, Long time, Integer tenantId) {
        Timestamp current = new Timestamp(System.currentTimeMillis());
        if (sessionValue == null) {
            sessionValue = new SessionValue();
        }

        if (sessionValue.getLoginProperties() == null) {
            sessionValue.setLoginProperties(new LoginProperties());
        }


        LoginProperties loginProperties = sessionValue.getLoginProperties();
        loginProperties.setTenantId(tenantId);
        sessionValue.setLoginProperties(loginProperties);

        sessionValue.setLastAccessTime(current);
        String json = JSON.toJSONString(sessionValue);
        String key;
        if (tenantId == null || UserConstant.defaultTenantId.equals(tenantId)) {
            key = Constants.Session.USER_SESSION_CACHE + token;
        } else {
            key = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + token;
        }

        stringRedisTemplate.opsForValue().set(key, json,
                time, TimeUnit.SECONDS);
        if (sessionValue.getUser() != null) {
            String generateLoginPropertiesKey = generateLoginPropertiesKey(sessionValue.getUser().getId(), sessionValue.getLoginProperties(), tenantId);
            stringRedisTemplate.opsForValue().set(generateLoginPropertiesKey, token, time, TimeUnit.SECONDS);
            log.info("[Session生命期延续],token:{},有效期:[24Hour]", token);
            setUserIdTokenKeys(sessionValue.getUser().getId(), generateLoginPropertiesKey, tenantId);
        }
    }

    /**
     * 设置用户token集合方便注销使用
     *
     * @param userId
     * @param key
     */
    private void setUserIdTokenKeys(long userId, String key, Integer tenantId) {
        if (0L != userId) {
            String setKey = getUserSessionSetKey(userId, tenantId);
            try {
                stringRedisTemplate.opsForSet().add(setKey, key);
                stringRedisTemplate.expire(setKey, Constants.Session.SESSION_VALID_TIME, TimeUnit.SECONDS);
            } catch (Exception e) {
                log.error("存储用户注销件失败，userId:{},Exception:{}", userId, e);
            }

        }
    }

    private SessionStruct findSessionBySessionId(String sessionId, Integer tenantId) {
        String sessionValue = findSessionValueBySessionId(sessionId, tenantId);
        if (StringUtils.isEmpty(sessionValue)) {
            log.warn("[SessionServiceImpl][findSessionBySessionId] session data 未找到：sid:{}", sessionId);
            return null;
        }
        try {
            SessionValue value = JSON.parseObject(sessionValue, SessionValue.class);
            if (null == value) {
                log.warn("[SessionServiceImpl][findSessionBySessionId] session data 未找到：sid:{},sessionValue:{}", sessionId, sessionValue);
                return null;
            }
            SessionStruct struct = new SessionStruct();
            struct.setSid(sessionId);
            struct.setValues(value);
            return struct;
        } catch (Exception e) {
            log.warn("[SessionServiceImpl][findSessionBySessionId] 序列化SessionValue出错：sid:{},sessionValue:{}", sessionId, sessionValue, e);
            return null;
        }

    }

    @Override
    @CacheEvict(value = "userextinfocache", key = "'extinfo' + #userId + '-' + #tenantId", cacheManager = "cacheManager")
    public void deleteByUserId(long userId, Integer tenantId) {
        //1.删除session关联
        String setKey = getUserSessionSetKey(userId, tenantId);
        Set useIdKeys = stringRedisTemplate.opsForSet().members(setKey);
        if (!CollectionUtils.isEmpty(useIdKeys)) {
            useIdKeys.forEach(key -> {
                log.info("删除用户userId={}的缓存信息", userId);
                String token = stringRedisTemplate.opsForValue().get(String.valueOf(key));
                stringRedisTemplate.delete(getUserTokenKey(token, tenantId));
                stringRedisTemplate.delete(String.valueOf(key));
            });
            //2.删除session缓存健
            stringRedisTemplate.delete(setKey);
        }
    }

    private String getUserTokenKey(String token, Integer tenantId) {
        String tokenKey2;
        if (tenantId == null || UserConstant.defaultTenantId.equals(tenantId)) {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
        } else {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + token;
        }
        return tokenKey2;
    }

    /**
     * 删除注销后缓存查询结果
     *
     * @param user
     */
    @Caching(evict = {@CacheEvict(value = "usercache", key = "'xyqbuser' + #user.phoneNo", cacheManager = "cacheManager"),
            @CacheEvict(value = "usercache", key = "'xyqbuser' + #user.uuid", cacheManager = "cacheManager"),
            @CacheEvict(value = "addresscache", key = "'address' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "contact", key = "'contact' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "userSpouseCache", key = "'spouse' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "btRegisterCache", key = "'userId' + #user.id", cacheManager = "cacheManager")})
    @Override
    public void deleteUserCatch(User user, Integer tenantId) {

    }

    @Override
    public List<SessionStruct> findByUserId(long userId, Integer tenantId) {
        List<SessionStruct> sessionStructList = new ArrayList<>();
        String setKey = getUserSessionSetKey(userId, tenantId);
        Set<String> userIdKeys = stringRedisTemplate.opsForSet().members(setKey);
        if (CollectionUtils.isEmpty(userIdKeys)) {
            return sessionStructList;
        }
        for (String userIdKey : userIdKeys) {
            String sessionId = stringRedisTemplate.opsForValue().get(userIdKey);
            SessionStruct sessionStruct = findSessionBySessionId(sessionId, tenantId);
            sessionStructList.add(sessionStruct);
        }
        return sessionStructList;
    }

    @Override
    public void persistSession(List<SessionStruct> sessionStructList, Integer tenantId) {
        for (SessionStruct sessionStruct : sessionStructList) {
            String sid = sessionStruct.getSid();
            SessionValue values = sessionStruct.getValues();
            persistSession(sid, values, tenantId);
        }
    }

    @Override
    public void deleteSession(String token, Integer tenantId) {

        SessionStruct sessionStruct = findSessionBySessionId(token, tenantId);

        if (null != sessionStruct) {

            SessionValue values = sessionStruct.getValues();
            User user = values.getUser();
            if (!user.getTenantId().equals(tenantId)) {
                log.info("登出失败，token对应的用户和租户信息不匹配，token:{} , user: {}", token, JSON.toJSONString(user));
                return;
            }

            String tokenKey2;
            if (UserConstant.defaultTenantId.equals(tenantId)) {
                tokenKey2 = Constants.Session.USER_SESSION_CACHE + sessionStruct.getSid();
            } else {
                tokenKey2 = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + sessionStruct.getSid();
            }

            stringRedisTemplate.delete(tokenKey2);


            String key = generateLoginPropertiesKey(user.getId(), values.getLoginProperties(), tenantId);

            stringRedisTemplate.delete(key);

            log.info("登出成功token:{} , user: {}", token, user);
        }
    }

    @Override
    public void persistSessionExchange(String token, SessionValue sessionValue, long expire, Integer tenantId) {
        Timestamp current = new Timestamp(System.currentTimeMillis());
        sessionValue.setLastAccessTime(current);
        String json = JSON.toJSONString(sessionValue);

        String tokenKey2;
        if (UserConstant.defaultTenantId.equals(tenantId)) {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
        } else {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + token;
        }
        // 单用户单次访问的最大的平均访问时间 SESSION_EXCHANGE_VALID_TIME
        // 获取下expire和SESSION_EXCHANGE_VALID_TIME的最小值
        long expireTime = Math.min(expire, Constants.Session.SESSION_EXCHANGE_VALID_TIME);
        stringRedisTemplate.opsForValue().set(tokenKey2, json,expireTime, TimeUnit.SECONDS);
        String key = generateLoginPropertiesKey(sessionValue.getUser().getId(), sessionValue.getLoginProperties(), tenantId);
        stringRedisTemplate.opsForValue().set(key, token, expireTime, TimeUnit.SECONDS);
        setUserIdTokenKeys(sessionValue.getUser().getId(), key, tenantId);
    }


    public void kdspDeleteSession(Long userId, LoginProperties loginProperties, Integer tenantId) {
        List<Long> createFromList = Arrays.asList(214L, 217L);
        for (Long createFrom : createFromList) {
            loginProperties.setCreatedFrom(createFrom);
            String key = generateLoginPropertiesKey(userId, loginProperties, tenantId);
            String token = stringRedisTemplate.opsForValue().get(key);
            deleteSession(token, tenantId);
        }
    }


    /**
     * 获取用户的会话缓存Set的Redis-Key
     *
     * @param userId - 用户主键
     * @return
     */
    private String getUserSessionSetKey(long userId, Integer tenantId) {
        return Constants.Session.USER_SESSION_KEY_SET + userId;
    }
}
