package cn.quantgroup.xyqb.controller.internal.sms;

import cn.quantgroup.sms.MsgParams;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaNewValidator;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.util.DateUtils;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by FrankChow on 15/7/6.
 */
@RestController
@RequestMapping("/api/sms")
public class SmsController implements IBaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(SmsController.class);
  private static final Random random = new Random();
  private static final long EXPIRE_MINUTES = 10;
  @Autowired
  private ISmsService smsService;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;
  @Value("${sms.is.debug}")
  private boolean smsIsDebug;
  private static final String IMAGE_IP_COUNT = "image:ip";
  private static final String IMAGE_PHONE_COUNT = "image:phone";
  private static final String IMAGE_DEVICEID_COUNT = "image:deviceId:";

  private static final Long IP_MAX_PER_DAY = 5000L;//ip上限
  private static final Long PHONE_MAX_PER_DAY = 20L;//手机号短信上限
  private static final Long PHONE_VOICE_MAX_PER_DAY = 5L;//手机号语音上限
  private static final Long DEVICE_MAX_PER_DAY = 20L;//设备每天上限
  /**
   * 短信验证码: for H5
   * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
   */
  @CaptchaValidator
  @RequestMapping("/send_sms_verification_code")
  public JsonResult verifyPhoneNoH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  @CaptchaValidator
  @RequestMapping("/send_reset_code")
  public JsonResult resetPasswordH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  /**
   * 短信验证码: for H5
   * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
   */
  @CaptchaValidator
  @RequestMapping("/send_sms_verification_code_voice")
  public JsonResult verifyPhoneNoH5New(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"4".equals(usage)) {
      LOGGER.error("参数校验失败,用户注册语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  @CaptchaValidator
  @RequestMapping("/send_reset_code_voice")
  public JsonResult resetPasswordH5New(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"5".equals(usage)) {
      LOGGER.error("参数校验失败,重置密码语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  /**
   * 快速登陆发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_login_code_voice")
  public JsonResult sendLoginCodeNew(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"6".equals(usage)) {
      LOGGER.error("参数校验失败,用户登录语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  /**
   * 快速登陆发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_login_code")
  public JsonResult sendLoginCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  /**
   * 快速注册发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_regist_code")
  public JsonResult sendRegistCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("快速注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  private JsonResult sendVerificationCode2(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }

    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      return JsonResult.buildErrorStateResult("1分钟内不能重复获取验证码", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    List<String> newList = new ArrayList<>();
    newList.add(randomCode);
        /*ConfirmableMsg confirmableMsg = new ConfirmableMsg(
                uniqueId, newList, "1", "1", phoneNo
        );*/
    MsgParams message = new MsgParams(Collections.singletonList(2), phoneNo, "1", "1", Collections.singletonList(randomCode), uniqueId);
    try {
      //smsService.getSmsSender().sendConfirmableMessage(confirmableMsg);
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      deleteRetSendCode(phoneNo);//删除用户重置密码，多次错误逻辑
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送短信验证码失败");
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }

  private JsonResult sendVerificationCode2Voice(String phoneNo, String usage) {
    String verificationCountKey = Constants.REDIS_VOICE_CODE_COUNT + phoneNo;
    Long getVerificationCount = redisTemplate.opsForHash().increment(verificationCountKey, usage.toString(), 1);
    redisTemplate.expire(verificationCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
    if (getVerificationCount > 5) {
      return JsonResult.buildErrorStateResult("今天已获取5次语音验证码，请使用短信验证码或明天再试", null);
    }

    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }
    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      return JsonResult.buildErrorStateResult("1分钟内不能重复获取验证码", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    MsgParams message = new MsgParams(Collections.singletonList(4), phoneNo, "1", "4", Collections.singletonList(randomCode), uniqueId);
    try {
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      deleteRetSendCode(phoneNo);//删除用户重置密码，多次错误逻辑
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送语音短信验证码失败");
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }

  /**
   * 快速登陆发送验证码新版
   */
  @CaptchaNewValidator
  @RequestMapping("/send_login_code_voice_new")
  public JsonResult sendLoginCodeVoiceNew(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage, @RequestParam(required = false) String deviceId) {
    if (StringUtils.isEmpty(usage) || !"6".equals(usage)) {
      LOGGER.error("参数校验失败,用户登录语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);

    return sendVerificationCode2VoiceNew(phoneNo, usage, deviceId);
  }

  /**
   * 快速登陆发送短信验证码
   */
  @CaptchaNewValidator
  @RequestMapping("/send_login_code_new")
  public JsonResult sendLoginSmsCodeNew(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom, @RequestParam(required = false) String deviceId) {
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2New(phoneNo, deviceId, true);
  }
  /**
   * 快速登陆发送短信验证码
   */
  @CaptchaNewValidator
  @RequestMapping("/send_login_code_new_forH5")
  public JsonResult sendLoginSmsCodeNewForH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom, @RequestParam(required = false) String deviceId) {
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2New(phoneNo, deviceId, false);
  }

  /**
   * 新版本验证码
   */
  private JsonResult sendVerificationCode2New(String phoneNo, String deviceId, boolean isApp) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }
    String verificationPhoneCountKey = Constants.REDIS_SMS_CODE_COUNT + phoneNo;
    Long getPhoneVerificationCount = redisTemplate.opsForHash().increment(verificationPhoneCountKey, Constants.REDIS_SMS_CODE_COUNT, 1);
    redisTemplate.expire(verificationPhoneCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
    String clientIp = getIp();
    if (getPhoneVerificationCount > PHONE_MAX_PER_DAY) {
      LOGGER.info("您手机号已经达到获取今天短信验证码上限:phoneNo:{},deviceId:{},ip:{}",phoneNo,deviceId,clientIp);
      return JsonResult.buildErrorStateResult("今天已获取20次短信验证码，请使用语音验证码或明天再试", null);
    }
    // Todo - 运维解决真实IP获取问题后，打开这段代码，实现按IP限制短信验证码获取量
    // Todo - 另：当前的计数器计数方式为乐观累加，而且还是提前计数，会导致边界值问题，即临界次数会提前一次记满，并且在后续请求到达时计数器会继续计数
    /*
    if (!StringUtils.isEmpty(clientIp)) {
      String verificationIPCountKey = Constants.REDIS_SMS_IP_COUNT+clientIp;
      Long getIPVerificationCount = redisTemplate.opsForHash().increment(verificationIPCountKey, Constants.REDIS_SMS_IP_COUNT, 1);
      if (getIPVerificationCount > IP_MAX_PER_DAY) {
        return JsonResult.buildErrorStateResult("您当前ip已经达到获取今天验证码上限", null);
      }
    }*/
    LOGGER.info("请求短信新版本接口:phoneNo:{},deviceId:{},IP:{}",phoneNo,deviceId,clientIp);
    if (!StringUtils.isEmpty(deviceId)) {
      String verificationDeviceCountKey = Constants.REDIS_SMS_DEVICE_COUNT + deviceId;
      Long getDeviceVerificationCount = redisTemplate.opsForHash().increment(verificationDeviceCountKey, Constants.REDIS_SMS_DEVICE_COUNT, 1);
      redisTemplate.expire(verificationDeviceCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
      if (getDeviceVerificationCount > DEVICE_MAX_PER_DAY) {
        LOGGER.info("您设备已经达到获取今天短信验证码上限:phoneNo:{},deviceId:{},ip:{}",phoneNo,verificationDeviceCountKey,clientIp);
        return JsonResult.buildErrorStateResult("您设备已经达到获取今天短信验证码上限", null);
      }
    }

    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      LOGGER.info("sendVerificationCode2New1分钟内不能重复获取验证码:phoneNo:{},deviceId:{},ip:{}",phoneNo,deviceId,clientIp);
      return JsonResult.buildErrorStateResult("1分钟内不能重复获取验证码", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    List<String> newList = new ArrayList<>();
    newList.add(randomCode);
    MsgParams message = new MsgParams(Collections.singletonList(2), phoneNo, "1", "1", Collections.singletonList(randomCode), uniqueId);
    try {
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      deleteRetSendCode(phoneNo);//删除用户重置密码，多次错误逻辑
      if(isApp && needImageVlidate(clientIp,deviceId,phoneNo)){
        return JsonResult.buildSuccessResult("发送成功", uniqueId,0003L);
      }
      LOGGER.info("sendVerificationCode2New获取短信成功:phone:{},deviceId:{},ip:{}",phoneNo,deviceId,clientIp);
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送短信验证码失败:phone:{},deviceId:{},ip:{}",phoneNo,deviceId,clientIp);
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }

  /**
   * 新版本语音验证码
   */
  private JsonResult sendVerificationCode2VoiceNew(String phoneNo, String usage, String deviceId) {

    String verificationCountKey = Constants.REDIS_VOICE_CODE_COUNT + phoneNo;
    Long getVerificationCount = redisTemplate.opsForHash().increment(verificationCountKey, usage.toString(), 1);
    redisTemplate.expire(verificationCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
    if (getVerificationCount > PHONE_VOICE_MAX_PER_DAY) {
      return JsonResult.buildErrorStateResult("今天已获取5次语音验证码，请使用短信验证码或明天再试", null);
    }
    String clientIp = getIp();
    // Todo - 运维解决真实IP获取问题后，打开这段代码，实现按IP限制短信验证码获取量
    // Todo - 另：当前的计数器计数方式为乐观累加，而且还是提前计数，会导致边界值问题，即临界次数会提前一次记满，并且在后续请求到达时计数器会继续计数
  /*  if (!StringUtils.isEmpty(clientIp)) {
      String verificationIPCountKey = Constants.REDIS_VOICE_IP_COUNT+clientIp;
      Long getIPVerificationCount = redisTemplate.opsForHash().increment(verificationIPCountKey, Constants.REDIS_VOICE_IP_COUNT, 1);
      if (getIPVerificationCount > IP_MAX_PER_DAY) {
        return JsonResult.buildErrorStateResult("您当前ip已经达到获取今天语音验证码上限", null);
      }
    }*/

    if (!StringUtils.isEmpty(deviceId)) {
      String verificationDeviceCountKey = Constants.REDIS_VOICE_DEVICE_COUNT + deviceId;
      Long getDeviceVerificationCount = redisTemplate.opsForHash().increment(verificationDeviceCountKey, Constants.REDIS_VOICE_DEVICE_COUNT, 1);
      redisTemplate.expire(verificationDeviceCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
      if (getDeviceVerificationCount > DEVICE_MAX_PER_DAY) {
        return JsonResult.buildErrorStateResult("您设备已经达到获取今天语音验证码上限", null);
      }
    }


    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }
    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      return JsonResult.buildErrorStateResult("1分钟内不能重复获取验证码", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    MsgParams message = new MsgParams(Collections.singletonList(4), phoneNo, "1", "4", Collections.singletonList(randomCode), uniqueId);
    try {
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      deleteRetSendCode(phoneNo);//删除用户重置密码，多次错误逻辑
      if(needImageVlidate(clientIp,deviceId,phoneNo)){

        return JsonResult.buildSuccessResult("发送成功", uniqueId,0003L);
      }
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送语音短信验证码失败");
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }

  /**
   * 判断下次是否提示图形验证码
   * @param clientIp
   * @param deviceId
   * @param phoneNo
   * @return
   */
  private boolean needImageVlidate(String clientIp, String deviceId, String phoneNo) {
    boolean need = false;
    String countIP = redisTemplate.opsForValue().get(IMAGE_IP_COUNT + clientIp);
    String countDeviceId = redisTemplate.opsForValue().get(IMAGE_DEVICEID_COUNT + deviceId);
    String countPhoneNo = redisTemplate.opsForValue().get(IMAGE_PHONE_COUNT + phoneNo);
    Long ip = StringUtils.isBlank(countIP) ? 1L : Long.valueOf(countIP);
    Long devId = StringUtils.isBlank(countDeviceId) ? 1L : Long.valueOf(countDeviceId);
    Long phNo = StringUtils.isBlank(countPhoneNo) ? 1L : Long.valueOf(countPhoneNo);
    if (ip >= Constants.Image_Need_Count || devId >= Constants.Image_Need_Count  || phNo >= Constants.Image_Need_Count ) {
      need = true;
    }
    return need;
  }

  /**
   * 删除用户重置密码时短信验证错误
   * @param phoneNo
   */
  private  void deleteRetSendCode(String phoneNo){
    String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT+phoneNo;
   redisTemplate.opsForHash().delete(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT);
  }

}
