package cn.quantgroup.xyqb.controller.internal.user.center;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.forbidden.AccessForbiddenValidator;
import cn.quantgroup.xyqb.aspect.limit.PasswordFreeAccessValidator;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.TenantUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by 11 on 2017/3/22.
 */
@Slf4j
@RestController
@RequestMapping("/user/center")
public class UserCenterController implements IBaseController {

    @Autowired
    private IUserInfoRepository userInfoRepository;

    @Autowired
    private IUserService userService;
    @Autowired
    private ApplicationEventPublisher applicationEventPublisher;

    @GetMapping("/attach")
    public JsonResult queryUserAttach(Long userId, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (userId == null) {
            return JsonResult.buildErrorStateResult("用户 ID 不能为空", null);
        }
        UserInfoEntity userAttached = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        return JsonResult.buildSuccessResultGeneric(userAttached);
    }

    /**
     * @param userIds
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/17291
     */
    @PostMapping("/attach/list")
    public JsonResult queryUserAttachList(@RequestBody List<Long> userIds, @RequestParam(required = false) Integer tenantId) {
        //默认羊小咩租户
        if (tenantId == null) {
            tenantId = getTenantId();
        }
        if (CollectionUtils.isEmpty(userIds)) {
            return JsonResult.buildErrorStateResult("用户 ID 不能为空", null);
        }
        int size = 500;
        if (userIds.size() > size) {
            return JsonResult.buildErrorStateResult("超出最大条数限制" + size, null);
        }

        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        List<UserInfoEntity> userInfoEntityList = userInfoRepository.findByTenantIdAndUserIdIn(tenantId,userIds);
        return JsonResult.buildSuccessResultGeneric(userInfoEntityList);
    }


    /**
     * 根据用户手机号查询昵称.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/queryNick")
    public JsonResult queryUserNick(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if (null != userInfoEntity) {
            return JsonResult.buildSuccessResult(null, userInfoEntity.getNickName());
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 修改用户的昵称
     *
     * @param phoneNo
     * @param nick
     * @return
     */
    @RequestMapping("/saveNick")
    public JsonResult saveUserNick(String phoneNo, String nick, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setNickName(nick);
        userInfoRepository.save(userInfoEntity);
        return JsonResult.buildSuccessResult(null, userInfoEntity.getNickName());
    }

    @RequestMapping("/kdsp/saveNick")
    public JsonResult saveUserNickForKdsp(Long userId, String nick, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setNickName(nick);
        userInfoRepository.save(userInfoEntity);
        return JsonResult.buildSuccessResult(null,  userInfoEntity.getNickName());
    }

    @RequestMapping("/kdsp/save/avatar")
    public JsonResult saveUserAvatarAddrForKdsp(Long userId, String avatarUrl, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(avatarUrl)) {
            log.error("参数不合法:avatarUrl:{}}", avatarUrl);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setPhoto(avatarUrl);
        userInfoRepository.save(userInfoEntity);

        return JsonResult.buildSuccessResult("保存成功", userInfoEntity);
    }

    /**
     * 上传后调用该接口保存用户头像
     *
     * @param phoneNo   用户手机号
     * @param avatarUrl 上传后生成的头像地址
     * @return
     */
    @RequestMapping("/save/avatar")
    public JsonResult SaveUserAvatarAddr(String phoneNo, String avatarUrl, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(avatarUrl) || StringUtils.isBlank(phoneNo)) {
            log.error("参数不合法:avatarUrl:{}, phoneNo:{}", avatarUrl, phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        if (StringUtils.isBlank(avatarUrl)) {
            avatarUrl = Constants.UserAvatar.AVATAR_DEFAULT;
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setPhoto(avatarUrl);
        userInfoRepository.save(userInfoEntity);

        return JsonResult.buildSuccessResult("保存成功", userInfoEntity);
    }

    /**
     * 查询用户是否实名认证.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/query/verified")
    public JsonResult queryVerified(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        if (null != userDetail) {
            return JsonResult.buildSuccessResult(null, userDetail);
        }
        return JsonResult.buildSuccessResult(null, null);
    }


    /**
     * 个人资料信息
     *
     * @param phoneNo
     * @return
     */
    @AccessForbiddenValidator
    @PasswordFreeAccessValidator
    @RequestMapping("/personalData")
    public JsonResult personalData(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        Map<String, Object> result = new HashMap<>();

        return JsonResult.buildSuccessResult(null, result);
    }


    /**
     * 根据手机号查询用户id, nodejs调用接口无法获取到userId.
     * 所以增加该接口来查询用户id
     *
     * @param phoneNo
     * @return
     */
    private Long queryUserId(String phoneNo, Integer tenantId) {
        User user = userService.findByPhoneInDb(phoneNo, tenantId);
        return null == user ? null : user.getId();
    }

}
