package cn.quantgroup.xyqb.service.config.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.Configuration;
import cn.quantgroup.xyqb.repository.IConfigurationRepository;
import cn.quantgroup.xyqb.service.config.IConfigurationService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * @author mengfan.feng
 * @time 2015-10-27 15:22
 */
@Service
public class ConfigurationServiceImpl implements IConfigurationService {

    @Autowired
    private IConfigurationRepository configurationRepository;

    private static final Map<String, String> CONFIGURATION_MAP = new HashMap<>();

    private static final Map<String, Pattern> IP_PATTERN_MAP = new HashMap<>();

    private static final Pattern INNER_IP_PATTERN = Pattern.compile("^172\\.16\\..*$");

    @PostConstruct
    private void init() {
        List<Configuration> configurations = configurationRepository.findSafeIp();
        List<Configuration> captchaConfig = configurationRepository.findByKeyLike(Constants.CONFIG_CAPTCHA);
        configurations.addAll(captchaConfig);
        if (CollectionUtils.isEmpty(configurations)) {
            return;
        }

        configurations.forEach(e -> CONFIGURATION_MAP.put(e.getKey(), e.getValue()));
    }

    @Override
    public String getValue(String key) {
        return CONFIGURATION_MAP.get(key);
    }

    @Override
    public void resetConfiguration() {
        CONFIGURATION_MAP.clear();
        IP_PATTERN_MAP.clear();

        this.init();
    }

    @Override
    public boolean validateIp(String loginFrom, String ip) {
        if (StringUtils.isEmpty(loginFrom) || StringUtils.isEmpty(ip)) {
            return false;
        }

        String key = "safe_ip_login_from_" + loginFrom;

        String value = this.getValue(key);

        if (StringUtils.isEmpty(value)) {
            return false;
        }

        if (StringUtils.contains(value, ip)) {
            return true;
        }

        Pattern pattern = IP_PATTERN_MAP.get(key);

        if (pattern == null) {
            pattern = Pattern.compile(value);

            IP_PATTERN_MAP.put(key, pattern);
        }

        return pattern.matcher(ip).matches() || INNER_IP_PATTERN.matcher(ip).matches();
    }
}
