package cn.quantgroup.xyqb.controller.internal.config;

import cn.quantgroup.xyqb.aspect.fplock.FPLock;
import cn.quantgroup.xyqb.aspect.fplock.FirstParamLockAspect;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.sms.SmsConfig.SMS;
import cn.quantgroup.xyqb.service.config.IConfigurationService;
import cn.quantgroup.xyqb.thirdparty.jcaptcha.AbstractManageableImageCaptchaService;
import cn.quantgroup.xyqb.util.PasswordUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.MatrixVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import static cn.quantgroup.xyqb.model.sms.SmsConfig.SMS_INDEX_KEY;

/**
 * @author mengfan.feng
 * @time 2015-10-28 11:37
 */
@RestController
@RequestMapping("/config")
public class ConfigController {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigController.class);

    @Value("${configuration.key}")
    private String configKey;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Autowired
    @Qualifier("customCaptchaService")
    private AbstractManageableImageCaptchaService imageCaptchaService;

    @Autowired
    private IConfigurationService configurationService;

    @Autowired
    private FirstParamLockAspect fpLockAspect;

    /**
     * 短信平台配置
     *
     * @param key 口令
     * @param sw  配置参数
     * @param in  短信发送下标
     * @return
     */
    @RequestMapping("/sms")
    public JsonResult smsConfig(String key, String sw, String in) {
        LOGGER.info("短信平台配置, key:{}, sw:{}", key, sw);

        if (!StringUtils.equals(configKey, PasswordUtil.MD5(key))) {
            return JsonResult.buildErrorStateResult("口令错误", null);
        }

        if (StringUtils.isNotEmpty(sw)) {
            switch (sw) {
                case "0":
                    for (SMS sms : SMS.values()) {
                        sms.setEnabled(false);
                    }
                    break;
                case "1":
                    for (SMS sms : SMS.values()) {
                        sms.setEnabled(true);
                    }
                    break;
                default:
                    SMS[] smses = SMS.values();
                    String[] groups = sw.split(";");
                    for (String group : groups) {
                        String[] strs = group.split(",");
                        smses[Integer.parseInt(strs[0])].setEnabled("1".equals(strs[1]));
                    }
                    break;
            }
        }

        int smsIndex = Integer.parseInt(redisTemplate.opsForValue().get(SMS_INDEX_KEY));
        if (StringUtils.isNotEmpty(in)) {
            int index = Integer.parseInt(in);
            smsIndex = index < 0 || index >= SMS.values().length ? -1 : index;
        }

        int first = -1;
        StringBuilder builder = new StringBuilder();
        for (SMS sms : SMS.values()) {
            builder.append(sms.ordinal()).append(",").append(sms.isEnabled() ? 1 : 0).append(";");
            if (first < 0 && sms.isEnabled()) {
                first = sms.ordinal();
            }
        }

        smsIndex = smsIndex >= 0 && first >= 0 ? smsIndex : first;

        builder.append(smsIndex);

        redisTemplate.opsForValue().set(SMS_INDEX_KEY, smsIndex + "");

        return JsonResult.buildSuccessResult("短信平台配置完成", builder.toString());
    }

    /**
     * 重置 Configuration
     *
     * @param key 口令
     * @return
     */
    @RequestMapping("/reset")
    public JsonResult resetConfig(String key) {
        LOGGER.info("重置 Configuration, key:{}", key);

        if (!StringUtils.equals(configKey, PasswordUtil.MD5(key))) {
            return JsonResult.buildErrorStateResult("口令错误", null);
        }

        configurationService.resetConfiguration();

        return JsonResult.buildSuccessResult("Configuration 重置完成", null);
    }

    /**
     * 设置短信发送限制
     * WGET /sms/send/limitation/day/duration=1;limit=5/hour/duration=1;limit=3
     *
     * @param key
     * @return
     */
    @RequestMapping("/smssend/limitation/{dayparam}/{hourparam}")
    public JsonResult configSmsSendLimitaion(String key,
                                             @MatrixVariable(pathVar = "dayparam", required = false) Map<String, List<String>> dayparam,
                                             @MatrixVariable(pathVar = "hourparam", required = false) Map<String, List<String>> hourparam) {

        LOGGER.info("手动设置短信发送限制, {}, {}", dayparam, hourparam);

        if (!StringUtils.equals(configKey, PasswordUtil.MD5(key))) {
            return JsonResult.buildErrorStateResult("口令错误", null);
        }

        if (dayparam == null || hourparam == null) {
            return JsonResult.buildErrorStateResult("参数错误", "");
        }

        try {
            Class<?> clz = Class.forName("cn.quantgroup.xyqb.controller.internal.sms.SmsController");
            Stream.of(clz.getDeclaredMethods()).filter(method -> method.getAnnotation(FPLock.class) != null).forEach(this::parseFpLockAnnotaion);
        } catch (ClassNotFoundException e) {
            return JsonResult.buildErrorStateResult("设置短信发送限制失败", e.getMessage());
        }

        String smsSendApiName = "verifyPhoneNoH5:46:";
        setRestriction(smsSendApiName, TimeUnit.DAYS, dayparam.get("duration").get(0), dayparam.get("limit").get(0));
        setRestriction(smsSendApiName, TimeUnit.HOURS, hourparam.get("duration").get(0), hourparam.get("limit").get(0));

        String smdResetApiName = "resetPasswordH5:54:";
        setRestriction(smdResetApiName, TimeUnit.DAYS, dayparam.get("duration").get(0), dayparam.get("limit").get(0));
        setRestriction(smdResetApiName, TimeUnit.HOURS, hourparam.get("duration").get(0), hourparam.get("limit").get(0));

        LOGGER.info("设置短信发送限制成功, 天:{}, 小时:{}", dayparam, hourparam);

        Map<String, Object> resultData = new HashMap<>();
        resultData.put("注册发送短信接口按小时限制", fpLockAspect.readLimitation(smsSendApiName + TimeUnit.HOURS).desc());
        resultData.put("注册发送短信接口按天限制", fpLockAspect.readLimitation(smsSendApiName + TimeUnit.DAYS).desc());
        resultData.put("密码重置发送短信接口按小时限制", fpLockAspect.readLimitation(smdResetApiName + TimeUnit.HOURS).desc());
        resultData.put("密码重置发送短信接口按天限制", fpLockAspect.readLimitation(smdResetApiName + TimeUnit.DAYS).desc());


        return JsonResult.buildSuccessResult("已设置短信发送限制", resultData);
    }



    @RequestMapping("/captcha/state")
    public JsonResult state(String key) {
        if (!StringUtils.equals(configKey, PasswordUtil.MD5(key))) {
            return JsonResult.buildErrorStateResult("口令错误", null);
        }
        return JsonResult.buildSuccessResult("验证码状态", getCaptchaEngineState(0));
    }

    /**
     * 手动执行垃圾, 即超时验证码清理
     *
     * @return
     */
    @RequestMapping("/captcha/garbage-collect")
    public JsonResult garbageCollectManually(String key) {
        if (!StringUtils.equals(configKey, PasswordUtil.MD5(key))) {
            return JsonResult.buildErrorStateResult("口令错误", null);
        }

        int sizeBefore = imageCaptchaService.getCaptchaStoreSize();
        imageCaptchaService.garbageCollectCaptchaStore();
        Map<String, Object> state = getCaptchaEngineState(sizeBefore);

        return JsonResult.buildSuccessResult("清理验证码成功", state);
    }

    private Map<String, Object> getCaptchaEngineState(int sizeBefore) {
        Map<String, Object> stateMap = new HashMap<>();
        stateMap.put("最大验证码容量", imageCaptchaService.getCaptchaStoreMaxSize());
        if (sizeBefore > 0) {
            // 执行了垃圾清理, 垃圾清理前验证码数量
            stateMap.put("清理前验证码数量", sizeBefore);
            // 垃圾清理后验证码数量
            stateMap.put("清理后验证码数量", imageCaptchaService.getCaptchaStoreSizeBeforeGarbageCollection());
        } else {
            stateMap.put("目前验证码数量", imageCaptchaService.getCaptchaStoreSize());
            stateMap.put("清理前验证码数量", imageCaptchaService.getCaptchaStoreSizeBeforeGarbageCollection());
        }

        // 验证成功的验证码
        stateMap.put("校验成功验证码数量", imageCaptchaService.getNumberOfCorrectResponses());
        // 验证失败的验证码
        stateMap.put("校验失败验证码数量", imageCaptchaService.getNumberOfUncorrectResponses());
        // 垃圾
        stateMap.put("可清理验证码数量", imageCaptchaService.getNumberOfGarbageCollectableCaptchas());
        stateMap.put("已清理验证码数量", imageCaptchaService.getNumberOfGarbageCollectedCaptcha());


        stateMap.put("Redis验证码数量", imageCaptchaService.getCustomStoreSize());

        return stateMap;
    }


    private void setRestriction(String apiUniqueName, TimeUnit timeUnit, String duration, String limit) {
        fpLockAspect.setLimitation(apiUniqueName + timeUnit.toString(), Integer.valueOf(duration), Integer.valueOf(limit));
    }

    private void parseFpLockAnnotaion(Method method) {
        fpLockAspect.parseFpLockAnnotaion(method);
    }
}
