package cn.quantgroup.xyqb.controller.internal.sms;

import cn.quantgroup.sms.ConfirmableMsg;
import cn.quantgroup.sms.SendAndForgetMsg;
import cn.quantgroup.sms.SmsSender;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.sms.SmsResult;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by FrankChow on 15/7/6.
 */
@RestController
@RequestMapping("/api/sms")
public class SmsController {

    private static final Logger LOGGER = LoggerFactory.getLogger(SmsController.class);
    @Autowired
    private ISmsService smsService;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Value("${sms.is.debug}")
    private boolean smsIsDebug;

    private static final Random random = new Random();

    private static final long EXPIRE_MINUTES = 10;

    /**
     * 短信验证码: for H5
     * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
     */
    @CaptchaValidator
    @RequestMapping("/send_sms_verification_code")
    public JsonResult verifyPhoneNoH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    @CaptchaValidator
    @RequestMapping("/send_reset_code")
    public JsonResult resetPasswordH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    /**
     * 快速登陆发送验证码
     *
     * @param phoneNo
     * @param registerFrom
     * @return
     */
    @CaptchaValidator
    @RequestMapping("/send_login_code")
    public JsonResult sendLoginCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    /**
     * 快速注册发送验证码
     *
     * @param phoneNo
     * @param registerFrom
     * @return
     */
    @CaptchaValidator
    @RequestMapping("/send_regist_code")
    public JsonResult sendRegistCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("快速注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    /**
     * 绑卡成功发送短信
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/send_bind_card_code")
    public JsonResult sendBindCardCode(@RequestParam String phoneNo) {
        LOGGER.info("绑卡成功-发送短信, phoneNo:{}", phoneNo);
        String content = Constants.Sms.BINDCARD_SMS;
        // return sendBindCardSms(phoneNo);
        return sendBindCardSms(phoneNo, content);
    }


    /**
     * 还款成功发送短信
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/send_repay_code")
    public JsonResult sendRepayCode(@RequestParam String phoneNo) {
        LOGGER.info("还款成功-发送短信, phoneNo:{}", phoneNo);
        String content = Constants.Sms.REPAY_SMS;
        // return sendRepaySms(phoneNo);
        return sendBindCardSms(phoneNo, content);
    }

    /**
     * 发送验证码
     *
     * @param phoneNo
     * @return
     */
    private JsonResult sendVerificationCode(String phoneNo) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号必须为11位数字", null);
        }

        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
        if (expire >= EXPIRE_MINUTES - 1) {
            return JsonResult.buildSuccessResult("发送成功", null);
        }

        String randomCode = smsIsDebug ? "000000" : String.valueOf(random.nextInt(899999) + 100000);
        String content = String.format(Constants.Sms.VERIFICATION_CODE, randomCode);
        SmsResult smsResult = smsService.send(phoneNo, content);

        if (smsResult.isRet()) {
            redisTemplate.opsForValue().set(key, randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
            return JsonResult.buildSuccessResult("发送成功", null);
        }
        return JsonResult.buildErrorStateResult("发送失败", null);
    }

    private JsonResult sendVerificationCode2(String phoneNo) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号格式有误", null);
        }

        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
        if (expire >= EXPIRE_MINUTES - 1) {
            return JsonResult.buildSuccessResult("发送成功", null);
        }
        String randomCode = smsIsDebug ? "000000" : String.valueOf(random.nextInt(899999) + 100000);
        String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
        List<String> newList = new ArrayList<>();
        newList.add(randomCode);
        ConfirmableMsg confirmableMsg = new ConfirmableMsg(
                uniqueId, newList, "1", "1", phoneNo
        );
        try {
            smsService.getSmsSender().sendConfirmableMessage(confirmableMsg);
            redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
            return JsonResult.buildSuccessResult("发送成功", uniqueId);
        } catch (Exception e) {
            LOGGER.error("发送短信验证码失败");
            return JsonResult.buildErrorStateResult("发送失败", null);
        }
    }

    private JsonResult sendBindCardSms(String phoneNo, String content) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号必须为11位数字", null);
        }
        SmsResult smsResult = smsService.send(phoneNo, content);

        if (smsResult != null && smsResult.isRet()) {
            return JsonResult.buildSuccessResult("发送成功", null);
        }
        return JsonResult.buildErrorStateResult("发送失败", null);
    }

    /**
     * 发送固定短信文案
     *
     * @param phoneNo
     * @return
     */
    private JsonResult sendBindCardSms(String phoneNo) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号必须为11位数字", null);
        }
        try {
            smsService.getSmsSender().sendAndForget(new SendAndForgetMsg(Collections.EMPTY_LIST, "2", "1", phoneNo));
        } catch (Exception e) {
            LOGGER.error("发送绑卡短信失败");
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    private JsonResult sendRepaySms(String phoneNo) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号必须为11位数字", null);
        }
        try {
            smsService.getSmsSender().sendAndForget(new SendAndForgetMsg(Collections.EMPTY_LIST, "3", "1", phoneNo));
        } catch (Exception e) {
            LOGGER.error("发送还款短信失败");
        }
        return JsonResult.buildSuccessResult(null, null);
    }
}
