package cn.quantgroup.xyqb.controller.external.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UuidPhoneMapping;
import cn.quantgroup.xyqb.event.UserinfoChangedEvent;
import cn.quantgroup.xyqb.exception.IdCardException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.Tuple;
import cn.quantgroup.xyqb.model.jr58.Jr58Authorization;
import cn.quantgroup.xyqb.model.jr58.Jr58RegisterParam;
import cn.quantgroup.xyqb.service.auth.IAuthApiService;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.jr58.Jr58Service;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.user.IUuidPhoneMappingService;
import cn.quantgroup.xyqb.util.GZipUtil;
import cn.quantgroup.xyqb.util.JR58GzipUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Base64Utils;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.WebRequest;
import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static cn.quantgroup.xyqb.Constants.Jr58;

/**
 * @author mengfan.feng
 * @time 2015-09-09 15:32
 */
@RestController
@RequestMapping("/jr58")
public class Jr58Controller implements ApplicationEventPublisherAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(Jr58Controller.class);

    private static final String ENTRY_POINT_URL_TEMPLATE = "%s/app-landing?registerFrom=175&channelId=1&token=%s";
    private static List<String> authorizationTypeList= new ArrayList<>();
    static {
        //授权类型
       authorizationTypeList.add("1");
       authorizationTypeList.add("2");
    }
    @Value("${xyqb.url}")
    private String xyqbUrl;

    @Autowired
    private IAuthApiService authApiService;

    // @Value("${xyqb.entrypoint.jr58}")
    // private String xyqbEntryPointForJr58;

    @Value("${lkb.url}")
    private String lkbUrl;

    @Value("${lkb.entrypoint.jr58}")
    private String lkbEntryPointForJr58;

    @Autowired
    private IUserService xyqbUserService;

    @Autowired
    private IIdCardService idCardService;

    @Autowired
    private Jr58Service jr58Service;

    @Autowired
    private IUuidPhoneMappingService uuidPhoneMappingService;

    private ApplicationEventPublisher applicationEventPublisher;

    @RequestMapping("/register")
    Map<String, Object> register(Jr58RegisterParam param) {
        LOGGER.info("保存58金融用户信息, param:{}", param);

        Map<String, Object> result = new HashMap<>();
        /********************* 校验参数 ************************/
        if (!ValidationUtil.validatePhoneNo(param.getPhone())) {
            result.put("errorcode", Jr58.ERROR_PHONE_NUMBER);
            return result;
        }
        if(StringUtils.isBlank(param.getAccount())){
            LOGGER.error("58金融授权账号-account={}",param.getAccount());
            result.put("errorcode", Jr58.ERROR_ACCOUNT);
            return result;
        }
        if(StringUtils.isBlank(param.getAuthorizationType()) || !authorizationTypeList.contains(param.getAuthorizationType())){
            LOGGER.error("58金融授权类型-authorization={} ",param.getAuthorizationType());
            result.put("errorcode", Jr58.ERROR_AUTH_TYPE);
            return result;
        }

        try {
            param.setIdCardInfo(idCardService.getIdCardInfoWithExceptions(param.getIdcard()));
        } catch (IdCardException | ParseException e) {
            result.put("errorcode", Jr58.ERROR_ID_CARD);
            return result;
        }
        try {
            if(StringUtils.isNotBlank(param.getWbUserName())){
                param.setWbUserName(URLDecoder.decode(param.getWbUserName(), "UTF-8"));
            }
            if(StringUtils.isNotBlank(param.getEmail())){
                param.setEmail(URLDecoder.decode(param.getEmail(), "UTF-8"));
            }
            if(StringUtils.isNotBlank(param.getName())){
                param.setName(URLDecoder.decode(param.getName(), "UTF-8"));
            }
            if(StringUtils.isNotBlank(param.getInfo())){
                param.setInfo(URLDecoder.decode(param.getInfo(), "UTF-8"));
            }

        } catch (Exception e) {
            LOGGER.error("URLDecoder error", e);
        }

        String uuid = jr58Service.register(param);

        applicationEventPublisher.publishEvent(new UserinfoChangedEvent(this, uuid, Constants.Channel.JR58));
        User user = xyqbUserService.findByUuid(uuid);
        LOGGER.info("58金融运营商授权状态更新开始");
        try {
            jr58Service.getRidOfFillingProfile(user, param.getAccount());
        } catch (Exception e) {
            LOGGER.error("58金融运营商授权状态更新失败,异常信息如下所示:error={}", e.getMessage(), e);
        }
        jr58Service.pushAuthorizationStatus(uuid,param.getAccount(),param.getAuthorizationType());
        LOGGER.info("58金融运营商授权状态更新结束");
        result.put("customerid", uuid);
        result.put("success", true);
        result.put("result", 1);
        result.put("errorcode", 0);
        result.put("monitorPhone", param.getMonitorPhone());//58金融用于监控接口是否正常，无其他意义

        LOGGER.info("返还给58用户, result：{}", result);
        return result;
    }

    @RequestMapping("/index")
    public ResponseEntity index(WebRequest request) throws MissingServletRequestParameterException, UnsupportedEncodingException {
        String userId = request.getParameter("userId");
        String from = Strings.nullToEmpty(request.getParameter("from"));

        LOGGER.info("收到来自58金融入口的请求, userId:{}, from:{}", userId, from);

        if (StringUtils.isEmpty(userId)) {
            return new ResponseEntity(HttpStatus.BAD_REQUEST);
        }

        String uuid = userId.toLowerCase();
        Tuple<String, User> entryUrlAndUser = getEntryPointUrl(uuid, from);
        String entryPointUrl = entryUrlAndUser.getKey();
        LOGGER.info("将来自58金融入口的用户导流至 {}", entryPointUrl);
        HttpHeaders responseHeaders = new HttpHeaders();
        responseHeaders.set("Location", entryPointUrl);
        return new ResponseEntity(responseHeaders, HttpStatus.FOUND);
    }

    private Tuple<String, User> getEntryPointUrl(String uuid, String from) {
        // 找到uuid，但是用户不在user表中，说明是老系统用户
        User userInXyqb = xyqbUserService.findByUuid(uuid);

        if (userInXyqb != null) {
            String token = authApiService.login(userInXyqb.getPhoneNo(), Constants.Channel.JR58);
            return new Tuple<>(String.format(ENTRY_POINT_URL_TEMPLATE, xyqbUrl, token), userInXyqb);
        }

        UuidPhoneMapping phoneMapping = uuidPhoneMappingService.findByUuid(uuid);
        String phoneNo = null;
        if(phoneMapping != null){
            phoneNo = phoneMapping.getPhoneNo();
        }else{
            LOGGER.error("58金融运营商授权数据导入接口 - uuid对应的UuidPhoneMapping为空");
        }
        // 手机号在新系统中存在，说明之前已经注册过
        userInXyqb = xyqbUserService.findByPhone(phoneNo);
        if (userInXyqb != null) {
            String token = authApiService.login(phoneNo, Constants.Channel.JR58);
            return new Tuple<>(String.format(ENTRY_POINT_URL_TEMPLATE, xyqbUrl, token), userInXyqb);
        } else {
            // 注册过
            userInXyqb = xyqbUserService.registerAndReturn(phoneNo, "abc1234$", Constants.Channel.JR58);
            String token = authApiService.login(phoneNo, Constants.Channel.JR58);
            return new Tuple<>(String.format(ENTRY_POINT_URL_TEMPLATE, xyqbUrl, token), userInXyqb);
        }
    }


    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    /**
     * 58金融运营商授权数据导入接口
     * @param authorizationData 入参
     * @return JsonResult
     */
    @RequestMapping(value = "/import_authorization_data", method= RequestMethod.POST)
    public JsonResult importAuthorizationData(@RequestBody Jr58Authorization authorizationData) {
        String authorizationDataStr = null;
        LOGGER.info("推送的运营商数据userId={}",authorizationData.getUserId());
        LOGGER.info("推送的运营商数据sourceFrom={}",authorizationData.getSourceFrom());
        if(StringUtils.isEmpty(authorizationData.getAuthorizationData())){
            LOGGER.error("58金融运营商授权数据导入接口 - authorizationData不能为空");
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入接口 - authorizationData不能为空",null);
        }
        try {
            authorizationDataStr = JR58GzipUtil.bytesTogzip(new BASE64Decoder().decodeBuffer(authorizationData.getAuthorizationData()), "utf-8");
            LOGGER.info("推送的运营商数据authorizationDataStr={}",authorizationDataStr);
        } catch (IOException e) {
            LOGGER.error("58金融生成借款申请失败,base64转换失败");
            return JsonResult.buildErrorStateResult("58金融导入运营商授权数据失败",null);
        }
        authorizationData.setOperator_data(new Gson().fromJson(authorizationDataStr, Jr58Authorization.OperatorData.class));
        if(authorizationData == null){
            LOGGER.error("58金融运营商授权数据导入接口 - 入参不完整");
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入 - 入参不完整",null);
        }
        String userId = authorizationData.getUserId();
        String phoneNo= null;
        if(authorizationData.getOperator_data() != null && authorizationData.getOperator_data().getOriginalData() != null){
            phoneNo= authorizationData.getOperator_data().getOriginalData().getPhone();
        }
        if(phoneNo == null){
            LOGGER.info("推送的运营商数据authorizationData={}",authorizationData.getAuthorizationData());
            LOGGER.error("58金融运营商授权数据导入接口 - phoneNo不能为空");
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入接口 - 入参不完整",null);
        }

        LOGGER.info("收到来自58金融运营商授权导入请求, userId:{}", userId);
        if (StringUtils.isEmpty(authorizationData.getSourceFrom())) {
            LOGGER.error("58金融运营商授权数据导入接口 - sourceFrom不能为空");
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入接口 - sourceFrom不能为空",null);
        }
        if (StringUtils.isEmpty(userId)) {
            LOGGER.error("58金融运营商授权数据导入接口 - userId不能为空");
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入接口 - userId不能为空",null);
        }

        if("1".equals(authorizationData.getSourceFrom()) && (null== authorizationData.getOperator_data()
                || null == authorizationData.getOperator_data().getOriginalData())){
            LOGGER.error("58金融运营商授权数据导入接口 - 入参不完整, sourceFrom : {}, authorizationDataStr={}", authorizationData.getSourceFrom(),authorizationDataStr);
            return JsonResult.buildErrorStateResult("58金融运营商授权数据导入接口 - 入参不完整",null);
        }

        String uuid = userId.toLowerCase();
        Tuple<String, User> entryUrlAndUser = getEntryPointUrl(uuid, String.valueOf(Constants.Channel.JR58));
        User user = entryUrlAndUser.getValue();

        // 用户不存在
        if (user  == null) {
            LOGGER.error("58金融运营商授权数据导入接口 - 用户不存在,userId=[{}]", userId);
            return JsonResult.buildErrorStateResult("用户不存在,userId="+userId,null);
        }
        // 推送授权to风控
        jr58Service.pushAuthorizationData(authorizationData);

        LOGGER.info("58金融运营商数据导入结束");

        return JsonResult.buildSuccessResult("success",null);
    }
}
