package cn.quantgroup.xyqb.aspect.token;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.exception.VerificationCodeErrorException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 单次令牌校验切面
 *
 * @author 任文超
 * @version 1.0.0
 * @since 2017-10-31
 */
@Aspect
@Component
public class TokenOnceValidateAdvisor {

    private static final Logger LOGGER = LoggerFactory.getLogger(TokenOnceValidateAdvisor.class);

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    /**
     * 单次令牌校验切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.token.TokenOnceValidator)")
    private void needTokenOnceValidate() {
    }

    /**
     * 在受单次令牌保护的接口方法执行前, 执行单次令牌校验
     *
     * @throws Throwable
     */
    @Around("needTokenOnceValidate()")
    private Object doTokenOnceValidate(ProceedingJoinPoint pjp) throws Throwable {
        boolean checkTokenForPhone = checkTokenForPhone();
        if (!checkTokenForPhone) {
            return JsonResult.buildSuccessResult("Token过期，请重新请求", "", 2L);
        }
        return pjp.proceed();
    }

    /**
     * 校验按手机号分发的单次令牌
     * @return True or False
     */
    private boolean checkTokenForPhone() {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        Map<String, String> phoneTokenMap = getHeaderParam(request);
        if(phoneTokenMap == null || phoneTokenMap.isEmpty()){
            return false;
        }
        // 当前用户手机号
        String phoneNo = phoneTokenMap.get("phoneNo");
        // 当前请求的TokenOnce
        String requestToken = phoneTokenMap.get("requestToken");
        if (StringUtils.isBlank(phoneNo) || StringUtils.isBlank(requestToken)){
            return false;
        }
        final String key = Constants.TOKEN_ONCE_KEY_FOR_PHONE + phoneNo;
        String tokenOnce = redisTemplate.opsForValue().get(key);
        // TokenOnce不应为空值（空白、空格、null）
        if (StringUtils.isBlank(tokenOnce)) {
            redisTemplate.delete(key);
            return false;
        }
        boolean valid = Objects.equals(tokenOnce, requestToken);
        // TokenOnce校验正确时删除key
        if(valid) {
            redisTemplate.delete(key);
        }else {
            LOGGER.info("Token过期，请重新请求, token_once:for_phone:={}, requestToken={}, clientIp={}", phoneNo, requestToken, request.getRemoteAddr());
        }
        return valid;
    }

    /**
     * 单次令牌参数解析
     *
     * @param request 当前请求，其首部行必须包含形如【TokenOnce MTM0NjEwNjc2NjI6NmFjMDY2NWItZTE5Yy00MzkyLWEyNDQtN2I2MTY5MDgzM2Y1】的UTF-8编码的Base64加密参数
     * @return 令牌参数Map 或 null
     */
    private Map<String, String> getHeaderParam(HttpServletRequest request) {
        String headerName = "TokenOnce";
        String credential = request.getHeader(headerName);
        if (StringUtils.isBlank(credential)) {
            LOGGER.info("令牌参数无效, credential:{}", credential);
            return null;
        }
        boolean headerParamValid = true;
        byte[] buf = Base64.getDecoder().decode(credential.getBytes());
        credential = new String(buf, Charset.forName("UTF-8"));
        if(!headerParamValid){
            LOGGER.info("令牌参数无效, credential:{}", credential);
            return null;
        }
        String[] credentialArr = credential.split(":");
        headerParamValid = headerParamValid && credentialArr.length==2;
        if (!headerParamValid) {
            LOGGER.info("令牌参数无效, credential:{}", credential);
            return null;
        }
        // 当前用户手机号
        String phoneNo = credentialArr[0];
        // 当前请求的TokenOnce
        String requestToken = credentialArr[1];
        headerParamValid = headerParamValid && ValidationUtil.validatePhoneNo(phoneNo);
        if (!headerParamValid) {
            LOGGER.info("令牌参数无效, credential:{}", credential);
            return null;
        }
        LOGGER.info("获取单次令牌, phoneNo:{}, requestToken:{}", phoneNo, requestToken);
        Map<String, String> phoneTokenMap = new HashMap<String, String>(2);
        phoneTokenMap.put("phoneNo", phoneNo);
        phoneTokenMap.put("requestToken", requestToken);
        return phoneTokenMap;
    }

}
