package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.user.enums.Relation;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.model.IdCardInfo;
import cn.quantgroup.xyqb.model.IdType;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserRegisterMqMessage;
import cn.quantgroup.xyqb.model.UserStatistics;
import cn.quantgroup.xyqb.repository.IAddressRepository;
import cn.quantgroup.xyqb.repository.IContactRepository;
import cn.quantgroup.xyqb.repository.IUserBtRegisterRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.ILkbUserService;
import cn.quantgroup.xyqb.service.user.IUserBtRegisterService;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.MqUtils;
import cn.quantgroup.xyqb.util.PasswordUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.persistence.criteria.Predicate;
import javax.transaction.Transactional;
import java.sql.Timestamp;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.TimeUnit;

/**
 * Created by Miraculous on 15/7/5.
 */
@Service
@Slf4j
public class UserServiceImpl implements IUserService {

  @Autowired
  RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private ILkbUserService lkbUserService;

  @Autowired
  private IUserRepository userRepository;

  @Autowired
  private ISmsService smsService;

  @Autowired
  private IAddressRepository addressRepository;

  @Autowired
  private IUserBtRegisterService userBtRegisterService;

  @Autowired
  private IContactRepository contactRepository;

  @Autowired
  private IUserBtRegisterRepository userBtRegisterRepository;

  @Autowired
  private IIdCardService idCardService;

  @Autowired
  private IUserDetailService userDetailRepository;


  private Long BAITIAO_CHANNEL = 222L;
  private final static Random random = new Random();

  @Override
  @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
  public User findByPhoneInDb(String phone) {
    return userRepository.findByPhoneNo(phone);
  }

  @Override
  public Map<Long, String> findPhoneByIdsInDb(List<Long> userIds) {
    if (CollectionUtils.isEmpty(userIds)) {
      return Maps.newHashMap();
    }
    Map<Long, String> userIdAndPhoneMap = Maps.newHashMap();
    int pageSize = 1000;
    int idSize = userIds.size();
    for (int i = 0; i < idSize; i += pageSize) {
      List<Long> subList = userIds.subList(i, Math.min(idSize, i + pageSize));
      List<User> users = userRepository.findByIdIn(subList);
      users.stream().forEach(user -> {
        userIdAndPhoneMap.put(user.getId(), user.getPhoneNo());
      });
      if (i + pageSize < idSize) {
        try {
          Thread.sleep(1);
        } catch (InterruptedException e) {
          log.error("线程sleep失败", e);
        }
      }
    }
    return userIdAndPhoneMap;
  }

  @Override
  public User findByUuidInDb(String uuid) {
    return userRepository.findByUuid(uuid);
  }

  @Override
  @CacheEvict(value = "usercache", key = "'xyqbuser' + #user.phoneNo", cacheManager = "cacheManager")
  public User saveUser(User user) {
    return userRepository.save(user);
  }

  @Override
  public User findById(Long userId) {
    return userRepository.findById(userId);
  }

  @Override
  public List<User> findByPhones(List<String> phones) {
    return userRepository.findAll((root, query, cb) -> {
      query.where(root.get("phoneNo").as(String.class).in(phones));
      return query.getRestriction();
    });
  }

  @Override
  public User registerAndReturn(String phoneNo, String password, Long registerFrom, Long btRegisterChannelId) {
    User user = registerAndReturn(phoneNo, password, registerFrom);
    if(null != user &&  null != btRegisterChannelId){
      UserBtRegister userBtRegister = new UserBtRegister();
      userBtRegister.setUserId(user.getId());
      userBtRegister.setRegisterBtMerchantId(btRegisterChannelId);
      userBtRegisterService.save(userBtRegister);
      log.info("白条注册渠道信息保存完成,user:[{}],userBtRegister:[{}]",user,userBtRegister);
    }
    return user;
  }


  @Override
  public boolean register(String phoneNo, String password, Long registerFrom, String ip, Long channelId, Long btRegisterChannelId,String dimension) {
    String uuid = lkbUserService.registerApp(phoneNo, password);
    Timestamp currentTime = new Timestamp(System.currentTimeMillis());

    User user = new User();
    user.setEnable(true);
    user.setPhoneNo(phoneNo);
    user.setUpdatedAt(currentTime);
    user.setCreatedAt(currentTime);
    user.setUuid(uuid);
    user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
    //解决线上白条registerFrom为1的问题
    if (channelId == 222L) {
      user.setRegisteredFrom(channelId);
    } else {
      user.setRegisteredFrom(registerFrom);
    }
    //user.setRegisteredFrom(registerFrom);

    user = userRepository.save(user);
    boolean registerSuccess=false;
    if(null != user){
      UserBtRegister userBtRegister = new UserBtRegister();
      userBtRegister.setUserId(user.getId());
      if(null==btRegisterChannelId){
        userBtRegister.setRegisterBtMerchantId(0L);
      }else{
        userBtRegister.setRegisterBtMerchantId(btRegisterChannelId);
      }

      userBtRegisterService.save(userBtRegister);
      log.info("白条注册渠道信息保存完成");
      registerSuccess=true;
    }

    if(registerFrom!=158412){////网易KA引入注册时不发送短信提醒
      smsService.sendAfterRegister(phoneNo);
    }
    if(registerSuccess){
      MqUtils.sendRegisterMessage(channelId,dimension, user);
    }
    return user != null;
  }


  @Override
  public User registerAndReturn(String phoneNo, String password, Long registerFrom) {
    String uuid = lkbUserService.registerApp(phoneNo, password);
    Timestamp currentTime = new Timestamp(System.currentTimeMillis());

    User user = new User();
    user.setEnable(true);
    user.setPhoneNo(phoneNo);
    user.setUpdatedAt(currentTime);
    user.setCreatedAt(currentTime);
    user.setUuid(uuid);
    user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
    user.setRegisteredFrom(registerFrom);

    return userRepository.save(user);
  }


  @Override
  @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
  public User findByPhoneWithCache(String phone) {
    return userRepository.findByPhoneNo(phone);
  }

  @Override
  @Cacheable(value = "usercache", key = "'xyqbuser' + #uuid", unless = "#result == null", cacheManager = "cacheManager")
  public User findByUuidWithCache(String uuid) {
    return userRepository.findByUuid(uuid);
  }

  @Override
  @Transactional(value = Transactional.TxType.REQUIRED,rollbackOn = Exception.class)
  public boolean register(String phoneNo, String password, Long registerFrom, String userIp, Long channelId) {
    String uuid = lkbUserService.registerApp(phoneNo, password);
    Timestamp currentTime = new Timestamp(System.currentTimeMillis());

    User user = new User();
    user.setEnable(true);
    user.setPhoneNo(phoneNo);
    user.setUpdatedAt(currentTime);
    user.setCreatedAt(currentTime);
    user.setUuid(uuid);
    user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
    //解决线上白条registerFrom为1的问题
    if (channelId == 222L) {
      user.setRegisteredFrom(channelId);
    } else {
      user.setRegisteredFrom(registerFrom);
    }
    //user.setRegisteredFrom(registerFrom);

    user = userRepository.save(user);
    if(registerFrom!=158412){//网易KA引入注册时不发送短信提醒
      smsService.sendAfterRegister(phoneNo);
    }

    return user != null;
  }

  @Override
  public boolean exist(String phoneNo) {
    return userRepository.findByPhoneNo(phoneNo) != null;
  }

  /**
   * 修改用户密码
   *
   * @param phoneNo
   * @param password
   * @return
   * @date 2017-02-15 修改用户修改密码时，更新updatedAt时间
   */
  @Override
  @CacheEvict(value = "usercache", key = "'xyqbuser' + #phone", cacheManager = "cacheManager")
  public boolean resetPassword(String phoneNo, String password) {

    User user = userRepository.findByPhoneNo(phoneNo);
    if (user == null) {
      throw new RuntimeException("用户[" + phoneNo + "]不存在");
    }
    user.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
    user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
    user = userRepository.save(user);
    stringRedisTemplate.expire("usercache:xyqbuser" + phoneNo, 1L, TimeUnit.MILLISECONDS);
    return StringUtils.equals(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT), user.getPassword());

  }

  @Override
  @Transactional(value = Transactional.TxType.REQUIRED,rollbackOn = Exception.class)
  public User registerAndReturn(Long registeredFrom, Long channelId, String phoneNo, String name, String idNo, String provinceCode, String province, String cityCode, String city, String districtCode, String district, String address, String contacts) {
    // 用户信息
    User user = new User();
    user.setPhoneNo(phoneNo);

    Timestamp now = new Timestamp(System.currentTimeMillis());

    user.setCreatedAt(now);
    user.setUpdatedAt(now);
    user.setEnable(true);

    UserBtRegister userBtRegister = null;
    // 白条渠道，需要设置`xyqb_user`.`user_bt_register`
    if (BAITIAO_CHANNEL.equals(channelId)) {
      user.setRegisteredFrom(channelId);
      userBtRegister = new UserBtRegister();

    } else {
      user.setRegisteredFrom(registeredFrom);
    }

    String randomCode = String.valueOf(random.nextInt(899999) + 100000);
    String uuid = lkbUserService.registerApp(phoneNo, randomCode);
    user.setUuid(uuid);
    user.setPassword(PasswordUtil.MD5(randomCode + Constants.PASSWORD_SALT));
    user = userRepository.save(user);
    Long userId = user.getId();

    if (null != userBtRegister){
      userBtRegister.setUserId(userId);
      userBtRegister.setIsActive(true);
      userBtRegister.setRegisterBtMerchantId(registeredFrom);
      userBtRegister.setCreatedAt(now);
      userBtRegister.setUpdatedAt(now);
      userBtRegisterRepository.save(userBtRegister);
    }

    // 地址信息
    Address addressObj = new Address();
    addressObj.setUserId(userId);
    addressObj.setProvinceCode(Long.valueOf(provinceCode));
    addressObj.setProvince(province);
    addressObj.setCityCode(Long.valueOf(cityCode));
    addressObj.setCity(city);
    addressObj.setDistrictCode(Long.valueOf(districtCode));
    addressObj.setDistrict(district);
    addressObj.setAddress(address);
    addressObj.setCreatedAt(now);
    addressObj.setUpdateAt(now);
    addressRepository.save(addressObj);
    try{
      IdCardInfo idCardInfo = idCardService.getIdCardInfo(idNo);
      UserDetail userDetail = new UserDetail();
      userDetail.setPhoneNo(phoneNo);
      userDetail.setIdNo(idNo);
      userDetail.setUserId(userId);
      userDetail.setName(name);
      userDetail.setGender(idCardInfo.getGender());
      userDetail.setIdType(IdType.ID_CARD);
      userDetail.setCreatedAt(now);
      userDetail.setUpdatedAt(now);
      userDetailRepository.saveUserDetail(userDetail);
    }catch (Exception e){
      throw  new RuntimeException("idCardService Exception");
    }


    // 联系人信息
    if (org.apache.commons.lang3.StringUtils.isNotBlank(contacts)) {
      List<Contact> contactList = JSONObject.parseObject(contacts, new TypeReference<List<Contact>>() { });

      if (org.apache.commons.collections.CollectionUtils.isNotEmpty(contactList)) {
        convertContactList(userId, contactList, 2, now);
        contactRepository.save(contactList);
      }
    }

    return user;
  }

  @Override
  public List<User> findRegisterUserByTime(String beginTime, String endTime) {
    return userRepository.findRegisterUserByTime(beginTime,endTime);
  }

  /**
     *
     * @param userId
     * @param contacts
     * @param number 联系人最大数量
     * @param now
     */
  private void convertContactList(Long userId, List<Contact> contacts,int number, Timestamp now) {
    int count = 1;
    for (Contact c : contacts) {

      if (count > number) {
        break;
      }
      c.setId(null);
      c.setUserId(userId);
      c.setRelation(c.getRelation() == null ? Relation.OTHER : c.getRelation());
      c.setCreatedAt(now);
      c.setUpdateAt(now);

      count++;
    }
  }
}
