package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserModel;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.repository.IUserDetailRepository;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.security.Principal;
import java.util.Random;

/**
 * Created by FrankChow on 15/7/5.
 */
@RestController
@RequestMapping("/user")
public class UserController implements IBaseController {


    private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);

    private static final int SMS_VERIFICATION_MAXLEN = 6;

    public final String pwdSalt = "_lkb";

    @Autowired
    private IUserService userService;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> stringRedisTemplate;

    @Autowired
    @Qualifier("redisConnectionFactory")
    private RedisConnectionFactory redisConnectionFactory;

    @Autowired
    @Qualifier("redisTemplate")
    private RedisTemplate redisTemplate;

    @Autowired
    private ISmsService smsService;

    @Autowired
    private ISessionService sessionService;

    @Autowired
    private IUserDetailRepository userDetailRepository;

    private static final char[] PWD_BASE = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
            'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w',
            'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};

    @RequestMapping("/login")
    public JsonResult login2(HttpServletRequest request) {
        String credential = request.getHeader("authorization");
        if(!credential.startsWith("Basic ")){
            return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
        }
        credential = credential.substring("Basic ".length(), credential.length());
        byte[] buf = Base64.decodeBase64(credential);
        String bufStr = "";
        try {
            bufStr = new String(buf, "UTF-8");
        } catch (UnsupportedEncodingException e){
            LOGGER.error("不支持的编码: ", e);
        }
        String[] credentialArr = bufStr.split(":");
        if(credentialArr.length != 2){
            return JsonResult.buildErrorStateResult("用户名或密码不正确.", null);
        }
        String userName = credentialArr[0];
        String pass = credentialArr[1];
        User user = userService.findByPhone(userName);
        if(user == null){
            return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
        }
        //验证密码
        if (!validatePassword(pass, user.getPassword())) {
            return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
        }
        //找到用户
        SessionStruct sessionStruct = sessionService.createSession(user);
        LOGGER.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", getCreatedFrom(), user.getPhoneNo(), getAppChannel());
        AuthBean authBean = new AuthBean();
        authBean.setPhoneNo(user.getPhoneNo());
        authBean.setToken(sessionStruct.getSid());
        return new JsonResult(authBean);
    }

    @RequestMapping("/test")
    public JsonResult test() {
        setChannelId(100L);
        return JsonResult.buildSuccessResult("", getCurrentUser());
    }

    private boolean validatePassword(String paramPass, String targetPassword){
        return StringUtils.defaultString(targetPassword, "").equals(PasswordUtil.MD5(paramPass.toLowerCase() + pwdSalt));
    }

    /**
     * 用户登录
     *
     * @param user
     * @return
     */
//    @RequestMapping("/login")
//    public JsonResult login(Principal user) {
//        if (user == null) {
//            throw new NullUserException();
//        }
//
//        AuthBean bean = new AuthBean(getRequest().getSession().getId(), user);
//
//        LOGGER.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", getCreatedFrom(), user.getName(), getAppChannel());
//
//        return new JsonResult(bean);
//    }


    /**
     * 用户快速登录
     *
     * @return
     */
    @RequestMapping("/login/fast")
    public JsonResult loginFast(Principal user) {
        AuthBean bean = new AuthBean(getRequest().getSession().getId(), user);
        LOGGER.info("用户快速登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", getCreatedFrom(), user.getName(), getAppChannel());
        return new JsonResult(bean);
    }

    /**
     * 用户快速注册
     *
     * @param phoneNo
     * @param verificationCode
     * @param channelId
     * @return
     */
    @RequestMapping("/register/fast")
    public JsonResult registerFast(@RequestParam String phoneNo, @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                                   @RequestParam(required = false) Long registerFrom) {
        String password = genRandomPwd();
        LOGGER.info("用户快速注册, phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},appChannel:{}", phoneNo, verificationCode, channelId, registerFrom, getAppChannel());
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            LOGGER.info("用户快速注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isEmpty(password)) {
            LOGGER.info("用户快速注册失败，密码不能为空, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (password.length() < 6 || password.length() > 20) {
            LOGGER.info("用户快速注册失败，密码长度须在6位至20位之间, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult("密码长度须在6位至20位之间", null);
        }
        if (null == registerFrom) {
            registerFrom = 1L;
        }
        if (userService.exist(phoneNo)) {
            LOGGER.info("用户快速注册失败，该手机号已经被注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
        }
        if (!validSmsVerificationCode(phoneNo, verificationCode)) {
            LOGGER.info("用户快速注册失败，短信验证码错误, registerFrom:{}, phoneNo:{}, verificationCode:{}", registerFrom, phoneNo, verificationCode);
            return JsonResult.buildErrorStateResult("短信验证码错误", null);
        }
        if (!userService.register(phoneNo, password, registerFrom, getIp())) {
            LOGGER.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
        }

        LOGGER.info("用户快速注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);

        return JsonResult.buildSuccessResult(null, null);
    }


    /**
     * 用户登注册
     *
     * @param phoneNo
     * @param password
     * @param verificationCode
     * @param channelId
     * @return
     */
    @RequestMapping("/register")
    public JsonResult register(@RequestParam String phoneNo, @RequestParam String password,
                               @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                               @RequestParam(required = false) Long registerFrom) {
        LOGGER.info("用户注册, phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},appChannel:{}", phoneNo, verificationCode, channelId, registerFrom, getAppChannel());
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            LOGGER.info("用户注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isEmpty(password)) {
            LOGGER.info("用户注册失败，密码不能为空, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (password.length() < 6 || password.length() > 20) {
            LOGGER.info("用户注册失败，密码长度须在6位至20位之间, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult("密码长度须在6位至20位之间", null);
        }
        if (null == registerFrom) {
            registerFrom = 1L;
        }
        if (userService.exist(phoneNo)) {
            LOGGER.info("用户注册失败，该手机号已经被注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
        }
        if (!validSmsVerificationCode(phoneNo, verificationCode)) {
            LOGGER.info("用户注册失败，短信验证码错误, registerFrom:{}, phoneNo:{}, verificationCode:{}", registerFrom, phoneNo, verificationCode);
            return JsonResult.buildErrorStateResult("短信验证码错误", null);
        }
        if (!userService.register(phoneNo, password, registerFrom, getIp())) {
            LOGGER.info("用户注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
        }

        LOGGER.info("用户注册成功, registerFrom:{}, phoneNo:{}, appChannel:{}", registerFrom, phoneNo, getAppChannel());

        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 检查验证码是否正确
     *
     * @param phoneNo
     * @param smsVerificationCode
     * @return
     */
    private boolean validSmsVerificationCode(String phoneNo, String smsVerificationCode) {
        if (StringUtils.isEmpty(smsVerificationCode) || smsVerificationCode.length() != SMS_VERIFICATION_MAXLEN) {
            return false;
        }
        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        String randomCode = stringRedisTemplate.opsForValue().get(key);
        if (!randomCode.contains(":")) {
            return StringUtils.equals(randomCode, smsVerificationCode);
        }
        String[] arr = randomCode.split(":");
        String unqiueId = arr[0];
        String code = arr[1];
        if (StringUtils.equals(code, smsVerificationCode)) {
            try {
                smsService.getSmsSender().confirmSmsResult("1", unqiueId);
            } catch (Exception e) {
                LOGGER.info("短信验证像短信中心确认失效");
            }
            return true;
        }
        return false;
    }

    /**
     * 检查用户是否存在
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/exist")
    public JsonResult exist(@RequestParam String phoneNo) {
        LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);

        if (userService.exist(phoneNo)) {
            LOGGER.info("该手机号已经注册, phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("该手机号已经注册", null);
        }
        return JsonResult.buildSuccessResult(null, null);
    }


    /**
     * 检查用户是否存在
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/exist_check")
    public JsonResult existForResetPwd(@RequestParam String phoneNo) {
        LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
        return JsonResult.buildSuccessResult(null, userService.exist(phoneNo));
    }

    /**
     * 重置密码
     *
     * @param phoneNo
     * @param password
     * @param verificationCode
     * @return
     */
    @RequestMapping("/reset_password")
    public JsonResult resetPassword(@RequestParam String phoneNo,
                                    @RequestParam String password,
                                    @RequestParam(required = false) String registerFrom,
                                    @RequestParam String verificationCode) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isEmpty(password)) {
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (password.length() < 6 || password.length() > 20) {
            return JsonResult.buildErrorStateResult("密码长度须在6位至20位之间", null);
        }
        if (!validSmsVerificationCode(phoneNo, verificationCode)) {
            return JsonResult.buildErrorStateResult("短信验证码错误", null);
        }
        if (!userService.resetPassword(phoneNo, password)) {
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }

        // TODO  加渠道号
        LOGGER.info("修改密码成功, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 检查token是否已经过期不存在了
     *
     * @param token
     * @return
     */
    @RequestMapping("/exists_token")
    public JsonResult checkToken(@RequestParam String token) {
        String tokenKey = Constants.SESSION_PREFIX + token;
        return JsonResult.buildSuccessResult(null, stringRedisTemplate.hasKey(tokenKey));
    }


    private String genRandomPwd() {
        int pwdMax = PWD_BASE.length;
        int i; // 生成的随机数
        int count = 0; // 生成的密码的长度
        StringBuffer pwd = new StringBuffer();
        Random r = new Random();
        while (count < 15) {
            i = Math.abs(r.nextInt(pwdMax)); // 生成的数最大为36-1
            if (i >= 0 && i < PWD_BASE.length) {
                pwd.append(PWD_BASE[i]);
                count++;
            }
        }
        return pwd.toString();
    }


   @RequestMapping("/syncUserInfo")
    public JsonResult syncUserInfo() {
       User user = getCurrentUser();
       if (null == user) {
           return JsonResult.buildErrorStateResult(null, null);
       }
       UserDetail detail = userDetailRepository.findByUserId(user.getId());
       UserModel userModel = new UserModel(user, detail);
       return JsonResult.buildSuccessResult("token校验成功", userModel);
    }

}
