package cn.quantgroup.xyqb.service.v2;

import cn.quantgroup.xyqb.constant.enums.LoginType;
import cn.quantgroup.xyqb.controller.req.v2.LoginReq;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.LoginBean;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.v2.login.BaseLoginParam;
import cn.quantgroup.xyqb.model.v2.login.PhonePasswordLoginParam;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.captcha.IGeetestLogService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.user.ILockIpv4Service;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.BctyptPasswordUtil;
import cn.quantgroup.xyqb.util.PasswordUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 账号密码登录
 */
@Service
public class PhonePasswordLoginStrategy implements LoginStrategy {

    private final IUserRepository userRepository;
    private final ILockIpv4Service lockIpv4Service;
    private final ISessionService sessionService;




    public PhonePasswordLoginStrategy(IUserRepository userRepository, ILockIpv4Service lockIpv4Service, ISessionService sessionService) {
        this.userRepository = userRepository;
        this.lockIpv4Service = lockIpv4Service;
        this.sessionService = sessionService;
    }

    @Override
    public Integer getType() {
        return 3101;
    }

    @Override
    public LoginBean login(BaseLoginParam param) {
        PhonePasswordLoginParam phonePasswordLoginParam = (PhonePasswordLoginParam) param;
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();

        //1、查询库中是否存在用户
        User user = userRepository.findByPhoneNoAndTenantId(phonePasswordLoginParam.getPhone(), sessionStruct.getTenantId());

        if (user == null) {
            // 向该ipv4添加错误计数器
            lockIpv4Service.countErrorByIpv4(sessionStruct.getIp());
            // 向该phoneNo添加错误计数器
            lockIpv4Service.countErrorByPhoneNo(phonePasswordLoginParam.getPhone());

            throw new BizException(BizExceptionEnum.UN_EXIT_USER);
        }

        // 优先校验新密码加密方式 如果有并且密码校验不通过
        if (StringUtils.isNotBlank(user.getCipherPassword())) {
            if (!BctyptPasswordUtil.BCryptCheckPw(phonePasswordLoginParam.getPassword(), user.getCipherPassword())) {
                // 向该ipv4添加错误计数器
                lockIpv4Service.countErrorByIpv4(sessionStruct.getIp());
                // 向该phoneNo添加错误计数器
                lockIpv4Service.countErrorByPhoneNo(phonePasswordLoginParam.getPhone());
                throw new BizException(BizExceptionEnum.ERROR_PASSWORD);
            }
        } else {
            //验证密码
            if (!PasswordUtil.validatePassword(phonePasswordLoginParam.getPassword(), user.getPassword(),user.getPasswordType())) {
                // 向该ipv4添加错误计数器
                lockIpv4Service.countErrorByIpv4(sessionStruct.getIp());
                // 向该phoneNo添加错误计数器
                lockIpv4Service.countErrorByPhoneNo(phonePasswordLoginParam.getPhone());
                throw new BizException(BizExceptionEnum.ERROR_PASSWORD);
            }
            // 校验老密码正确更新新加密方式
            user.setCipherPassword(BctyptPasswordUtil.BCryptWithSalt(phonePasswordLoginParam.getPassword()));
            userRepository.save(user);
        }
        // 向该ipv4添加成功计数器
        lockIpv4Service.countSuccessByIpv4(sessionStruct.getIp());
        //尝试解锁
        lockIpv4Service.unLockPhone(phonePasswordLoginParam.getPhone());

        LoginProperties loginProperties = new LoginProperties(1, sessionStruct.getRegisteredFrom(),sessionStruct.getTenantId());
        AuthBean authBean = sessionService.createSession(user, loginProperties, LoginType.ACCOUNTPASSWORD.ordinal(),sessionStruct.getTenantId(),false);
        LoginBean loginBean = new LoginBean();
        if (authBean != null) {
            authBean.setRegister(false);
            loginBean.setToken(authBean.getToken());
            loginBean.setPhoneNo(authBean.getPhoneNo());
            loginBean.setUuid(authBean.getUuid());
            loginBean.setHasPassword(authBean.isHasPassword());
            loginBean.setRegister(authBean.getRegister());
            loginBean.setUserId(user.getId());
            loginBean.setTenantId(user.getTenantId());
        }
        return loginBean;
    }

    @Override
    public BaseLoginParam checkParam(LoginReq loginReq) {
        PhonePasswordLoginParam param = loginReq.getData().toJavaObject(PhonePasswordLoginParam.class);
        validator.validate(param);
        return param;
    }

}
