package cn.quantgroup.xyqb.controller.external;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.accessable.IpValidator;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaFiniteValidator;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaNewValidator;
import cn.quantgroup.xyqb.aspect.forbidden.AccessForbiddenValidator;
import cn.quantgroup.xyqb.aspect.limit.PasswordFreeAccessValidator;
import cn.quantgroup.xyqb.aspect.lock.PasswordErrorFiniteValidator;
import cn.quantgroup.xyqb.config.data.WechatConfiguration;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.constant.enums.LoginType;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.LoginRecord;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.exception.VerificationCodeErrorException;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.session.SessionValue;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.service.captcha.IGeetestLogService;
import cn.quantgroup.xyqb.service.http.IHttpService;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.ILockIpv4Service;
import cn.quantgroup.xyqb.service.user.ILoginRecordService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.*;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.*;

import static cn.quantgroup.xyqb.Constants.VERIFICATION_CODE_FINITE_COUNT_NEW;
import static cn.quantgroup.xyqb.constant.UserConstant.USER_ERROR_OR_ENABLE_ERROR;

/**
 * Http服务接口：用户注册、登录、重置密码
 * Created by FrankChow on 15/7/5.
 */
@Slf4j
@RestController
@RequestMapping("/user")
public class UserController implements IBaseController {

    @Autowired
    private IUserService userService;
    @Autowired
    private ILockIpv4Service lockIpv4Service;
    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> stringRedisTemplate;
    @Autowired
    private ISmsService smsService;

    @Autowired
    private ISessionService sessionService;

    @Autowired
    private IUserInfoRepository userInfoRepository;
    @Autowired
    private IMerchantService merchantService;
    @Autowired
    private IWechatService wechatService;
    @Autowired
    private IUserRegisterService userRegisterService;

    @Value("${xyqb.user.service.host}")
    private String userHost;

    @Value("${token.prefix}")
    private String prefix;

    @Autowired
    private IHttpService httpService;

    @Autowired
    private ILoginRecordService loginRecordService;

    @Autowired
    private IGeetestLogService geetestLogService;

    @Autowired
    private WechatConfiguration wechatConfiguration;


    /**
     * 登录（账号 + 密码）
     * 密码错误达到限定次数时执行图形验证码校验
     * 图形验证码累计错误达到限定次数时须重新获取
     *
     * @param channelId
     * @param appChannel
     * @param createdFrom
     * @param userId
     * @param key
     * @param request
     * @param dimension
     * @return
     * @yapi unknown
     */
    @CaptchaFiniteValidator
    @RequestMapping("/loginV1")
    public JsonResult loginV1(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "") String userId,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) String dimension,
            @RequestHeader(value = Constants.X_AUTH_APP_ID, defaultValue = UserConstant.defaultAppId) String appId,
            @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
            ,
            HttpServletRequest request) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }
        log.info("loginV1 -> channelId:{},appChennel:{},createdFrom:{},userId:{},key:{},dimension:{}", channelId, appChannel, createdFrom, userId, key, dimension);
        return login(channelId, appChannel, createdFrom, userId, key, dimension, null, request, appId, tenantId);
    }


    /**
     * 登录（账号 + 密码）
     * 密码错误达到限定次数时执行图形验证码校验
     * 图形验证码累计错误达到限定次数时须重新获取
     * 加入极验验证码并落库
     *
     * @param channelId
     * @param appChannel
     * @param createdFrom
     * @param userId
     * @param key
     * @param request
     * @param dimension
     * @return
     * @yapi unknown
     */
    @CaptchaNewValidator
    @RequestMapping("/loginV2")
    public JsonResult loginV2(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "") String userId,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) String dimension,
            @RequestParam Long geetestlog_id,
            HttpServletRequest request,
            @RequestParam(value = Constants.X_AUTH_APP_ID, defaultValue = UserConstant.defaultAppId) String appId,
            @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }

        log.info("loginV2 -> channelId:{},appChennel:{},createdFrom:{},userId:{},key:{},dimension:{}", channelId, appChannel, createdFrom, userId, key, dimension);
        return login(channelId, appChannel, createdFrom, userId, key, dimension, geetestlog_id, request, appId, tenantId);
    }


    /**
     * 慢速登陆
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/7310
     */
    @PasswordErrorFiniteValidator
    @RequestMapping("/login")
    public JsonResult login(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "") String userId,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) String dimension,
            @RequestParam(required = false) Long geetestLogId,
            HttpServletRequest request,
            @RequestHeader(value = Constants.X_AUTH_APP_ID, defaultValue = UserConstant.defaultAppId) String appId,
            @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }
        log.info("login -> channelId:{},appChannel:{},createdFrom:{},userId:{},key:{},dimension:{}", channelId, appChannel, createdFrom, userId, key, dimension);
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("未知的连接", null);
        }
        if (StringUtils.length(userId) > Constants.UUID_MIN_LENGTH) {
            return loginWithUserId(channelId, appChannel, createdFrom, userId, merchant, geetestLogId, request, appId, tenantId);
        } else {
            return loginWithHttpBasic(channelId, appChannel, createdFrom, merchant, dimension, geetestLogId, request, appId, tenantId);
        }
    }

    /**
     * 快速登录（手机号 + 短信验证码），H5专用入口
     * 短信验证码错误达到限定次数时执行图形验证码校验
     * 图形验证码累计错误达到限定次数时须重新获取
     *
     * @param channelId
     * @param appChannel
     * @param createdFrom
     * @param key
     * @param btRegisterChannelId
     * @param dimension
     * @param request
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/1786
     */
    @RequestMapping("/login/fastV1")
    public JsonResult loginFastV1(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false) String dimension,
            @RequestParam(name = "click_id", required = false) String clickId,
            @RequestParam(required = false) Integer tenantId,
            @RequestHeader(value = Constants.X_AUTH_APP_ID, defaultValue = UserConstant.defaultAppId) String appId,
            HttpServletRequest request) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }
        log.info("login/fastV1 -> channelId:{},ZappChennel:{},createdFrom:{},key:{},btRegisterChannelId:{},dimension:{},clickId:{}", channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, clickId);
        return loginFast(channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, clickId, tenantId, null, appId, request);
    }

    /**
     * 快速登陆 - 同h5login代码实现一致
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/2066
     */
    @RequestMapping("/login/fast")
    public JsonResult loginFast(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false) String dimension,
            @RequestParam(name = "click_id", required = false) String clickId,
            @RequestParam(required = false) Integer tenantId,
            @RequestParam(required = false) Long geetestLogId,
            @RequestHeader(value = Constants.X_AUTH_APP_ID, defaultValue = UserConstant.defaultAppId) String appId,
            HttpServletRequest request) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }
        Map<String, JsonResult> validMap = getHeaderParam(request);
        log.info("login/fast -> channelId:{},appChannel:{},createdFrom:{},btRegisterChannelId:{},key:{},dimension:{},clickId:{}", channelId, appChannel, createdFrom, btRegisterChannelId, key, dimension, clickId);
        JsonResult failResult = validMap.get(Constants.CHECK_FAIL);
        if (null != failResult) {
            return failResult;
        }
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("未知的连接", null);
        }
        JsonResult successResult = validMap.get("success");
        String phoneNo = successResult.getData().toString();
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            log.info("用户快速登录失败，手机号错误, createdFrom:{},phoneNo:{}", createdFrom, phoneNo);
            throw new UserNotExistException("手机号错误");
        }
        String verificationCode = successResult.getMsg();
        // 执行短信验证码检查
        verifyPhoneAndCodeForOnce(phoneNo, verificationCode);
        //默认羊小咩租户
        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        return userService.loginFast(channelId, appChannel, createdFrom, btRegisterChannelId, dimension, clickId, request, merchant, phoneNo, tenantId, geetestLogId, appId);
    }

    /**
     * 快速登录验证
     *
     * @param request
     * @return
     */
    private Map<String, JsonResult> getHeaderParam(HttpServletRequest request) {
        Map<String, JsonResult> result = new HashMap<>();
        String verificationHeader = "Verification ";
        String credential = request.getHeader("authorization");
        if (StringUtils.isBlank(credential)) {
            result.put(Constants.CHECK_FAIL, JsonResult.buildErrorStateResult("登录失败", null));
            return result;
        }
        if (!credential.startsWith(verificationHeader)) {
            result.put(Constants.CHECK_FAIL, JsonResult.buildErrorStateResult("登录失败", null));
            return result;
        }
        credential = credential.substring(verificationHeader.length(), credential.length());
        byte[] buf = Base64.decodeBase64(credential);
        credential = new String(buf, Charset.forName("UTF-8"));
        String[] credentialArr = credential.split(Constants.SPLIT_CHAR);
        if (credentialArr.length != Constants.VERIFICATION_LEN) {
            result.put(Constants.CHECK_FAIL, JsonResult.buildErrorStateResult("登录失败", null));
            return result;
        }
        String phoneNo = credentialArr[0];
        String verificationCode = credentialArr[1];
        log.info("用户快速登录,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
        if (!ValidationUtil.validatePhoneNo(phoneNo) || StringUtils.isBlank(verificationCode)) {
            result.put(Constants.CHECK_FAIL, JsonResult.buildErrorStateResult("登录失败", null));
            return result;
        }
        result.put("success", JsonResult.buildSuccessResult(verificationCode, phoneNo));
        return result;
    }

    /**
     * 用户注册
     *
     * @param phoneNo
     * @param password
     * @param verificationCode
     * @param channelId
     * @return
     * @yapi unknown
     */
    @RequestMapping("/register")
    public JsonResult register(@RequestParam String phoneNo, @RequestParam String password,
                               @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                               @RequestParam(required = false) Long registerFrom,
                               @RequestParam(required = false) Long btRegisterChannelId, @RequestParam(required = false) String dimension,
                               @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        log.info("用户注册：register -> phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},btRegisterChannelId:{},dimension:{}", phoneNo, verificationCode, channelId, registerFrom, btRegisterChannelId, dimension);
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            log.info("用户注册失败，手机号错误：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isEmpty(password)) {
            log.info("用户注册失败，密码不能为空：register -> registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (!ValidationUtil.validatePassword(password)) {
            log.info("用户注册失败，{}：register -> registerFrom:{}, phoneNo:{}, password:{}", PasswordUtil.TOAST_MSG, registerFrom, phoneNo, password);
            return JsonResult.buildErrorStateResult(PasswordUtil.TOAST_MSG, null);
        }
        if (null == registerFrom) {
            registerFrom = 1L;
        }
        verifyPhoneAndCode(phoneNo, verificationCode);
        User user = userService.findByPhoneInDb(phoneNo, tenantId);
        if (user != null) {
            // 用户注册使用新加密方式
//            user.setPassword(PasswordUtil.MD5WithSalt(password));
            user.setCipherPassword(BctyptPasswordUtil.BCryptWithSalt(password));
            userService.saveUser(user);
            log.info("用户注册失败，该手机号已经被注册：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            //已存在的用户, 经过短信认证, 也认为是注册成功的
            return JsonResult.buildSuccessResult(null, null);
        }
        if (!userRegisterService.register(phoneNo, password, registerFrom, getIp(), channelId, btRegisterChannelId, dimension)) {
            log.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
        }
        log.info("用户注册成功：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 检查用户是否存在
     *
     * @param phoneNo 手机号
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/2388
     * @Deprecated 20210318
     */
    @IpValidator
    @RequestMapping("/exist")
    @Deprecated
    public JsonResult exist(@RequestParam String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        log.info("检查用户是否存在, phoneNo:{}", phoneNo);
        if (userService.exist(phoneNo, tenantId)) {
            log.info("该手机号已经注册, phoneNo:{}，remoteIp:{}", phoneNo, getIp());
            return JsonResult.buildErrorStateResult("该手机号已经注册", null);
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 检查用户是否存在
     *
     * @param phoneNo 手机号
     * @return
     * @yapi unknown
     * @Deprecated 20210318
     */
    @IpValidator
    @RequestMapping("/exist_check")
    @Deprecated
    public JsonResult existForResetPwd(@RequestParam String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        log.info("检查用户是否存在, phoneNo:{}，remoteIp:{}", phoneNo, getIp());
        return JsonResult.buildSuccessResult(null, userService.exist(phoneNo, tenantId));
    }

    /**
     * 重置密码
     *
     * @param phoneNo
     * @param password
     * @param verificationCode
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/3263
     */
    @RequestMapping("/reset_password")
    public JsonResult resetPassword(@RequestParam String phoneNo,
                                    @RequestParam String password,
                                    @RequestParam(required = false) String registerFrom,
                                    @RequestParam String verificationCode, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isBlank(password)) {
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (!ValidationUtil.validatePassword(password)) {
            return JsonResult.buildErrorStateResult(PasswordUtil.TOAST_MSG, null);
        }
        verifyPhoneAndCode(phoneNo, verificationCode);
        if (!userService.exist(phoneNo, tenantId)) {
            log.info("修改密码失败，该手机号尚未注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }
        if (!userService.resetPassword(phoneNo, password, tenantId)) {
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }

        // TODO  加渠道号
        log.info("修改密码成功, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        //修改密码成功也要清除一下
        lockIpv4Service.unLockPhone(phoneNo);
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 重置密码
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/3263
     * @Deprecated 20210318, 仅有一次调用2021-03-15 00:38:57.752
     */
    @Deprecated
    @PasswordFreeAccessValidator
    @RequestMapping(path = "/resetPassword", method = RequestMethod.POST)
    public JsonResult resetPassword(@RequestParam String phoneNo, @RequestParam String password, @RequestParam(required = false) String passwordNew, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号错误", null);
        }
        if (StringUtils.isBlank(passwordNew)) {
            return JsonResult.buildErrorStateResult("密码不能为空", null);
        }
        if (!ValidationUtil.validatePassword(password)) {
            return JsonResult.buildErrorStateResult(PasswordUtil.TOAST_MSG, null);
        }
        User user = userService.findByPhoneWithCache(phoneNo, tenantId);
        if (Objects.isNull(user)) {
            log.info("修改密码失败，该手机号尚未注册, phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }
        if (!user.getEnable()) {
            log.info("修改密码失败，该用户已禁用, phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }
        // 验证密码：原密码不存在时，必须为空
        if (StringUtils.isBlank(user.getPassword()) ^ StringUtils.isBlank(password)) {
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }
        // 优先校验新的密码
        if (StringUtils.isNotBlank(user.getCipherPassword())) {
            if (!BctyptPasswordUtil.BCryptCheckPw(password, user.getCipherPassword())) {
                return JsonResult.buildErrorStateResult("修改密码失败", null);
            }
        } else {
            if (StringUtils.isNotBlank(user.getPassword()) && !PasswordUtil.validatePassword(password, user.getPassword(), user.getPasswordType())) {
                return JsonResult.buildErrorStateResult("修改密码失败", null);
            }
        }

        if (!userService.resetPassword(phoneNo, passwordNew, tenantId)) {
            return JsonResult.buildErrorStateResult("修改密码失败", null);
        }
        return JsonResult.buildSuccessResult("修改密码成功");
    }


    /**
     * 检查token是否已经过期不存在了
     *
     * @param token - sid,session的id
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/4012
     */
    @IpValidator
    @RequestMapping("/exists_token")
    public JsonResult checkToken(@RequestParam String token, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(token)) {
            return JsonResult.buildSuccessResult(null, false);
        }
        if (token.contains(Constants.TOKEN_MASTER)) {
            return JsonResult.buildSuccessResult(null, false);
        }
        String tokenKey = Constants.SESSION_PREFIX + token;

        String tokenKey2;
        if (tenantId == null || UserConstant.defaultTenantId.equals(tenantId)) {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
        } else {
            tokenKey2 = Constants.Session.USER_SESSION_CACHE + tenantId + ":" + token;
        }
        // 判断token是否存在
        boolean exist = stringRedisTemplate.hasKey(tokenKey) || stringRedisTemplate.hasKey(tokenKey2);
        if (!token.contains(prefix) && !exist) {
            findTokenExchange(token, tenantId);
            exist = stringRedisTemplate.hasKey(tokenKey) || stringRedisTemplate.hasKey(tokenKey2);
        }
        return JsonResult.buildSuccessResult("token valid", exist);
    }

    /**
     * 获取 token 信息
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/9191
     */
    @RequestMapping("/token")
    public JsonResult token(@RequestParam String token, @RequestParam(required = false) Integer tenantId, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantIdHeader) {

        if (tenantId == null) {
            tenantId = tenantIdHeader;
        }
        Long registeredFrom = getRegisteredFrom();
        Map<String, Object> result = new HashMap<>();
        result.put("exist", false);
        if (StringUtils.isEmpty(token)) {
            return JsonResult.buildSuccessResult(null, result);
        }
        if (token.contains(Constants.TOKEN_MASTER)) {
            return JsonResult.buildSuccessResult(null, result);
        }
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token, tenantId,registeredFrom);
        if (sessionStruct == null || sessionStruct.getValues() == null) {
            // 使用token去电商查询信息, 如果token 不包含本系统token前缀
            if (!token.contains(prefix)) {
                findTokenExchange(token, tenantId);
                sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token, tenantId,registeredFrom);
                if (sessionStruct == null || sessionStruct.getValues() == null) {
                    return JsonResult.buildSuccessResult(null, result);
                }
            } else {
                return JsonResult.buildSuccessResult(null, result);
            }
        }
        if (sessionStruct.getValues() == null || sessionStruct.getValues().getUser() == null ) {
            log.info("当前token对应的用户不存在，入参tenantId:{},token:{}", tenantId, token);
            return JsonResult.buildSuccessResult(null, result);
        }
        User user = sessionStruct.getValues().getUser();
        if (user.getTenantId() == null || !user.getEnable()) {
            log.info("当前token对应的用户非当前租户，userId:{},用户tenantId：{},入参tenantId:{},token:{}", user.getId(), user.getTenantId(), tenantId, token);
            return JsonResult.buildSuccessResult(null, result);
        }



        String phoneNo = user.getPhoneNo();
        result.put("phoneNo", phoneNo);
        result.put("userId", user.getId());
        result.put("exist", true);
        result.put("uuid", user.getUuid());
        return JsonResult.buildSuccessResult(null, result);
    }


    /**
     * token 交换
     */
    @RequestMapping("/tokenExchange")
    public JsonResult tokenExchange(@RequestParam String token, @RequestParam(required = false) Integer tenantId, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantIdHeader) {

        if (tenantId == null) {
            tenantId = tenantIdHeader;
        }
        Long registeredFrom = getRegisteredFrom();
        TokenExchange tokenExchange = new TokenExchange();
        if (StringUtils.isEmpty(token)) {
            return JsonResult.buildSuccessResult(null, tokenExchange);
        }
        if (token.contains(Constants.TOKEN_MASTER)) {
            return JsonResult.buildSuccessResult(null, tokenExchange);
        }
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token, tenantId,registeredFrom);
        if (sessionStruct == null || sessionStruct.getValues() == null) {
            return JsonResult.buildSuccessResult(null, tokenExchange);
        }
        User user = sessionStruct.getValues().getUser();
        if (user.getTenantId() == null || !user.getEnable()) {
            log.info("当前token对应的用户非当前租户，userId:{},用户tenantId：{},入参tenantId:{},token:{}", user.getId(), user.getTenantId(), tenantId, token);
            return JsonResult.buildSuccessResult(null, tokenExchange);
        }

        tokenExchange.setLoginProperties(JSONObject.toJSONString(sessionStruct.getValues().getLoginProperties()));
        tokenExchange.setUserId(user.getId());
        tokenExchange.setPhoneNo(user.getPhoneNo());
        tokenExchange.setUuid(user.getUuid());
        tokenExchange.setExpire(sessionStruct.getExpire());
        return JsonResult.buildSuccessResult(null, tokenExchange);

    }

    /**
     * 交换token
     *
     * @param token
     * @return
     */
    private void findTokenExchange(String token, Integer tenantId) {
        if (tenantId == null) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        // 请求其他系统信息
        HashMap<String, String> parameters = new HashMap<>();
        HashMap<String, String> headers = new HashMap<>();
        headers.put("qg-tenant-id", tenantId.toString());
        parameters.put("token", token);

        try {
            String resultStr = httpService.post(userHost + "/api/finance-gateway/finance-user/finance/tokenExchange", headers, parameters);
            JsonResult<TokenExchange> tokenExchangeResult = JSONObject.parseObject(resultStr, new TypeReference<JsonResult<TokenExchange>>() {
            });
            // 如果有返回值
            TokenExchange tokenExchange = tokenExchangeResult.getData();
            if (tokenExchange != null) {
                SessionValue sessionValue = new SessionValue();
                User user = userService.findById(tokenExchange.getUserId(), tenantId);
                if (user != null) {
                    LoginProperties loginProperties = JSONObject.parseObject(tokenExchange.getLoginProperties(), LoginProperties.class);

                    // 根据返回值生成token, 返回值包含user:session:token的值
                    sessionValue.setUser(user);
                    sessionValue.setLoginProperties(loginProperties);
                    sessionService.persistSessionExchange(token, sessionValue, tokenExchange.getExpire(), tenantId);
                }

            }
        } catch (Exception e) {
            log.error("请求金融卡包网关出错--", e);
        }
    }

    /**
     * 用户中心首页，显示用户头像、昵称、姓名
     *
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/267
     */
    @AccessForbiddenValidator
    @PasswordFreeAccessValidator
    @RequestMapping("/center/index")
    public JsonResult userCenterIndex(@RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {

        UserBrief brief = new UserBrief();
        //TODO:he 返回""，待解决
        Long userId = getCurrentUserFromRedis().getId();
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        if (userDetail != null) {
            brief.setName(userDetail.getName());
            brief.setAvatar(userDetail.getPhoto());
            brief.setNick(userDetail.getNickName());
            brief.setSex(Optional.ofNullable(userDetail.getGender()).orElse(cn.quantgroup.xyqb.model.Gender.UNKNOWN).ordinal() + "");
            brief.setPhoneNo(userDetail.getPhoneNo().substring(0, 3) + "****" + userDetail.getPhoneNo().substring(7, 11));
        } else {
            return JsonResult.buildSuccessResult("", brief);
        }

        return JsonResult.buildSuccessResult(null, brief);
    }

    /**
     * @yapi unknown
     * @Deprecated 20210318, 20210201-20210203有集中调用274次，其他时间没有
     */
    @Deprecated
    @RequestMapping("/syncUserInfo")
    public JsonResult syncUserInfo() {
        User user = getCurrentUserFromRedis();
        if (null == user) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        UserInfoEntity detail = userInfoRepository.findByUserIdAndTenantId(user.getId(), user.getTenantId());
        UserModel userModel = new UserModel(user, detail);
        return JsonResult.buildSuccessResult("token校验成功", userModel);
    }

    /**
     * 登出接口
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/23661
     */
    @RequestMapping(value = "/logout", method = RequestMethod.GET)
    public JsonResult logout(HttpServletRequest request, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {

        String token = request.getHeader("x-auth-token");

        if (org.apache.commons.lang3.StringUtils.isEmpty(token)) {
            return JsonResult.buildErrorStateResult("服务器异常,请稍后再试", null);
        }

        try {
            userService.logout(token, tenantId);
        } catch (Exception e) {
            log.error("登出接口系统异常token:{}", token);
            return JsonResult.buildErrorStateResult("服务器异常,请稍后再试", null);
        }

        return JsonResult.buildSuccessResult("登出成功");
    }

    private JsonResult loginWithHttpBasic(Long channelId, String appChannel, Long createdFrom, Merchant merchant, String dimension, Long geetestLogId, HttpServletRequest request, String appId, Integer tenantId) {
        User user = verificateUserNameAndPassword(request, tenantId);
        if (user == null) {
            return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
        } else if (!user.getEnable()) {
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_ENABLE_ERROR, null);
        } else if (!wechatRelateUserIfNecessary(user, request, appId, tenantId)) {
            return JsonResult.buildErrorStateResult("登录时微信关联失败", null);
        }
        LoginProperties loginProperties = new LoginProperties("", 1, channelId, createdFrom, appChannel, merchant.getId(), merchant.getName(), tenantId);
        AuthBean authBean = sessionService.createSession(user, loginProperties, LoginType.ACCOUNTPASSWORD.ordinal(), tenantId);
        if (authBean != null) {
            authBean.setRegister(false);
        }

        if (geetestLogId != null) {
            geetestLogService.updateByUidGeetestLog(geetestLogId, user.getId());
        }
        return new JsonResult(authBean);
    }

    private User verificateUserNameAndPassword(HttpServletRequest request, Integer tenantId
    ) {
        String credential = request.getHeader("authorization");
        if (StringUtils.isBlank(credential) || !credential.startsWith(Constants.PASSWORD_HEADER)) {
            return null;
        }
        credential = credential.substring(Constants.PASSWORD_HEADER.length());
        byte[] buf = Base64.decodeBase64(credential);
        String bufStr = "";
        try {
            bufStr = new String(buf, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("不支持的编码: ", e);
        }
        String clientIp = getIp();
        String[] credentialArr = bufStr.split(":");
        if (credentialArr.length != Constants.AUTHORIZE_HEADER_SIZE) {
            log.info("用户登录失败：{}", bufStr);
            // 向该ipv4添加错误计数器
            lockIpv4Service.countErrorByIpv4(clientIp);
            return null;
        }
        String phoneNo = credentialArr[0];
        String pass = credentialArr[1];
        User user = userService.findByPhoneWithCache(phoneNo, tenantId);
        if (user == null) {
            // 向该ipv4添加错误计数器
            lockIpv4Service.countErrorByIpv4(clientIp);
            // 向该phoneNo添加错误计数器
            lockIpv4Service.countErrorByPhoneNo(phoneNo);
            return null;
        }

        // 优先校验新密码加密方式 如果有并且密码校验不通过
        if (StringUtils.isNotBlank(user.getCipherPassword())) {
            if (!BctyptPasswordUtil.BCryptCheckPw(pass, user.getCipherPassword())) {
                // 向该ipv4添加错误计数器
                lockIpv4Service.countErrorByIpv4(clientIp);
                // 向该phoneNo添加错误计数器
                lockIpv4Service.countErrorByPhoneNo(phoneNo);
                return null;
            }
        } else {
            //验证密码
            if (!PasswordUtil.validatePassword(pass, user.getPassword(), user.getPasswordType())) {
                // 向该ipv4添加错误计数器
                lockIpv4Service.countErrorByIpv4(clientIp);
                // 向该phoneNo添加错误计数器
                lockIpv4Service.countErrorByPhoneNo(phoneNo);
                return null;
            }
            // 校验老密码正确更新新加密方式
            user.setCipherPassword(BctyptPasswordUtil.BCryptWithSalt(pass));
            userService.saveUser(user);
        }

        // 向该ipv4添加成功计数器
        lockIpv4Service.countSuccessByIpv4(clientIp);
        //尝试解锁
        lockIpv4Service.unLockPhone(phoneNo);

        return user;
    }

    private JsonResult loginWithUserId(Long channelId, String appChannel, Long createdFrom, String userId, Merchant merchant, Long geetestLogId, HttpServletRequest request, String appId, Integer tenantId
    ) {
        //查询用户
        User user = userService.findByUuidInDb(userId, tenantId);
        if (Objects.isNull(user) || !user.getEnable()) {
            log.error("用户不存在，或者已经注销，userId:{}", userId);
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_ENABLE_ERROR, null);
        } else if (!wechatRelateUserIfNecessary(user, request, appId, tenantId)) {
            return JsonResult.buildErrorStateResult("登录时微信关联失败", null);
        }
        LoginProperties loginProperties = new LoginProperties("", 4, channelId, createdFrom, appChannel, merchant.getId(), merchant.getName(), tenantId);
        //尝试解锁
        lockIpv4Service.unLockPhone(user.getEncryptedPhoneNo());

        // 更新极验用户userID
        if (geetestLogId != null) {
            geetestLogService.updateByUidGeetestLog(geetestLogId, user.getId());
        }

        //更新session
        return new JsonResult(sessionService.createSession(user, loginProperties, LoginType.ACCOUNTPASSWORD.ordinal(), tenantId));
    }

    /**
     * 如果必要的话，关联用户和微信
     *
     * @param user    - 用户标识
     * @param request - 当前请求
     * @return true - 继续登录，false - 微信关联失败，重新登录
     */
    private boolean wechatRelateUserIfNecessary(User user, HttpServletRequest request, String appId, Integer tenantId) {
        Objects.requireNonNull(request, "无效请求");
        String clientIp = IpUtil.getRemoteIP(request);
        Set<String> paramKeys = request.getParameterMap().keySet();
        boolean ready = paramKeys.contains(Constants.WECHAT_OPEN_ID);
        if (!ready) {
            return true;
        } else if (Objects.isNull(user) || Objects.isNull(user.getId()) || StringUtils.isBlank(request.getParameter(Constants.WECHAT_OPEN_ID))) {
            log.warn("微信关联失败，user:{}, request-Header:{}", user, JSON.toJSONString(getRequestHeaderMap(request)));
            return false;
        }
        Long userId = user.getId();
        String phoneNo = user.getEncryptedPhoneNo();
        try {
            int rows = wechatService.relateUser(userId, phoneNo, request.getParameter(Constants.WECHAT_OPEN_ID), appId, tenantId);
            return rows > 0;
        } catch (Exception e) {
            log.error("微信关联失败，user:{}, request-Header:{}", user, JSON.toJSONString(getRequestHeaderMap(request)), e);
        }
        return false;
    }

    /**
     * 校验短信验证码
     *
     * @param phoneNo
     * @param verificationCode
     */
    private void verifyPhoneAndCode(String phoneNo, String verificationCode) {
        if (!smsService.verifyPhoneAndCode(phoneNo, verificationCode)) {
            // 是否需要重新发送短信验证码
            if (smsService.needResendCode(phoneNo)) {
                throw new VerificationCodeErrorException("验证码失效，请重新获取");
            }
            log.info("验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
            throw new VerificationCodeErrorException("短信验证码错误");
        }
    }

    /**
     * 校验验证码 不论成功与否都删除
     *
     * @param phoneNo
     * @param verificationCode
     */
    private void verifyPhoneAndCodeForOnce(String phoneNo, String verificationCode) {
        if (!smsService.verifyPhoneAndCode(phoneNo, verificationCode)) {
            // 是否需要重新发送短信验证码
            if (smsService.needResendCode(phoneNo, VERIFICATION_CODE_FINITE_COUNT_NEW, false)) {
                throw new VerificationCodeErrorException("验证码失效，请重新获取");
            }

            smsService.deleteOnlyCodeFromCache(phoneNo);

            log.info("验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
            throw new VerificationCodeErrorException("短信验证码错误");
        } else {
            smsService.deleteCodeFromCache(phoneNo);
        }
    }

    /**
     * 修改手机号登出接口(KDSP调用)
     *
     * @yapi
     */
    @RequestMapping(value = "/kdsp/logout", method = RequestMethod.POST)
    public JsonResult kdspLogout(
            @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "") Long userId,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) String dimension
            , @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId

    ) {
        //查询用户
        User user = userService.findById(userId, tenantId);
        if (Objects.isNull(user) || !user.getEnable()) {
            log.error("用户不存在，或者已经注销，userId:{}", userId);
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_ENABLE_ERROR, null);
        }

        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("未知的连接", null);
        }

        LoginProperties loginProperties = new LoginProperties("", 4, channelId, createdFrom, appChannel, merchant.getId(), merchant.getName(), tenantId);
        try {
            userService.kdspLogout(userId, loginProperties, tenantId);
        } catch (Exception e) {
            return JsonResult.buildErrorStateResult("服务器异常,请稍后再试", null);
        }

        return JsonResult.buildSuccessResult("登出成功");
    }

    @RequestMapping(value = "/kdsp/loginRecord", method = RequestMethod.POST)
    public JsonResult loginRecord(@RequestParam(required = true) String deviceId) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(deviceId)) {
            return JsonResult.buildErrorStateResult("请传入正确的设备ID", null);
        }

        LoginRecord loginRecord = loginRecordService.findFirstLoginRecord(deviceId);
        return JsonResult.buildSuccessResult("", loginRecord);
    }
}
