package com.netty.server.server;


import com.alibaba.fastjson.JSON;
import com.netty.server.channel.ChannelInit;
import com.netty.server.config.ServerProperties;
import com.netty.server.handler.WebsocketMessageHandler;
import com.netty.server.model.ServerInfo;
import com.netty.server.utils.CacheUtil;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import javax.annotation.PreDestroy;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 启动 Server
 *
 * @author qiding
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class TcpServer implements ITcpServer {

    private final ChannelInit channelInit;

    private final ServerProperties serverProperties;

    private EventLoopGroup bossGroup;

    private EventLoopGroup workerGroup;

    private Channel channel;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Autowired
    private CacheService cacheService;

    @Autowired
    private WebsocketMessageHandler websocketHandler;


    @Override
    public void start() throws Exception {
        log.info("初始化 TCP server ...");
        bossGroup = serverProperties.isUseEpoll() ? new EpollEventLoopGroup() : new NioEventLoopGroup();
        workerGroup = serverProperties.isUseEpoll() ? new EpollEventLoopGroup() : new NioEventLoopGroup();
        this.tcpServer();
    }


    /**
     * 初始化
     */
    private void tcpServer() {
        try {
            ServerBootstrap serverBootstrap = new ServerBootstrap();
            serverBootstrap.group(bossGroup, workerGroup)
                    .channel(serverProperties.isUseEpoll() ? EpollServerSocketChannel.class : NioServerSocketChannel.class)
                    .localAddress(new InetSocketAddress(serverProperties.getPort()))
                    // 配置 编码器、解码器、业务处理
                    .childHandler(new ChannelInit(cacheService, websocketHandler))
                    // tcp缓冲区
                    .option(ChannelOption.SO_BACKLOG, 128)
                    // 将网络数据积累到一定的数量后,服务器端才发送出去,会造成一定的延迟。希望服务是低延迟的,建议将TCP_NODELAY设置为true
                    .childOption(ChannelOption.TCP_NODELAY, false)
                    // 保持长连接
                    .childOption(ChannelOption.SO_KEEPALIVE, true);
            ChannelFuture channelFuture = serverBootstrap.bind(new InetSocketAddress(serverProperties.getPort())).syncUninterruptibly();
            this.channel = channelFuture.channel();
            log.info("websocket server启动成功！开始监听端口：{}", serverProperties.getPort());
            String ip = InetAddress.getLocalHost().getHostAddress();
            Date date = new Date();
            //每3秒向注册中心注册一下自己的服务端信息 如果5秒没有注册redis便清除此服务端信息
            CacheUtil.executorService.submit(() -> {
                try {
                    while (channel.isActive()) {
                        redisTemplate.opsForValue().set("newnettyServer" + ip, JSON.toJSONString(new ServerInfo(ip, serverProperties.getPort(), date)), 5 * 1000, TimeUnit.MILLISECONDS);
                        Thread.sleep(3 * 1000);
                    }
                } catch (Exception e) {
                    log.error(e.getMessage());
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
            bossGroup.shutdownGracefully();
            workerGroup.shutdownGracefully();
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    @Override
    public void destroy() {
        if (null == channel) {
            return;
        }
        channel.close();
        bossGroup.shutdownGracefully();
        workerGroup.shutdownGracefully();
    }

}
