package cn.quantgroup.handler;

import cn.quantgroup.server.IStmsServer;
import cn.quantgroup.server.CacheService;
import cn.quantgroup.store.ChannelStore;
import cn.quantgroup.store.WebSocketSession;
import cn.quantgroup.utils.CacheUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.websocketx.WebSocketFrame;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;


/**
 * 消息处理,单例启动
 *
 * @author qiding
 */
@Slf4j
@Component
@ChannelHandler.Sharable
@RequiredArgsConstructor
public class MessageHandler extends SimpleChannelInboundHandler<WebSocketFrame> {

    private  WebsocketMessageHandler websocketHandler;
    private CacheService cacheService;


    public MessageHandler(CacheService cacheService, WebsocketMessageHandler websocketHandler, IStmsServer stmsServer) {
        this.cacheService = cacheService;
        this.websocketHandler = new WebsocketMessageHandler(cacheService, stmsServer);
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        if (msg instanceof FullHttpRequest) {
            websocketHandler.handleHttpRequest(ctx, (FullHttpRequest) msg);
            log.info("\n");
            log.info("客户端{}握手成功!", ctx.channel().id());
        }
        super.channelRead(ctx, msg);
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, WebSocketFrame frame) throws Exception {
        log.info("\n");
        log.info("channelId:" + ctx.channel().id());
        websocketHandler.handleWebSocketFrame(ctx, frame);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
        log.info("\n");
        log.info("断开连接");
        cacheService.getRedisUtil().remove(ctx.channel().id().toString());
        CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
        // 释放缓存
        ChannelStore.closeAndClean(ctx);
        WebSocketSession.clear(ctx.channel().id());
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        log.info("\n");
        log.info("成功建立连接,channelId：{}", ctx.channel().id());
        super.channelActive(ctx);
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        log.info("心跳事件时触发");
        //判断evt是否是IdleStateEvent(用于触发用户事件，包含读空闲/写空闲/读写空闲)
        if(evt instanceof IdleStateEvent){
            IdleStateEvent idleStateEvent = (IdleStateEvent) evt;
            if(idleStateEvent.state() == IdleState.READER_IDLE){
                log.info("进入读空闲...");
            }else if(idleStateEvent.state() == IdleState.WRITER_IDLE){
                log.info("进入写空闲...");
            }else if(idleStateEvent.state() == IdleState.ALL_IDLE){
                log.info("进入读写空闲...");
                cacheService.getRedisUtil().remove(ctx.channel().id().toString());
                CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
                Channel channel = ctx.channel();
                //关闭无用channel，避免浪费资源
                channel.close();
            }
        }
    }

    /**
     * 发现异常关闭连接打印日志
     */
//    @Override
//    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
//        ctx.close();
//        cacheService.getRedisUtil().remove(ctx.channel().id().toString());
//        CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
//        log.error("异常信息：\r\n" + cause.getMessage());
//    }
}
