package com.netty.server.handler;

import com.netty.server.model.DeviceChannelInfo;
import com.netty.server.server.CacheService;
import com.netty.server.store.ChannelStore;
import com.netty.server.store.WebSocketSession;
import com.netty.server.utils.CacheUtil;
import com.netty.server.utils.NetWorkUtils;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.websocketx.TextWebSocketFrame;
import io.netty.handler.codec.http.websocketx.WebSocketFrame;
import io.netty.handler.codec.http.websocketx.WebSocketServerHandshaker;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;


/**
 * 消息处理,单例启动
 *
 * @author qiding
 */
@Slf4j
@Component
@ChannelHandler.Sharable
@RequiredArgsConstructor
public class MessageHandler extends SimpleChannelInboundHandler<WebSocketFrame> {

    private  WebsocketMessageHandler websocketHandler;
    private CacheService cacheService;

    public MessageHandler(CacheService cacheService, WebsocketMessageHandler websocketHandler) {
        this.cacheService = cacheService;
        this.websocketHandler = new WebsocketMessageHandler(cacheService);
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        if (msg instanceof FullHttpRequest) {
            websocketHandler.handleHttpRequest(ctx, (FullHttpRequest) msg);
            log.debug("\n");
            log.debug("客户端{}握手成功!", ctx.channel().id());
        }
        super.channelRead(ctx, msg);
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, WebSocketFrame frame) throws Exception {
        log.debug("\n");
        log.debug("channelId:" + ctx.channel().id());
        websocketHandler.handleWebSocketFrame(ctx, frame);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
        log.debug("\n");
        log.debug("断开连接");
        cacheService.getRedisUtil().remove(ctx.channel().id().toString());
        CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
        // 释放缓存
        ChannelStore.closeAndClean(ctx);
        WebSocketSession.clear(ctx.channel().id());
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        log.debug("\n");
        log.debug("成功建立连接,channelId：{}", ctx.channel().id());
        super.channelActive(ctx);
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        log.debug("心跳事件时触发");
        //判断evt是否是IdleStateEvent(用于触发用户事件，包含读空闲/写空闲/读写空闲)
        if(evt instanceof IdleStateEvent){
            IdleStateEvent idleStateEvent = (IdleStateEvent) evt;

            if(idleStateEvent.state() == IdleState.READER_IDLE){
                log.debug("进入读空闲...");
            }else if(idleStateEvent.state() == IdleState.WRITER_IDLE){
                log.debug("进入写空闲...");
            }else if(idleStateEvent.state() == IdleState.ALL_IDLE){
                log.debug("进入读写空闲...");
                cacheService.getRedisUtil().remove(ctx.channel().id().toString());
                CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
                Channel channel = ctx.channel();
                //关闭无用channel，避免浪费资源
                channel.close();
            }
        }
    }

    /**
     * 发现异常关闭连接打印日志
     */
//    @Override
//    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
//        ctx.close();
//        cacheService.getRedisUtil().remove(ctx.channel().id().toString());
//        CacheUtil.cacheChannel.remove(ctx.channel().id().toString(), ctx.channel());
//        log.error("异常信息：\r\n" + cause.getMessage());
//    }
}
