package com.netty.server.utils;

import com.alibaba.fastjson.JSON;
import com.netty.server.model.DeviceChannelInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


/**
 * @author jie
 */
@Service
public class RedisUtil {

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Value("${netty.deviceIds.redis.key}")
    private String nettyDeviceIds;

    @Value("${netty.channelRelations.redis.key}")
    private String nettyChannelRelations;

    /**
     * 向redis存入设备连接信息
     *
     * @param deviceChannelInfo
     */
    public void pushObj(DeviceChannelInfo deviceChannelInfo) {
        redisTemplate.opsForHash().put(nettyDeviceIds, deviceChannelInfo.getChannelId(), JSON.toJSONString(deviceChannelInfo));
    }

    public void pushChannelRelation(String channelId, String relationInfo){
        Object channelIdObj = redisTemplate.opsForHash().get(nettyChannelRelations, relationInfo);
        String channelIds = "";
        if (null != channelIdObj) {
            channelIds = channelIdObj.toString() + ",";
        }
        channelIds = channelIds + channelId;
        redisTemplate.opsForHash().put(nettyChannelRelations, relationInfo, channelIds);
    }

    public String getChannelRelation(String relationInfo){
        Object channelIdObj = redisTemplate.opsForHash().get(nettyChannelRelations, relationInfo);
        if (null != channelIdObj) {
            return channelIdObj.toString();
        }
        return null;
    }

    /**
     * 查询redis中的设备连接信息
     *
     * @return
     */
    public List<DeviceChannelInfo> popList() {
        List<Object> values = redisTemplate.opsForHash().values(nettyDeviceIds);
        if (null == values) {
            return new ArrayList<>();
        }
        List<DeviceChannelInfo> deviceChannelInfoList = new ArrayList<>();
        for (Object strJson : values) {
            deviceChannelInfoList.add(JSON.parseObject(strJson.toString(), DeviceChannelInfo.class));
        }
        return deviceChannelInfoList;
    }

    /**
     * 根据channelId查询连接信息
     *
     * @param channelId
     * @return
     */
    public DeviceChannelInfo selectByChannel(String channelId) {
        Object deviceIds = redisTemplate.opsForHash().get(nettyDeviceIds, channelId);
        if (deviceIds == null) {
            return null;
        }
        return JSON.parseObject(deviceIds.toString(), DeviceChannelInfo.class);
    }

    /**
     * 移除某个设备信息
     *
     * @param channelId
     */
    public void remove(String channelId) {
        DeviceChannelInfo deviceChannelInfo = selectByChannel(channelId);
        if (deviceChannelInfo != null) {
            removeChannelRelations(channelId, deviceChannelInfo.getRelationInfo());
        }
        redisTemplate.opsForHash().delete(nettyDeviceIds, channelId);
    }

    private void removeChannelRelations(String channelId, String relationInfo) {
        Object channelIdObj = redisTemplate.opsForHash().get(nettyChannelRelations, relationInfo);
        if (null != channelIdObj) {
            String channelIds = "";
            List<String> channelIdlist = new ArrayList<>(Arrays.asList(channelIdObj.toString().split(",")));
            channelIdlist.remove(channelId);
            if (channelIdlist.size() == 0) {
                redisTemplate.opsForHash().delete(nettyChannelRelations, relationInfo);
            } else {
                channelIds = String.join(",", channelIdlist);
                redisTemplate.opsForHash().put(nettyChannelRelations, relationInfo, channelIds);
            }
        }
    }

    /**
     * 清空设备信息
     */
    public void clear() {
        redisTemplate.delete(nettyDeviceIds);
    }
}
