package com.service.report.impl;

import com.QgException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.config.ReportConfig;
import com.controller.ReportController;
import com.emums.BusinessType;
import com.emums.ErrorReson;
import com.emums.InfoType;
import com.emums.PathType;
import com.entity.FileEntry;
import com.entity.report.ReportRecord;
import com.google.common.collect.ImmutableMap;
import com.mapper.data.DataMapper;
import com.mapper.report.ReportRecordMapper;
import com.service.report.ReportService;
import com.sun.jersey.api.client.ClientResponse;
import com.timer.ReportTimer;
import com.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @author ：dongjianhua
 * @date ：Created in 2019/10/30 14:14
 * @description：上报实现类
 * @modified By：
 * @version: 1.0
 */
@Service
@Slf4j
public class IReportServiceImpl implements ReportService {
    @Autowired
    private ReportRecordMapper reportRecordMapper;
    @Autowired
    private DataMapper dataMapper;
    @Autowired
    private ReportConfig reportConfig;
    @Autowired
    private Utils utils;

    @Autowired
    @Qualifier("reportJerseyClient")
    private JerseyClient reportJerseyClient;

    @Autowired
    @Qualifier("taskJerseyClient")
    private JerseyClient taskJerseyClient;

    @Autowired
    private DingTalk dingTalk;
    /**
     * 脱敏的时候名字
     */
    public static final String[] NAMES = {"伟霆", "诗诗", "艺兴", "笔畅", "德华", "振宇", "千玺", "幂"};
    /**
     * 脱敏的时候身份证前两位随机取的数组
     */
    public static final Integer[] IDCARDNOPRE = {11, 12, 13, 14, 15, 21, 22, 23, 31, 32, 33, 34, 35, 36, 37, 41, 42, 43, 44, 45, 46, 50, 51, 52, 53, 54, 61, 62, 63, 64, 65, 71, 81, 82};
    //金额字段的位置
    public static final Integer[] MONEYINDEX = {9, 11, 12};
    /**
     * 所有的key位置是固定的
     */
    public static final String[] KEYS = {"name", "pidType", "pid", "businessHappenOrgan", "loanId", "businessType", "businessKind", "openDate", "dueDate", "creditMoney", "businessHappenDate", "balance", "currentOverdueTotal", "repaymentStatus"};
    /**
     * 姓名 在key中的位置
     */
    public static final Integer NAMEINDEX = 0;
    /**
     * 证件类型 在keys 数组的位置
     */
    public static final Integer CERTIFICATETYPEINDEX = 1;
    /**
     * 证件号码在 keys 数组的位置
     */
    public static final Integer IDCARDEINDEX = 2;
    /**
     * 统一社会信用代码的在 keys 数组的位置
     */
    public static final Integer CODEINDEX = 3;
    /**
     * 日期类型的字段 在key中的位置
     */
    public static final Integer DATETYPEINDEXS[] = {7, 8, 10};

    /**
     * 换行符
     */
    public static final String NEWLINE = "\r\n";
    /**
     * 分割符号
     */
    public static final String SPLIT = ",";
    /**
     * zip后缀
     */
    public static final String ZIP_SUFFIX = ".zip";
    /**
     * enc后缀
     */
    public static final String ENC_SUFFIX = ".enc";
    /**
     * txt后缀
     */
    public static final String TXT_SUFFIX = ".txt";
    /**
     * 字符集名称
     */
    public static final String CHARSET_NAME = "GBK";


    @Override
    public FileEntry creatZipFile(List<Map<String, Object>> dataList, InfoType infoType, BusinessType businessType) throws QgException {
        return creatZipFile(null, dataList, infoType, businessType);
    }

    @Override
    public FileEntry creatZipFile(String date, List<Map<String, Object>> dataList, InfoType infoType, BusinessType businessType) throws QgException {
        if (dataList == null || dataList.size() == 0) {
            return null;
        }
        String packageName = getPackageName(date, infoType, businessType,null);

        String str = formatData(dataList, reportConfig.isDesensitization());

        String savepath = reportConfig.getFilePathPre() + File.separator + DateUtil.getCurrentServerDate(DateUtil.YYYYMMDD) + File.separator + PathType.REPORT.getPath();
        //modify 2020.05.29
        if(savepath!=null && savepath.indexOf("//")!=-1){
            String tmp_savepath = savepath;
            savepath = savepath.replaceAll("//", "/");
            log.info("发现文件路径中有//替换, 原: {} , 新: {} ", tmp_savepath, savepath);
        }

        File file = new File(savepath);
        if (!file.exists()) {
            file.mkdirs();
        }
        String zipFilePath = savepath + File.separator + packageName + ZIP_SUFFIX;
        String entryFileName = "";
        if (infoType.equals(InfoType.CREDIT_FINANCING_INFO)) {
            entryFileName = reportConfig.getBusinfoFileName();
        } else {
            entryFileName = reportConfig.getDelinfoFileName();
        }
        createFile(null, str, zipFilePath, entryFileName);
        FileEntry fileEntry = new FileEntry();
        fileEntry.setFileName(packageName);
        fileEntry.setFilePath(savepath);
        return fileEntry;
    }

    @Override
    public FileEntry creatZipFileByString(List<String> dataList, InfoType infoType, BusinessType businessType, int index) throws QgException {
        String packageName = getPackageName(null, infoType, businessType,"000"+index);

        StringBuilder str = new StringBuilder();

        for (String string : dataList) {
            str.append(string).append(NEWLINE);
        }
        String savepath = reportConfig.getFilePathPre() + File.separator + DateUtil.getCurrentServerDate(DateUtil.YYYYMMDD) + File.separator + PathType.REPORT.getPath();

        File file = new File(savepath);
        if (!file.exists()) {
            file.mkdirs();
        }
        String zipFilePath = savepath + File.separator + packageName + ZIP_SUFFIX;
        String entryFileName = "";
        if (infoType.equals(InfoType.CREDIT_FINANCING_INFO)) {
            entryFileName = reportConfig.getBusinfoFileName();
        } else {
            entryFileName = reportConfig.getDelinfoFileName();
        }
        createFile(null, str.toString(), zipFilePath, entryFileName);
        FileEntry fileEntry = new FileEntry();
        fileEntry.setFileName(packageName);
        fileEntry.setFilePath(savepath);
        return fileEntry;
    }

    @Override
    public void creatTxtFile(String filePath, List<Map<String, Object>> dataList, InfoType infoType, BusinessType businessType, boolean isDesensitization) throws QgException {
        if (dataList == null || dataList.size() == 0) {
            return;
        }
        String str = formatData(dataList, isDesensitization);

//        String savepath = reportConfig.getFilePathPre() + File.separator + DateUtil.getCurrentServerDate(DateUtil.YYYYMMDD) + File.separator + PathType.REPORT.getPath();

        File file = new File(filePath);
        if (!file.exists()) {
            file.mkdirs();
        }
        String entryFileName = "";

        if (infoType.equals(InfoType.CREDIT_FINANCING_INFO)) {
            entryFileName = reportConfig.getBusinfoFileName();
        } else {
            entryFileName = reportConfig.getDelinfoFileName();
        }

        String txtFile = filePath + File.separator + entryFileName + TXT_SUFFIX;
        File txt = new File(txtFile);
        try {
            if (txt.exists()) {
                str = NEWLINE + str;
            }
            FileUtils.write(txt, str, CHARSET_NAME, true);
        } catch (Exception e) {
            log.error("创建文件异常：e:", e);
            throw new QgException("创建文件异常", e);
        }
    }

    @Override
    public List<Map<String, Object>> getDataAsync(Map<String, String> param) {
        CountDownLatch countDownLatch = new CountDownLatch(3);
        List<Map<String, Object>> loanDayData = new ArrayList<>();
        ExecutorService instance = ThreadPoolUtil.getInstance();
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> dayLoanData = dataMapper.getLoanDayData(param);
                synchronized (loanDayData) {
                    if (dayLoanData != null && dayLoanData.size() > 0) {
                        loanDayData.addAll(dayLoanData);
                    }
                }

                countDownLatch.countDown();
            }
        });
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> repayDayData = dataMapper.getRepayDayData(param);
                synchronized (loanDayData) {
                    if (repayDayData != null && repayDayData.size() > 0) {
                        loanDayData.addAll(repayDayData);
                    }
                }
                countDownLatch.countDown();
            }
        });
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> overDayData = dataMapper.getOverDayData(param);
                synchronized (loanDayData) {
                    if (overDayData != null && overDayData.size() > 0) {
                        loanDayData.addAll(overDayData);
                    }
                }
                countDownLatch.countDown();
            }
        });
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        instance.shutdown();
        return loanDayData;
    }

    @Override
    public FileEntry creatZipFileByFile(MultipartFile multipartFile, InfoType infoType, BusinessType businessType) throws QgException {
        String packageName = getPackageName(null, infoType, businessType,null);
        String savepath = reportConfig.getFilePathPre() + File.separator + DateUtil.getCurrentServerDate(DateUtil.YYYYMMDD) + File.separator + PathType.REPORT.getPath();

        File file = new File(savepath);
        if (!file.exists()) {
            file.mkdirs();
        }
        String zipFilePath = savepath + File.separator + packageName + ZIP_SUFFIX;

        String entryFileName = "";
        if (infoType.equals(InfoType.CREDIT_FINANCING_INFO)) {
            entryFileName = reportConfig.getBusinfoFileName();
        } else {
            entryFileName = reportConfig.getDelinfoFileName();
        }
        createFile(multipartFile, null, zipFilePath, entryFileName);
        FileEntry fileEntry = new FileEntry();
        fileEntry.setFileName(packageName);
        fileEntry.setFilePath(savepath);
        return fileEntry;
    }

    private void createFile(MultipartFile multipartFile, String file, String zipFilePath, String entryFileName) {
        ZipOutputStream out = null;
        BufferedOutputStream bufferedOutputStream = null;
        FileOutputStream fileOutputStream = null;
        byte[] bytes = null;
        try {
            if (multipartFile == null) {
                bytes = file.getBytes(CHARSET_NAME);
            } else {
                bytes = multipartFile.getBytes();
            }
            fileOutputStream = new FileOutputStream(zipFilePath);
            bufferedOutputStream = new BufferedOutputStream(fileOutputStream);
            out = new ZipOutputStream(bufferedOutputStream);
            out.putNextEntry(new ZipEntry(entryFileName + TXT_SUFFIX));
            out.write(bytes);
            out.closeEntry();
        } catch (Exception e) {
            log.error("创建文件异常：e:", e);
            throw new QgException("创建文件异常", e);
        } finally {
            try {
                if (out != null) {
                    out.close();
                }
                if (bufferedOutputStream != null) {
                    bufferedOutputStream.close();
                }
                if (fileOutputStream != null) {
                    fileOutputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }

        }
    }

    @Override
    public void reportData(FileEntry fileEntry) throws QgException {
        if (fileEntry == null) {
            throw new QgException("文件不存在", "fileEntry:" + fileEntry);
        }
        ClientResponse resp = null;

        String zipFilePath = fileEntry.getFilePath() + File.separator + fileEntry.getFileName() + ZIP_SUFFIX;
//		// zip通过国密算法加密为enc文件
        String encFilePath = fileEntry.getFilePath() + File.separator + fileEntry.getFileName() + ENC_SUFFIX;
        File zipFile = new File(zipFilePath);
        if (!zipFile.exists()) {
            throw new QgException("文件不存在", zipFilePath);
        }
        File file = utils.encryptFile(zipFilePath, encFilePath);
        // 读取enc文件为字符串
        String base64EncString = utils.file2Base64String(file);
        // 构造body
        Map<String, String> bodyMap = new HashMap<String, String>();
        bodyMap.put("sdata", base64EncString);
//        log.info("[prefixparedMap]: \n" + utils.map2Json(bodyMap));
        // 构造header
        Map<String, String> headerMap = getHeaderMap(bodyMap, fileEntry.getFileName());
        log.info("[header]: \n" + utils.map2Json(headerMap));
        try {
            resp = reportJerseyClient.post(headerMap, bodyMap, reportConfig.getDataUri());
            log.info("上报返回结果, fileName: {} , resp: {} ", fileEntry.getFileName(), resp);
            String entitystr = resp.getEntity(String.class);
            log.info("上报返回结果, fileName: {} , entitystr: {} ", fileEntry.getFileName(), entitystr);
            ReportRecord record = new ReportRecord();
            record.setFilename(fileEntry.getFileName());
            record.setFilepath(fileEntry.getFilePath());
            record.setCreatetime(new Date());
            record.setReportmsg(entitystr.length() > 250 ? entitystr.substring(0, 249) : entitystr);
            if (resp.getStatus() == 200) {
                JSONObject jsonEntity = JSONObject.parseObject(entitystr);
                if (jsonEntity.get("msgCode").equals("20000")) {
                    record.setStatus(1);
                    reportRecordMapper.insert(record);
                } else {
                    record.setStatus(0);
                    reportRecordMapper.insert(record);
                    throw new QgException("上报接口返回失败", entitystr);
                }
                /*
                 *  不修改任何配置时返回：
                 *  {"msgCode":"70000","msgContent":"文件包名重复!"}
                 *  因为MA05M6KK9201810221201120002这个文件已经报送过
                 *
                 *  报送成功时返回：
                 *  {"msgCode":"20000","msgContent":"成功!"}
                 */
            } else {
                record.setStatus(0);
                reportRecordMapper.insert(record);
                throw new QgException("上报接口返回失败", entitystr);
            }
        } catch (QgException e) {
            log.error("上报接口请求异常QgException, fileName: {} ", fileEntry.getFileName(), e);
            throw e;
        } catch (Exception e) {
            log.error("上报接口请求异常Exception, fileName: {} ", fileEntry.getFileName(), e);
            ReportRecord record = new ReportRecord();
            record.setFilename(fileEntry.getFileName());
            record.setFilepath(fileEntry.getFilePath());
            record.setCreatetime(new Date());
            record.setStatus(0);
            record.setReportmsg("上报接口请求异常");
            reportRecordMapper.insert(record);
            throw new QgException("上报接口请求异常", e);
        } finally {
            if (file != null) {
                if (file.exists()) {
                    file.delete();
                }
            }
        }

    }

    @Override
    public void quaryReportedStatus() {
        List<ReportRecord> reportedRecord = reportRecordMapper.getReportedRecord();
        reportedRecord.parallelStream().forEach(record -> {
            FileEntry fileEntry = new FileEntry();
            fileEntry.setFilePath(record.getFilepath());
            fileEntry.setFileName(record.getFilename());
            try {
                quaryReportedStatus(fileEntry);
            } catch (QgException qge) {
                dingTalk.talk("查询上报状态接口告警", qge);
            } catch (Exception e){
                log.error("查询上报状态接口异常Filename: {} ",record.getFilename(),e);
                dingTalk.talk("查询上报状态接口异常", "filename:"+record.getFilename());
            }
        });
    }

    //查询报送状态
    @Override
    public void quaryReportedStatus(FileEntry fileEntry) throws QgException {

        ClientResponse resp = null;
        // 构造body
        Map<String, String> preparedMap = new HashMap<String, String>();
        preparedMap.put("sdatacode", fileEntry.getFileName());
        log.info("[preparedMap]: \n" + utils.map2Json(preparedMap));
        // 构造header
        Map<String, String> headerMap = getHeaderMap(preparedMap);
        log.info("查看报送状态入参, [header]: \n" + utils.map2Json(headerMap));
        try {
            resp = taskJerseyClient.post(headerMap, null, reportConfig.getTaskUri());
        } catch (Exception e) {
            //log.error("查看 fileName: {} ,异常e: {} ", fileEntry.getFileName(), e);
            log.error("查看报送状态接口异常, fileName: {} ", fileEntry.getFileName(), e);
            throw new QgException(fileEntry.getFileName() + "查看报送状态接口异常", e);
        }
        if (resp.getStatus() == 200) {
            /*
             * 不修改任何配置时默认返回：
             * {"ordernum":"1","sfilename":"MA05M6KK9201810221201120001",
             * "sfilesymbol":"个人债权融资信息","createtime":"2018-10-23 19:19:00",
             * "istatus":"处理完毕有反馈","returnresult":null,
             * "returnresultdata":"BHz5wCGG+sI9NCiUKRvQnqMoi8......",
             * "loadcount":2,"errorcount":1,"msgCode":"200","msgContent":"成功!"}
             * 存在入库失败的数据，见tmp/test.txt
             */
            String responseString = resp.getEntity(String.class);
            log.info("查看报送状态结束200, fileName: {} , entity: {} ", fileEntry.getFileName(), responseString);
            JSONObject json = JSON.parseObject(responseString);
            // 如果有反馈文件 说明有数据未成功入库
            // 反馈报文保存为txt文件
            if (json.containsKey("returnresultdata") && json.get("returnresultdata") != null && !json.get("returnresultdata").equals("null")) {
                String enc = fileEntry.getFilePath() + File.separator + PathType.FEEDBACK.getPath() + File.separator + getFeedbackFileName(fileEntry.getFileName()) + ENC_SUFFIX;
                String txt = fileEntry.getFilePath() + File.separator + PathType.FEEDBACK.getPath() + File.separator + getFeedbackFileName(fileEntry.getFileName()) + TXT_SUFFIX;
                File decEncFile = new File(enc); //生成enc解密文件
                try {
                    FileUtils.writeByteArrayToFile(decEncFile, Base64.decodeBase64(json.get("returnresultdata").toString()));
                } catch (IOException e) {
                    log.error("查看报送状态失败！FileName :  {} ,e : {}", fileEntry.getFileName(), e);
                    throw new QgException(fileEntry.getFileName() + "查看报送状态文件转换异常", e);
                }
                utils.deccryptFile(decEncFile.getPath(), txt);
                decEncFile.delete();
                ReportRecord record = new ReportRecord();
                record.setFeedback(json.getString("istatus"));
                record.setErrorcount(json.getInteger("errorcount"));
                record.setLoadcount(json.getInteger("loadcount"));
                record.setIssearch(1);
                record.setFilename(fileEntry.getFileName());
                reportRecordMapper.updateByPrimaryKeySelectiveByFileName(record);
                throw new QgException("查看报送状态接口有反馈", "文件位置" + txt);
            } else {
                ReportRecord record = new ReportRecord();
                record.setFeedback(json.getString("istatus"));
                record.setErrorcount(json.getInteger("errorcount"));
                record.setLoadcount(json.getInteger("loadcount"));
                record.setIssearch(1);
                record.setFilename(fileEntry.getFileName());
                reportRecordMapper.updateByPrimaryKeySelectiveByFileName(record);
                if (json.getString("istatus") == null || json.getString("istatus").equals("null")) {
                    throw new QgException("查看报送状态接口返回", json.toJSONString());
                }
                if(json.getString("istatus").equals("处理异常")){
                    throw new QgException("查看报送状态接口返回", json.toJSONString());
                }
            }
        } else {
            log.error("报送状态查询接口失败, fileName: {} , Status: {} , entity: {} ", fileEntry.getFileName(), resp.getStatus(), resp.getEntity(String.class));
            //log.info("[ERROR] Status: " + resp.getStatus());
//            log.info(resp.getEntity(String.class));
            throw new QgException("报送状态查询接口失败", "http 状态="+resp.getStatus()+", fileName="+fileEntry.getFileName());
        }
    }

    @Override
    public void reportLoanDayData() {
        try {
            log.info("开始上报T+1放款数据");
            Map<String, String> param = new HashMap<>();
            param.put("startDate", DateUtil.getCurrentDay(-1));
            param.put("endDate", DateUtil.getCurrentDay(0));
            List<Map<String, Object>> loanDayData = dataMapper.getLoanDayData(param);
            if (loanDayData == null || loanDayData.size() == 0) {
                return;
            }
            log.info("获取T+1放款数据条数 size : {} ", loanDayData.size());
            FileEntry fileEntry = creatZipFile(loanDayData, InfoType.CREDIT_FINANCING_INFO, BusinessType.LOAN);
            log.info("放款数据生成文件位置fileEntry : {}", fileEntry);
            reportData(fileEntry);
            log.info("放款数据上报结束");
            ReportTimer.result.put("T+1放款数据条数：",loanDayData.size());
            insertReportLogAsync(loanDayData, fileEntry,"1");
        } catch (QgException e) {
            log.error("上报T+1放款数据异常，e : {} ", e);
            dingTalk.talk("上报T+1放款数据告警", e);
        } catch (Exception e) {
            log.error("上报T+1放款数据系统异常，e : {} ", e);
            dingTalk.talk("上报T+1放款数据系统异常告警", e);
        }
    }

    @Override
    public void reportRepayDayData() {
        try {
            Map<String, String> param = new HashMap<>();
            param.put("startDate", DateUtil.getCurrentDay(-1));
            param.put("endDate", DateUtil.getCurrentDay(0));
            log.info("开始上报T+1还款数据");
            List<Map<String, Object>> repayDayData = dataMapper.getRepayDayData(param);
            if (repayDayData == null || repayDayData.size() == 0) {
                return;
            }
            log.info("获取T+1还款数据条数 size : {} ", repayDayData.size());
            FileEntry fileEntry = creatZipFile(repayDayData, InfoType.CREDIT_FINANCING_INFO, BusinessType.REPAYMENT);
            log.info("还款数据生成文件位置fileEntry : {}", fileEntry);
            reportData(fileEntry);
            log.info("还款数据上报结束");
            ReportTimer.result.put("T+1还款数据条数：",repayDayData.size());
            insertReportLogAsync(repayDayData,fileEntry,"2");
        } catch (QgException e) {
            log.error("上报T+1还款数据异常，e : {} ", e);
            dingTalk.talk("上报T+1还款数据告警", e);
        } catch (Exception e) {
            log.error("上报T+1还款数据系统异常，e : {} ", e);
            dingTalk.talk("上报T+1还款数据系统异常告警", e);
        }

    }

    @Override
    public void reportOverDayData() {
        try {
            Map<String, String> param = new HashMap<>();
            param.put("startDate", DateUtil.getCurrentDay(-1));
            param.put("endDate", DateUtil.getCurrentDay(0));
            log.info("开始上报T+1逾期数据");
            List<Map<String, Object>> overDayData = dataMapper.getOverDayData(param);

            if (overDayData == null || overDayData.size() == 0) {
                return;
            }
            log.info("获取T+1逾期数据条数 size : {} ", overDayData.size());
            FileEntry fileEntry = creatZipFile(overDayData, InfoType.CREDIT_FINANCING_INFO, BusinessType.OVERDUE);
            log.info("逾期数据生成文件位置fileEntry : {}", fileEntry);
            reportData(fileEntry);
            log.info("逾期数据上报结束");
            ReportTimer.result.put("T+1逾期数据条数：",overDayData.size());
            insertReportLogAsync(overDayData, fileEntry,"3");
        } catch (QgException e) {
            log.error("上报T+1逾期数据异常，e : {} ", e);
            dingTalk.talk("上报T+1逾期数据告警", e);
        } catch (Exception e) {
            log.error("上报T+1逾期数据系统异常，e : {} ", e);
            dingTalk.talk("上报T+1逾期数据系统异常告警", e);
        }

    }

    @Override
    public void reportOverMonthData() {
        try {
            Map<String, String> param = new HashMap<>();
            param.put("startDate", DateUtil.getCurrentDay(-1, DateUtil.YYYY_MM_DD));
            param.put("endDate", DateUtil.getCurrentDay(0, DateUtil.YYYY_MM_DD));
            log.info("开始上报月逾期数据param:{}",param);
            List<Map<String, Object>> overMonthData = dataMapper.getOverMonthData(param);
            if (overMonthData == null || overMonthData.size() == 0) {
                return;
            }
            log.info("获取月逾期数据条数 size : {} ", overMonthData.size());
            FileEntry fileEntry = creatZipFile(overMonthData, InfoType.CREDIT_FINANCING_INFO, BusinessType.OVERDUEMON);
            log.info("月逾期数据生成文件位置fileEntry : {}", fileEntry);
            reportData(fileEntry);
            log.info("月逾期数据上报结束 上报总条数: {}",overMonthData.size());
            ReportTimer.result.put("月逾期数据条数：",overMonthData.size());
            insertReportLogAsync(overMonthData, fileEntry,"4");
        } catch (QgException e) {
            log.error("上报月逾期数据异常，e : {} ", e);
            dingTalk.talk("上报月逾期数据告警", e);
        } catch (Exception e) {
            log.error("上报月逾期数据系统异常，e : {} ", e);
            dingTalk.talk("上报月逾期数据系统异常告警", e);
        }

    }


    @Async
    @Override
    public void runData(String date) {
        ReportController.isrun = true;
        try {
//            String date = "2017-10-01 00:00:00";
            while (!date.equals("2019-11-13 00:00:00")) {
                Map<String, String> param = new HashMap<>();
                param.put("startDate", DateUtil.getCurrentDay(date, -1));
                param.put("endDate", date);
                Map<String, List<Map<String, Object>>> dataSync = getDataSync(param);
                Set<Map.Entry<String, List<Map<String, Object>>>> entries = dataSync.entrySet();
                for (Map.Entry<String, List<Map<String, Object>>> entry : entries) {
                    List<List<Map<String, Object>>> lists = utils.subList(entry.getValue(), 1000, entry.getKey(),null);
                    for (List<Map<String, Object>> list : lists) {
                        reportRecordMapper.banchInsert(list);
                    }
                }
                date = DateUtil.getCurrentDay(date, 1);
            }
        } finally {
            ReportController.isrun = false;
        }
    }

    @Async
    @Override
    public void runMonthData(String date) {
        ReportController.isrun = true;
        try {
            while (!date.equals("2019-12-01")) {
                log.info("开始跑取" + date + "的数据");
                Map<String, String> param = new HashMap<>();
                param.put("startDate", DateUtil.getCurrentDay(date, -1, DateUtil.YYYY_MM_DD, DateUtil.YYYY_MM_DD));
                param.put("endDate", date);
                List<Map<String, Object>> monthData = getMonthData(param);
                List<List<Map<String, Object>>> lists = utils.subList(monthData, 1000, "4",null);
                for (List<Map<String, Object>> list : lists) {
                    reportRecordMapper.banchInsert(list);
                }
                log.info("跑取" + date + "的数据结束");
                date = DateUtil.getCurrentMonth(date, 1, DateUtil.YYYY_MM_DD, DateUtil.YYYY_MM_DD);
            }
        } finally {
            ReportController.isrun = false;
        }

    }

    @Override
    @Async
    public void autoReReport(String path) {
        List<String> dataFromFile = utils.getDataFromFile(path);
        Map<ErrorReson, List<String>> classify = classify(dataFromFile);
        reReportClassify(classify);
    }

    @Override
    public void insertReportLog(String filePath) {
        log.info("开始处理filename:{}",filePath);
        List<Map<String, Object>> maps = Utils.unZip(filePath);
        List<List<Map<String, Object>>> lists = utils.subList(maps, 1000, null,null);
        for (List<Map<String, Object>> logs : lists) {
            reportRecordMapper.banchInsert(logs);
        }
        log.info("filename:{},处理完成",filePath);
    }

    @Override
    public void insertReportLogAsync(final List<Map<String, Object>> list,final FileEntry fileEntry,final String type) {
        ThreadPoolUtil.getInstance().execute(()->{
            List<List<Map<String, Object>>> lists = utils.subList(list, 1000, type,fileEntry.getFileName());

            for (List<Map<String, Object>> logs : lists) {
                formatData(logs);
                reportRecordMapper.banchInsert(logs);
            }
        });
    }

    private void formatData(List<Map<String, Object>> list){
        int size = list.size();
        for (int j = 0; j < size; j++) {
            Map<String, Object> objectMap = list.get(j);
            for (int i = 0; i < KEYS.length; i++) {
                Object value = objectMap.get(KEYS[i]);
                if (i == IDCARDEINDEX) {
                    if (String.valueOf(objectMap.get(KEYS[CERTIFICATETYPEINDEX])).equals("0")) {//身份证的情况下保证最后一位是大写
                        value = utils.manageIdCard(value.toString());
                    }
                }
                if (i == CODEINDEX) {
                    value = reportConfig.getCode();//组织结构代码
                }

                for (Integer moneyindex : MONEYINDEX) {
                    if (i == moneyindex) {
                        value = utils.getInt(value);
                        break;
                    }
                }
                objectMap.put(KEYS[i],value);
            }

        }
    }
    @Override
    public void insertLog() {
        List<ReportRecord> reportedRecord = reportRecordMapper.getAllRecord();
        int count = 0;
        for (ReportRecord record : reportedRecord) {
            count++;
            String filepath = record.getFilepath();
            String filename = record.getFilename();
            String path =  filepath +File.separator+filename+ZIP_SUFFIX;
            insertReportLog(path);
            log.info("处理进度 {}/{}",count,reportedRecord.size());
        }
    }

    /**
     * -----------------------------------------------------------------------------<br>
     * 描  述: 请求被查询人的信用报告 <br>
     * 创建人: yanhui.Hao <br>
     * 创建时间: 2020.04.08 17:18 <br>
     * 最后修改人:  <br>
     * 最后修改时间: 2020.04.08 17:18 <br>
     * 入参说明: 入参说明: [sname 姓名, stype 证件类型(0-身份证;A-香港身份证;B-澳门身份证;C-台湾身份证;X-其他证件;1-户口簿;2-护照;3-军官证;4-士兵证;5-港澳居民来往内地通行证;6-台湾同胞来往内地通行证;7-临时身份证;8-外国人居留证;9-警官证;D-组织机构代码),
     *          sreason 查询原因(a-贷前审批;b-贷后管理;c-担保资格审查;e-其他),
     *          sno] <br>
     * 出参说明: java.util.Map<java.lang.String,java.lang.String> <br>
     * -----------------------------------------------------------------------------
     */
    @Override
    public Map<String, String> creditReport(String sname, String stype, String sreason, String sno) {
        log.info("中互金共享数据查询入参, sname: {} , stype: {} , sreason: {} , sno: {} ", sname, stype, sreason, sno);
        if(StringUtils.isAnyBlank(sname, stype, sreason, sno)){
            return ImmutableMap.of("code","301","msg","参数不能为空","data", null);
        }
        // 构造body
        Map<String, String> bodyMap = new HashMap();
        bodyMap.put("sname", sname);
        bodyMap.put("stype",stype);
        bodyMap.put("sreason", sreason);
        bodyMap.put("sno", sno);
        Map<String, String> infoHeaderMap = getInfoHeaderMap(sname, stype, sreason, sno);

        log.info("中互金共享数据查询开始, sname: {} , stype: {} , sreason: {} , sno: {} , header: {} , body: {} ", sname, stype, sreason, sno, JSON.toJSONString(infoHeaderMap), JSON.toJSONString(bodyMap));

        ClientResponse resp = null;
        try {
            resp = reportJerseyClient.post(infoHeaderMap, bodyMap, reportConfig.getInfoUrl());

            if (resp!=null){
                String result = resp.getEntity(String.class);
                int status = resp.getStatus();
                log.info("中互金共享数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , result: {} , status: {} ", sname, stype, sreason, sno, result, status);

                if (resp.getStatus() == 200) {
                    return ImmutableMap.of("code","200","msg","查询成功","data", result);
                    /*
                     * 不修改任何配置时默认返回：
                     * {"loancount":"58","loanamt":"5800000","outstandcount":"58","loanbal":"5800000",
                     * "generationcount":"0","generationamount":"0","overduecount":"0","overdueamt":"0",
                     * "overduemorecount":"0","overduemoreamt":"0",
                     * "totalorg":"最近1个月被6家机构查询",
                     * "queryatotalorg":"最近1个月被6家机构以“贷前审批”原因查询",
                     * "infoquerybean":[{"ordernum":"1","ddate":"2018-10-23","s_value":"贷前审批"},{"ordernum":"2","ddate":"2018-10-23","s_value":"其他"......}],
                     * "msgCode":"200","msgContent":"成功!"}
                     */
                }else {
                    log.info("中互金共享数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , 调用失败: {} ", sname, stype, sreason, sno, JSON.toJSONString(resp));
                    return ImmutableMap.of("code","999","msg","查询失败status="+status,"data", result);
                }
            }else{
                log.info("中互金共享数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , result is Null.", sname, stype, sreason, sno);
                return ImmutableMap.of("code","500","msg","查询返回结果为空");
            }
        }catch (Exception e){
            log.error("中互金共享数据查询异常, sname: {} , stype: {} , sreason: {} , sno: {} ", sname, stype, sreason, sno, e);
            return ImmutableMap.of("code","500","msg","调用中互金共享数据接口请求异常");
        }
    }

    /**
     * -----------------------------------------------------------------------------<br>
     * 描  述: 被查询人是否被列入司法失信被执行人黑名单 <br>
     * 创建人: yanhui.Hao <br>
     * 创建时间: 2020.04.08 17:26 <br>
     * 最后修改人:  <br>
     * 最后修改时间: 2020.04.08 17:26 <br>
     * 入参说明: 入参说明: [sname 姓名, stype 证件类型(0-身份证:目前只支持身份证号查询), sreason 查询原因(预留字段，暂填 0), sno 证件号码] <br>
     * 出参说明: java.util.Map<java.lang.String,java.lang.String> <br>
     * -----------------------------------------------------------------------------
     */
    @Override
    public Map<String, String> judicial(String sname, String stype, String sreason, String sno) {
        log.info("中互金司法数据查询入参, sname: {} , stype: {} , sreason: {} , sno: {} ", sname, stype, sreason, sno);
        if(StringUtils.isAnyBlank(sname, stype, sreason, sno)){
            return ImmutableMap.of("code","301","msg","参数不能为空","data", null);
        }
        // 构造body
        Map<String, String> bodyMap = new HashMap();
        bodyMap.put("sname", sname);
        bodyMap.put("stype",stype);
        bodyMap.put("sreason", sreason);
        bodyMap.put("sno", sno);
        Map<String, String> infoHeaderMap = getInfoHeaderMap(sname, stype, sreason, sno);

        log.info("中互金司法数据查询开始, sname: {} , stype: {} , sreason: {} , sno: {} , header: {} , body: {} ", sname, stype, sreason, sno, JSON.toJSONString(infoHeaderMap), JSON.toJSONString(bodyMap));

        ClientResponse resp = null;
        try {
            resp = reportJerseyClient.post(infoHeaderMap, bodyMap, reportConfig.getInfoPeopleUrl());

            if (resp!=null){
                String result = resp.getEntity(String.class);
                int status = resp.getStatus();
                log.info("中互金司法数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , result: {} , status: {} ", sname, stype, sreason, sno, result, status);

                if (resp.getStatus() == 200) {
                    return ImmutableMap.of("code","200","msg","查询成功","data", result);
                }else {
                    log.info("中互金司法数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , 调用失败: {} ", sname, stype, sreason, sno, JSON.toJSONString(resp));
                    return ImmutableMap.of("code","999","msg","查询接口失败status="+status,"data", result);
                }
            }else{
                log.info("中互金司法数据查询结束, sname: {} , stype: {} , sreason: {} , sno: {} , result is Null.", sname, stype, sreason, sno);
                return ImmutableMap.of("code","500","msg","中互金司法数据查询接口返回结果为空");
            }
        }catch (Exception e){
            log.error("中互金司法数据查询异常, sname: {} , stype: {} , sreason: {} , sno: {} ", sname, stype, sreason, sno, e);
            return ImmutableMap.of("code","500","msg","调用中互金司法数据接口请求异常");
        }
    }

    /**
     * -----------------------------------------------------------------------------<br>
     * 描  述: 查看最大查询量的耗用情况、查询状态 <br>
     * 创建人: yanhui.Hao <br>
     * 创建时间: 2020.04.08 17:26 <br>
     * 最后修改人:  <br>
     * 最后修改时间: 2020.04.08 17:26 <br>
     * 入参说明: 入参说明: [] <br>
     * 出参说明: java.util.Map<java.lang.String,java.lang.String> <br>
     * -----------------------------------------------------------------------------
     */
    @Override
    public Map<String, String> querycontral() {
        // 构造body
        Map<String, String> headerMap = new HashMap();
        headerMap.put("sbankcode", reportConfig.getCode());
        Map bodyMap = new HashMap<String, String>();

        log.info("中互金耗用查询量查询开始, headerMap: {} ", JSON.toJSONString(headerMap));
        ClientResponse resp = null;
        try {
            resp = reportJerseyClient.post(headerMap, bodyMap, reportConfig.getInfoQueryContralUrl());
            if (resp!=null){
                String result = resp.getEntity(String.class);
                int status = resp.getStatus();
                log.info("中互金耗用查询量查询结束, headerMap: {} , result: {} , status: {} ", JSON.toJSONString(headerMap), result, status);

                if (resp.getStatus() == 200) {
                    return ImmutableMap.of("code","200","msg","查询成功","data", result);
                }else {
                    return ImmutableMap.of("code","999","msg","查询接口失败status="+status, "data", result);
                }
            }else{
                log.info("中互金耗用查询量查询结束, headerMap: {} , result is Null.", JSON.toJSONString(headerMap));
                return ImmutableMap.of("code","500","msg","中互金耗用查询量查询返回结果为空");
            }
        }catch (Exception e){
            log.error("中互金耗用查询量查询异常, headerMap: {} ", JSON.toJSONString(headerMap), e);
            return ImmutableMap.of("code","500","msg","调用中互金耗用查询量查询请求异常");
        }
    }


    public Map<ErrorReson, List<String>> classify(List<String> data) {
        Map<ErrorReson, List<String>> rsult = new HashMap<>();
        for (String datum : data) {
            String[] split = datum.split("\\|", 0);
            if (split[1].equals(ErrorReson.E032.getErrorCode())) {
                List<String> stringList = rsult.get(ErrorReson.E032);
                if (stringList == null) {
                    stringList = new ArrayList<>();
                }
                stringList.add(split[2]);
                rsult.put(ErrorReson.E032, stringList);
            }
        }
        return rsult;
    }

    public void reReportClassify(Map<ErrorReson, List<String>> map) {
        for (ErrorReson errorReson : map.keySet()) {
            List<List<String>> arr = new ArrayList<>();
            switch (errorReson) {
                case E032:
                    List<String> stringList = map.get(errorReson);
                    Set<String> set = new HashSet<>(stringList);
                    Iterator<String> iterator = set.iterator();
                    List<String> line = null;
                    while (iterator.hasNext()) {
                        String next = iterator.next();
                        int index = -1;
                        for (int i = 0; i < arr.size(); i++) {
                            boolean flg = false;
                            List<String> strings = arr.get(i);
                            for (String string : strings) {
                                if (isEque(string, next)) {
                                    flg = true;
                                    break;
                                }
                            }
                            if (!flg) {
                                index = i;
                                break;
                            }
                        }
                        //刚开始的情况
                        if (arr.size() == 0) {
                            line = new ArrayList<>();
                            arr.add(line);
                            index = 0;
                        } else {
                            //有数据的情况 index 没有改变 就新增一个
                            if (index == -1) {
                                line = new ArrayList<>();
                                arr.add(line);
                                index = arr.size() - 1;
                            }

                        }
                        arr.get(index).add(next);
                    }
                    break;

            }
            for (int j = 0; j < arr.size(); j++) {
                List<String> stringList = arr.get(j);
                //重新報送從0004開始
                creatZipFileByString(stringList, InfoType.CREDIT_FINANCING_INFO, null,j+4);
            }


        }
    }

    public boolean isEque(String str1, String str2) {
        return (Utils.getStr(str1).equals(Utils.getStr(str2)) && Utils.getStr(str1, 10).equals(Utils.getStr(str2, 10)));
    }

    public Map<String, List<Map<String, Object>>> getDataSync(Map<String, String> param) {
        Map<String, List<Map<String, Object>>> result = new HashMap<>();
        CountDownLatch countDownLatch = new CountDownLatch(3);
        List<Map<String, Object>> loanDayData = new ArrayList<>();
        List<Map<String, Object>> repDayData = new ArrayList<>();
        List<Map<String, Object>> oveDayData = new ArrayList<>();
        ExecutorService instance = ThreadPoolUtil.getInstance();
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> dayLoanData = dataMapper.getLoanDayData(param);
                synchronized (loanDayData) {
                    if (dayLoanData != null && dayLoanData.size() > 0) {
                        loanDayData.addAll(dayLoanData);
                    }
                }

                countDownLatch.countDown();
            }
        });
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> repayDayData = dataMapper.getRepayDayData(param);
                synchronized (repDayData) {
                    if (repayDayData != null && repayDayData.size() > 0) {
                        repDayData.addAll(repayDayData);
                    }
                }
                countDownLatch.countDown();
            }
        });
        instance.submit(new Runnable() {
            @Override
            public void run() {
                List<Map<String, Object>> overDayData = dataMapper.getOverDayData(param);
                synchronized (oveDayData) {
                    if (overDayData != null && overDayData.size() > 0) {
                        oveDayData.addAll(overDayData);
                    }
                }
                countDownLatch.countDown();
            }
        });
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        instance.shutdown();
        result.put("1", loanDayData);
        result.put("2", repDayData);
        result.put("3", oveDayData);
        return result;
    }

    List<Map<String, Object>> getMonthData(Map<String, String> param) {
        return dataMapper.getOverMonthData(param);
    }


    /**
     * 格式化数据
     *
     * @param dataList
     * @param isDesensitization
     * @return
     */
    public String formatData(List<Map<String, Object>> dataList, boolean isDesensitization) throws QgException {
        StringBuffer sbf = new StringBuffer();
        try {
            int size = dataList.size();
            for (int j = 0; j < size; j++) {
                Map<String, Object> objectMap = dataList.get(j);
                for (int i = 0; i < KEYS.length; i++) {
                    Object value = objectMap.get(KEYS[i]);
                    if (i == NAMEINDEX) {
                        if (isDesensitization) {
                            value = desensitizationName(value.toString());
                        }
                    }
                    if (i == IDCARDEINDEX) {
                        if (String.valueOf(objectMap.get(KEYS[CERTIFICATETYPEINDEX])).equals("0")) {//身份证的情况下保证最后一位是大写
                            value = utils.manageIdCard(value.toString());
                            //如果需要脱敏
                            if (isDesensitization) {
                                value = desensitizationICardNo(value.toString());
                            }
                        }
                    }
                    if (i == CODEINDEX) {
                        value = reportConfig.getCode();//组织结构代码
                    }

//                    for (Integer index : DATETYPEINDEXS) {
//                        if (i == index) {
//                            value = DateUtil.formatterDate(value, DateUtil.YYYYMMDD);
//                        }
//                    }
                    for (Integer moneyindex : MONEYINDEX) {
                        if (i == moneyindex) {
                            value = utils.getInt(value);
                        }
                    }

                    sbf.append(value);
                    if (i < KEYS.length - 1) {
                        sbf.append(SPLIT);
                    }

                }
                if (j < size - 1) {
                    sbf.append(NEWLINE);
                }
            }

        } catch (Exception e) {
            log.error("数据格式处理异常e: {} ", e);
            throw new QgException("数据格式处理异常", e);
        }
        return sbf.toString();
    }

    /**
     * 脱敏姓名
     *
     * @param oldName
     * @return
     */
    public String desensitizationName(String oldName) {
        StringBuffer sb = new StringBuffer(oldName);
        return sb.replace(1, oldName.length(), NAMES[utils.random(NAMES.length)]).toString();
    }

    /**
     * 脱敏身份证件号
     *
     * @param oldNo
     * @return
     */
    public String desensitizationICardNo(String oldNo) {
        StringBuffer sb = new StringBuffer(oldNo);
        String pre = sb.substring(0, 2);
        String nepre = "";
        do {
            nepre = IDCARDNOPRE[utils.random(IDCARDNOPRE.length)].toString();
        } while (nepre.equals(pre));
        return sb.replace(0, 2, nepre).toString();
    }

    /**
     * 根据信息类型和业务类型生成zip包的前缀名字
     *
     * @param infoType
     * @param businessType
     * @return
     */
    public String getPackageName(String data, InfoType infoType, BusinessType businessType, String code) {
        StringBuffer sb = new StringBuffer();
        sb.append(reportConfig.getCode())
                .append(data == null ? DateUtil.getCurrentServerDate(DateUtil.YYYYMMDDHHMM) : data)
                .append(infoType.getCode())
                .append(code == null ? businessType.getCode() : code);
        return sb.toString();
    }

    private Map<String, String> getHeaderMap(Map<String, String> map,
                                             String filename) {
        //
        String scode = utils.string2Md5(map.get("sdata"));
        // 构造签名 用于身份验证
        // .append(sbankcode).append(sdatacode).append(scode).append("cfcc");
        String preparedSign = reportConfig.getCode() //sbankcode
                + filename //sdatacode
                + scode //scode
                + reportConfig.getInterFaceKey(); //key
        String sign = utils.sha256(preparedSign);
        sign = sign.trim().replace("\n", "").replace("\r", "");
        // 构造header
		/*
		 * .header("sbankcode", sbankcode)
    					.header("sdatacode", sdatacode)
    					.header("scode", scode)
    					.header("sign",sign)
		 */
        Map<String, String> headerMap = new HashMap<String, String>();
        headerMap.put("sbankcode", reportConfig.getCode());
        headerMap.put("sdatacode", filename);
        headerMap.put("scode", scode);
        headerMap.put("sign", sign);
        return headerMap;

    }

    private Map<String, String> getHeaderMap(Map<String, String> map) {
        String scode = utils.getRandomNumber(10); //10位随机数
        // 构造签名 用于身份验证
        // sbankcode + scode + sdatacode + "cfcc";
        String preparedSign = reportConfig.getCode() // sbankcode
                + scode //scode
                + map.get("sdatacode")  //sdatacode
                + reportConfig.getInterFaceKey();  //key
        String sign = utils.sha256(preparedSign);
        // 构造header
        // header("sbankcode", sbankcode).header("scode", scode).header("sign", sign).header("sdatacode", sdatacode)
        Map<String, String> headerMap = new HashMap<String, String>();
        headerMap.put("sbankcode", reportConfig.getCode());
        headerMap.put("scode", scode);
        headerMap.put("sign", sign);
        headerMap.put("sdatacode", map.get("sdatacode"));
        return headerMap;

    }
    //查询共享信息
    private  Map<String, String> getInfoHeaderMap(String sname ,String stype,String sreason,String sno) {
        //
        String scode = utils.getRandomNumber(10); //10位随机数
        // 构造签名 用于身份验证
        String preparedSign = reportConfig.getCode() // sbankcode
                + scode
                + sname
                + stype
                + sreason
                + sno
                + reportConfig.getInterFaceKey();  //key
        String sign = utils.sha256(preparedSign);
        // 构造header
        Map<String, String> headerMap = new HashMap<String, String>();
        headerMap.put("sbankcode", reportConfig.getCode());
        headerMap.put("scode", scode);
        headerMap.put("sign", sign);
        return headerMap;

    }

    /**
     * 获取反馈文件名
     * <p>
     * 反馈文件命名规则：
     * a.文件名长度为 28 位；
     * b.第 1～27 位为对应正常文件的文件包名；
     * c.第 28 位，标识位，填“1”表示错误反馈文件。
     * d.压缩加密前的文件文件名后缀为“txt”。
     *
     * @param normalFileName
     * @return
     */
    private String getFeedbackFileName(String normalFileName) {
        return normalFileName + "1";
    }

}
