/**
 * 
 */
package cn.quantgroup.report.config.aop;

import cn.quantgroup.report.cmpt.CommonAlarmCmpt;
import cn.quantgroup.report.cmpt.MonitorCmpt;
import cn.quantgroup.report.constant.TransactionCodeEnum;
import cn.quantgroup.report.enums.RequestUrlType;
import cn.quantgroup.report.response.GlobalResponse;
import cn.quantgroup.report.response.TransactionResult;
import cn.quantgroup.report.utils.StringUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 1.监控数据源
 * 2.统一结果处理
 */
@Slf4j
@Aspect
@Component
public class MonitorAspect {

	protected static ExecutorService dbLogPool = new ThreadPoolExecutor(20,50,200, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());

	//@Autowired
	//private ThirdpartyApiMonitorService thirdpartyApiMonitorService;

	@Autowired
	private CommonAlarmCmpt iMonitorAlarmService;

	@Autowired
	protected MonitorCmpt monitorCmpt;

    @Pointcut("@annotation(cn.quantgroup.report.config.aop.Monitor)")
    private void userQryPointCut() {
    }

    @Around("userQryPointCut()")
    public Object process(ProceedingJoinPoint point) throws Throwable {
    	log.info("enter MonitorAspect point : {}", point);
    	long begin = System.currentTimeMillis();
		Object result = point.proceed();
		long timeConsume = System.currentTimeMillis()- begin;
		dbLogPool.execute(()-> {
			try {
				Signature sig = point.getSignature();
				if (!(sig instanceof MethodSignature)) {
					log.warn("process msig :{}", sig);
					return;
				}
				MethodSignature msig = (MethodSignature) sig;
				Object target = point.getTarget();
				Method currentMethod = target.getClass().getMethod(msig.getName(), msig.getParameterTypes());
				Monitor annotation = currentMethod.getAnnotation(Monitor.class);
				MonitorType[] monitorTypes = annotation.value();

				if (!(result instanceof GlobalResponse)) return;

				GlobalResponse response = (GlobalResponse) result;
				log.info("handle response : {}" , StringUtil.shortPrintLog(response.toString()));
				Arrays.stream(monitorTypes).forEach(o -> {
					try {
						switch (o) {
							case FEE:
								handleFee(response);
								break;
							case EXCEPTION:
								handleException(response, point.getArgs(), timeConsume);
								break;
							case RESULT:
								handleResult(response);
								break;
							case HIT:
								handleHitStat(response);
								break;
						}
					}catch (Exception e){
						log.error("处理结果异常", e);
					}
				});
			}catch (Exception e){
				log.error("统一处理异常", e);
			}
		});
		return result;
    }

	/**
	 * 统计查询命中情况并告警
	 * @param response
	 */
	private void handleHitStat(GlobalResponse response) {
		log.info("handleHitStat start transactionId {}", response.getTransactionId());
		List<TransactionResult> transactionResults = response.getTransactionResult();
		transactionResults.stream().forEach(o -> {
			TransactionCodeEnum tansactionCode = o.getTansactionCode();
			if (TransactionCodeEnum.SuccessAndMiss.equals(tansactionCode)) {
				monitorCmpt.statisticHitRate(o.getUrlType().name(), 0);
			} else if(TransactionCodeEnum.SuccessAndHit.equals(tansactionCode)){
				monitorCmpt.statisticHitRate(o.getUrlType().name(), 1);
			}
		});

	}

	/**
	 * 处理查询结果，记录查询
	 */
	private void handleResult(GlobalResponse response) {

		String transactionId = response.getTransactionId();
		log.info("handleResult start transactionId {}", transactionId);

		List<TransactionResult> transactionResults = response.getTransactionResult();

		for (TransactionResult transactionResult : transactionResults) {
			RequestUrlType urlType = transactionResult.getUrlType();
			TransactionCodeEnum tansactionCode = transactionResult.getTansactionCode();
			if(StringUtils.isNotBlank(response.getUuid())) {
				//thirdpartyApiMonitorService.saveTransactionLog(urlType, transactionId, response.getUuid(), tansactionCode);
			} else if(StringUtils.isNotBlank(response.getIdentity())) {
				//thirdpartyApiMonitorService.saveTransactionLogByIdentifyNumber(urlType, transactionId, response.getIdentity(), tansactionCode);
			}else if(StringUtils.isNotBlank(response.getPhone())) {
				//thirdpartyApiMonitorService.saveTransactionLogByPhone(urlType, transactionId, response.getPhone(), tansactionCode);
			} else {
				//thirdpartyApiMonitorService.saveTransactionLog(urlType, transactionId, null, tansactionCode);
			}
		}
	}

	/**
	 * 处理异常，统一告警
	 */
	private void handleException(GlobalResponse response, Object[] args, long timeConsume) {
		log.info("handleception start transactionId {}, url_type, timeConsume {}", response.getTransactionId(), timeConsume, response.getTransactionResult());
		List<TransactionResult> transactionResult = response.getTransactionResult();
		transactionResult.stream().forEach(o -> {
			TransactionCodeEnum tansactionCode = o.getTansactionCode();
			if(TransactionCodeEnum.Fail.equals(tansactionCode)){
				int code = response.getCode();
				String error = response.getError();
				boolean isReadCache = (boolean) args[args.length-1];
				if(!isReadCache){
					iMonitorAlarmService.alarm("WARN","数据源告警 : " + o.getUrlType().getUrlName(),"code : " + code + "; msg : " + error + " args : " + Arrays.toString(args));
				}else{
					log.info("handleException is ReadCache: {} ", JSON.toJSONString(args));
				}
			} else {
				//超时告警
				if(timeConsume > 60000){
					iMonitorAlarmService.alarm("WARN","数据源调用超时 : " + o.getUrlType().getUrlName(),JSON.toJSONString(args)+"耗时 : " + timeConsume + "ms");
				}
			}
		});
	}

	/**
	 * record fee
	 */
	private void handleFee(GlobalResponse response) {
//		log.info("handleFee response transactionId: {}" , response.getTransactionId());
//		if(response.getIsBilling()){
//			List<TransactionResult> transactionResult = response.getTransactionResult();
//			transactionResult.stream().forEach(o -> {
//				thirdpartyApiMonitorService.recordCost(o.getUrlType(), o.getUrlType().getSourceName());
//				}
//			);
//		}
	}

}
