package cn.quantgroup.qgblservice.utils;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * -----------------------------------------------------------------------------<br>
 * 描述： <br>
 * 作者： Haoyanhui <br>
 * 时间：2019.09.09 20:53 <br>
 * 授权: (C) Copyright (c) 2017 <br>
 * 公司: 北京众信利民信息技术有限公司 <br>
 * -----------------------------------------------------------------------------
 */
public class ReadExcelUtils {
    private static Workbook wb;
    private static Sheet sheet;
    private static Row row;

    private static final String EXCEL_XLS = "xls";
    private static final String EXCEL_XLSX = "xlsx";

    private static final Logger logger = LoggerFactory.getLogger(ReadExcelUtils.class);


    /**
     * 读取表头
     *
     * @param inputStream inputStream
     * @param suffix      file suffix
     * @return map <index,value>
     */
    public static Map<Integer, String> readExcelTitle(InputStream inputStream, String suffix) {
        getWorkbook(inputStream, suffix);
        sheet = wb.getSheetAt(0);
        row = sheet.getRow(0);
        // 标题总列数
        int colNum = row.getPhysicalNumberOfCells();
        Map<Integer, String> map = new HashMap<>();
        for (int i = 0; i < colNum; i++) {
            map.put(i, row.getCell(i).getStringCellValue());
        }
        return map;
    }

    /**
     * 读取excel内容
     *
     * @param inputStream 文件
     * @return Map<行, Map < 下标, Object>>
     */
    public static Map<Integer, Map<Integer, String>> readExcelContent(InputStream inputStream, String suffix) {
        getWorkbook(inputStream, suffix);
        Map<Integer, Map<Integer, String>> content = new HashMap<>();
        sheet = wb.getSheetAt(0);
        // 得到总行数
        int rowNum = sheet.getLastRowNum();
        row = sheet.getRow(0);
        int colNum = row.getPhysicalNumberOfCells();
        // 正文内容应该从第二行开始,第一行为表头的标题
        for (int i = 1; i <= rowNum; i++) {
            row = sheet.getRow(i);
            int j = 0;
            Map<Integer, String> cellValue = new HashMap<>();
            while (j < colNum) {
                String obj = getCellFormatValue(row.getCell(j));
                cellValue.put(j, obj);
                j++;
            }
            content.put(i, cellValue);
        }
        return content;
    }

    private static String getCellFormatValue(Cell cell) {
        String cellValue = "";
        if (cell != null) {
            // 判断当前Cell的Type
            switch (cell.getCellType()) {
                // 如果当前Cell的Type为NUMERIC
                case Cell.CELL_TYPE_NUMERIC:
                case Cell.CELL_TYPE_FORMULA: {
                // 判断当前的cell是否为Date
                    if (org.apache.poi.ss.usermodel.DateUtil.isCellDateFormatted(cell)) {
                        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("YYYY-MM-dd HH:mm:ss");
                        Instant instant = cell.getDateCellValue().toInstant();
                        ZoneId zoneId = ZoneId.systemDefault();
                        LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, zoneId);
                        cellValue = dateTimeFormatter.format(localDateTime);
                    } else {
                        // 如果是纯数字
                        // 取得当前Cell的数值
                        cellValue = String.valueOf(cell.getNumericCellValue());
                    }
                    break;
                }
                // 如果当前Cell的Type为STRING
                case Cell.CELL_TYPE_STRING:
                    // 取得当前的Cell字符串
                    cellValue = cell.getRichStringCellValue().getString();
                    break;
                default:
                // 默认的Cell值
                    cellValue = "";
            }
        }
        return cellValue;
    }

    //HSSFWorkbook，XSSFWorkbook的区别：前者是解析出来excel 2007 以前版本的，后缀名为xls的，后者是解析excel 2007 版的，后缀名为xlsx
    private static void getWorkbook(InputStream inputStream, String suffix) {
        try {
            //2003
            if (EXCEL_XLS.equals(suffix)) {
                wb = new HSSFWorkbook(inputStream);
            //2007/2010
            } else if (EXCEL_XLSX.equals(suffix)) {
                wb = new XSSFWorkbook(inputStream);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                inputStream.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }


    private static Workbook getWorkbookByPath(String  filePath) throws IOException {
        Workbook wb = null;
        InputStream inputStream = null;
       try {
            inputStream = new FileInputStream(filePath);
            //Excel2003
            if (filePath.endsWith(".xls")) {
                wb = new HSSFWorkbook(inputStream);
                //Excel2007,Excel2010
            } else if (filePath.endsWith(".xlsx")) {
                wb = new XSSFWorkbook(inputStream);
            }
        } catch (IOException e) {
            //e.printStackTrace();
            throw e;
        } finally {
            try {
                if(inputStream!=null){
                    inputStream.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return wb;
    }


    /**
     * @描述
     * @参数  [path, haveTitle]
     * @返回值  java.util.Map<java.lang.String,java.lang.Object>
     *  <br> code: 0 成功,1 失败  ；  msg: 失败说明  ； result: List<String> 识别结果
     * @创建人  yanhui.Hao
     * @创建时间  2019.09.10
     */
    public static Map<String,Object> readExcel(String path, String titleNames, boolean haveTitle){
        Map<String,Object> readMap = new HashMap<String,Object>();
       try {
            List<String> readList = new ArrayList<String>();
            //HSSFWorkbook wb = new HSSFWorkbook(new FileInputStream(path));
            Workbook wb = getWorkbookByPath(path);
            if (wb==null) {
                readMap.put("code","1");
                readMap.put("msg","文件["+path+"]不是 Excel.");
                return readMap;
            }

            //HSSFSheet sheet = wb.getSheetAt(0);
            Sheet sheet = wb.getSheetAt(0);

            for (Iterator<Row> iter = (Iterator<Row>) sheet.rowIterator(); iter.hasNext();) {
                Row row = iter.next();
                // 获得当前行数
                int rowNum =row.getRowNum();
                if(rowNum==0){  //第一行标题
                    if(haveTitle){
                        StringBuffer titles = new StringBuffer();
                        for (Iterator<Cell> iter1 = (Iterator<Cell>) row.cellIterator(); iter1.hasNext();) {
                            Cell cell1 = iter1.next();
                            int thisColNum = cell1.getColumnIndex();
                            String content = getCellFormatValue(cell1);
                            if(thisColNum==0){
                                titles.append(content.trim());
                            }else{
                                titles.append(",");
                                titles.append(content.trim());
                            }
                        }
                        //if(titles.toString().indexOf(titleNames)!=-1){
                        if(titles.toString().indexOf("idcard,name,mobile,dateBack")==-1){
                            readMap.put("code","1");
                            readMap.put("msg","Excel 表格的列标题顺序必须是  [idcard,name,mobile,dateBack...other...]");
                            return readMap;
                        }
                        continue;
                    }
                }

                StringBuffer context = new StringBuffer();
                for (Iterator<Cell> iter2 = (Iterator<Cell>) row.cellIterator(); iter2.hasNext();) {
                    Cell cell = iter2.next();
                    // 获得当前列数
                    int col = cell.getColumnIndex();
                    //String content = cell.getStringCellValue();// 除非是sring类型，否则这样迭代读取会有错误
                    String content = getCellFormatValue(cell);
                    if(col==0){
                        context.append(content);
                    }else{
                        context.append("|");
                        context.append(content);
                    }
                }
                readList.add(context.toString());
            }
            readMap.put("code","0");
            readMap.put("msg","ok");
            readMap.put("result",readList);
        } catch (IOException e) {
            //e.printStackTrace();
            //System.err.println("readExcel() IOException,"+e.getMessage());
            logger.error("readExcel() IOException,"+e.getMessage(),e);
            readMap.put("code","1");
            readMap.put("msg",e.getMessage());
        }
        return readMap;
    }

    public static void main(String[] args) {
        //code: 0 成功,1 失败  ；  msg: 失败说明  ； result: List<String> 识别结果
        Map<String, Object> readMap = ReadExcelUtils.readExcel("D:\\JavaTeam\\shmf.xls", "idcard,name,mobile,dateBack", false);
        if(readMap!=null && readMap.get("code").toString().equals("0") && readMap.get("result")!=null) {
            List<String> readList = (List<String>) readMap.get("result");
            for(String tmp:readList){
                System.out.println(tmp);
            }
        }
        System.out.println(readMap.toString());

    }

}
