package cn.quantgroup.report.service.baihang;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.json.JSONUtil;
import cn.quantgroup.report.domain.baihang.*;
import cn.quantgroup.report.error.QGException;
import cn.quantgroup.report.mapper.baihang.ApplyLoanInfoMapper;
import cn.quantgroup.report.mapper.baihang.LoanInfoMapper;
import cn.quantgroup.report.mapper.baihang.RepaymentLoanInfoMapper;
import cn.quantgroup.report.mapper.master.ApplyLoanInfoZhuDaiMapper;
import cn.quantgroup.report.mapper.master.ApplyLoanInfoZhuDaiRefuseMapper;
import cn.quantgroup.report.mapper.master.LoanInfoDbMapper;
import cn.quantgroup.report.mapper.master.RepaymentLoanInfoDbMapper;
import cn.quantgroup.report.service.baihang.client.BhFileCreditApiClient;
import cn.quantgroup.report.service.baihang.constant.Constant;
import cn.quantgroup.report.service.baihang.request.FileUploadRequest;
import cn.quantgroup.report.service.baihang.response.FileUploadResponse;
import cn.quantgroup.report.service.baihang.util.TuoMinUtils;
import cn.quantgroup.report.service.http.IHttpService;
import cn.quantgroup.report.utils.DateUtils;
import cn.quantgroup.report.utils.SftpUtil;
import cn.quantgroup.report.utils.dingtalk.DingTalk;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import static cn.quantgroup.report.error.QGExceptionType.COMMON_ILLEGAL_PARAM_TOAST;

/**
 * @author jian.zheng
 * @date 2021/6/4 - 15:23
 **/
@Slf4j
@Service
public class BaiHangFileReportService {


    @Value("${isDebug}")
    private Boolean isDebug;
    @Autowired
    public ApplyLoanInfoMapper applyLoanInfoMapper;
    @Autowired
    public IHttpService iHttpService;
    @Autowired
    public LoanInfoMapper loanInfoMapper;
    @Autowired
    public RepaymentLoanInfoMapper repaymentLoanInfoMapper;
    @Autowired
    public LoanInfoDbMapper loanInfoDbMapper;
    @Autowired
    public RepaymentLoanInfoDbMapper repaymentLoanInfoDbMapper;
    @Autowired
    public ApplyLoanInfoZhuDaiMapper applyLoanInfoZhuDaiMapper;
    @Autowired
    public ApplyLoanInfoZhuDaiRefuseMapper applyLoanInfoZhuDaiRefuseMapper;

    private static final String HOST_PATH = "/home/quant_group/reportFile/";

//    private static final String HOST_PATH = "C:\\Users\\musterHunter\\Desktop\\TEST_REPORT\\";
    private static final Integer A1_INITIAL_CAPACITY = 5000;

    private static final Integer D2_INITIAL_CAPACITY = 5000;

    private static final Integer D3_INITIAL_CAPACITY = 5000;

    private static final Integer LISTS_PARTITION_SIZE = 5000;

    @Autowired
    private RedisTemplate<String,String> redisTemplate;

    @Value("${baiHang.ftp.username}")
    private String baiHangFtpUsername;

    @Value("${baiHang.ftp.password}")
    private String baiHangFtpPassword;

    @Value("${baiHang.ftp.host}")
    private String baiHangFtpHost;

    @Value("${baiHang.ftp.port}")
    private Integer baiHangFtpPort;

    @Value("${baiHang.ftp.dataDirectory}")
    private String baiHangFtpDataDirectory;

    @Autowired
    private DingTalk dingTalk;




    /**
     * 创建报送文件 通常用于存量报送
     * @param type 报送类型 A1:申请信息 D2:放款信息 D3:还款信息
     * @param start 报送区间的开始时间 格式:"2019-05-07T00:00:00"
     * @param end 报送区间的截止时间 格式:"2019-05-07T00:00:00"
     */
    @Async
    public void createReportFile(String type, String start, String end,int daySplitCount, String loanStartDateStr, String loanEndDateStr, String prefix) {
        try {
            if (!increment(Constant.QG_ZHU_DAI_CREATE_REPORT_FILE_LOCK_KEY)) {
                log.error("创建报送文件任务已经开始执行...请勿重复操作");
                return ;
            }
            redisTemplate.expire(Constant.QG_ZHU_DAI_CREATE_REPORT_FILE_LOCK_KEY, 15, TimeUnit.MINUTES);
            if (StringUtils.isAnyBlank(type, start, end)) {
                throw new QGException(COMMON_ILLEGAL_PARAM_TOAST,"存在为空参数");
            }
            String encryptFileName = null;
            List<String> encryptFileNameList = new LinkedList();
            daySplitCount = Objects.isNull(daySplitCount) ? 10 : daySplitCount;
            String dateFormat = "yyyy-MM-dd";
            Date stepStartDate = DateUtils.parseDate(dateFormat,start);
            Date stepEndDate = org.apache.commons.lang3.time.DateUtils.addDays(stepStartDate,daySplitCount);
            // D3 还款记录 防止漏期
            Date endDate = "D3R".equalsIgnoreCase(type) ? org.apache.commons.lang3.time.DateUtils.addDays(new Date(),1) : DateUtils.parseDate(dateFormat,end);
            Date loanStartDate = StringUtils.isBlank(loanStartDateStr) ? new Date() : DateUtils.parseDate(dateFormat,loanStartDateStr);
            Date loanEndDate = StringUtils.isBlank(loanEndDateStr) ? new Date() : DateUtils.parseDate(dateFormat,loanEndDateStr);
            Stopwatch stopwatch = Stopwatch.createStarted();
            while (stepEndDate.getTime() <= endDate.getTime()){
                encryptFileName = createReportFile(type,stepStartDate,stepEndDate,loanStartDate,loanEndDate,prefix);
                if (StringUtils.isNotBlank(encryptFileName)) {
                    encryptFileNameList.add(encryptFileName);
                }
                stepStartDate = new Date(stepEndDate.getTime());
                stepEndDate = org.apache.commons.lang3.time.DateUtils.addDays(stepEndDate,daySplitCount);
            }
            log.info("当前 {} -- {}--{} 任务执行结束 总耗时 : {}",type,start,end,stopwatch.elapsed(TimeUnit.MILLISECONDS));
            if (stepEndDate.getTime() > endDate.getTime()){
                encryptFileName = createReportFile(type,stepStartDate,endDate,loanStartDate,loanEndDate,prefix);
                if (StringUtils.isNotBlank(encryptFileName)) {
                    encryptFileNameList.add(encryptFileName);
                }
            }
            if (!CollectionUtils.isEmpty(encryptFileNameList)) {
                File file = new File(HOST_PATH + type + "_" + DateUtils.formatDate(new Date(),"yyyyMMdd") + ".txt");
                FileUtils.writeLines(file,encryptFileNameList,Boolean.TRUE);
            }
        } catch (Exception e) {
            log.error("创建报送文件异常!",e);
        } finally {
            deleteReportRedisKey(Constant.QG_ZHU_DAI_CREATE_REPORT_FILE_LOCK_KEY);
        }
    }

    private String createReportFile(String type, Date startDate, Date endDate, Date loanStartDate, Date loanEndDate, String prefix) {
        String encryptFileName = null;
        Stopwatch stopwatch = Stopwatch.createStarted();
        String starTime = DateUtils.formatDate(startDate,"yyyy-MM-dd HH:mm:ss");
        String endTime = DateUtils.formatDate(endDate,"yyyy-MM-dd HH:mm:ss");
        String loanStartTime = DateUtils.formatDate(loanStartDate,"yyyy-MM-dd HH:mm:ss");
        String loanEndTime = DateUtils.formatDate(loanEndDate,"yyyy-MM-dd HH:mm:ss");
        if (!starTime.equals(endTime)) {
            log.info("开始生成报送文件{}----- {} -- {}",type,starTime,endTime);
            if ("A1".equalsIgnoreCase(type)) {
                encryptFileName = reportA1(starTime,endTime);
            } else if ("A1Refuse".equalsIgnoreCase(type)) {
                encryptFileName = reportA1Refuse(starTime,endTime);
            } else if ("D2".equalsIgnoreCase(type)) {
                encryptFileName = reportD2(starTime,endTime,prefix);
            } else if ("D3".equalsIgnoreCase(type)) {
               // encryptFileName = reportD3(starTime,endTime,prefix);
                throw new QGException(COMMON_ILLEGAL_PARAM_TOAST,"暂不支持该类型数据报送");
            } else if ("D3O".equalsIgnoreCase(type)) {
                encryptFileName = reportD3O(starTime,endTime,prefix);
            } else if ("D3R".equalsIgnoreCase(type)) {
                encryptFileName = reportD3R(starTime,endTime,loanStartTime,loanEndTime,prefix);
            } else if ("D3O30".equalsIgnoreCase(type)) {
                encryptFileName = reportD3OFor30Day(starTime,endTime,prefix);
            } else {
                throw new QGException(COMMON_ILLEGAL_PARAM_TOAST,"报送类型不正确");
            }
            log.info("生成报送文件结束{}----- {} -- {},耗时 {}",type,starTime,endTime,stopwatch.elapsed(TimeUnit.MILLISECONDS));
            dingTalk.alarmDingTalk("Info","百行"+type+"征信数据报送","生成报送文件结束,开始时间:"+starTime+"结束时间:"+endTime);
        }
        return encryptFileName;
    }

    /**
     * 贷款申请信息（A1）
     * @param starTime 开始时间
     * @param endTime 截止时间
     */
    private String reportA1(String starTime, String endTime){
        String encryptFileName = null;
        try {
            Stopwatch queryWatch1 = Stopwatch.createStarted();
            BaiHangTimeRecord timeRecord = BaiHangTimeRecord.builder().startTime(starTime).endTime(endTime).build();
            List<ApplyLoanInfoZhuDai> applyLoanInfos = applyLoanInfoMapper.findRealTimeApplyLoanZhuDai(timeRecord);
            log.info("量化派助贷TO百行报送(A1)-非联合贷申请查询完成, 大小: {} , 耗时: {} ", (applyLoanInfos != null ? applyLoanInfos.size() : 0), (queryWatch1.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            if (applyLoanInfos == null) {
                applyLoanInfos = new ArrayList<ApplyLoanInfoZhuDai>();
            }
            AtomicInteger atomicInteger = new AtomicInteger();
            Stopwatch sendWatch = Stopwatch.createStarted();
            List<String> reportList = new ArrayList<>();
            reportList.add("#applyInfo");
            String reqId_log = "";
            List<ApplyLoanInfoZhuDai> zhuDaiLogList = null;
            for (int i = 0; i < applyLoanInfos.size(); i++) {
                try {
                    ApplyLoanInfoZhuDai applyLoanInfo = applyLoanInfos.get(i);
                    reqId_log = applyLoanInfo.getReqID();
                    zhuDaiLogList = applyLoanInfoZhuDaiMapper.findApplyLoanInfoZhuDaiLog(applyLoanInfo.getReqID());
                    if (CollectionUtils.isEmpty(zhuDaiLogList)) {
                        try {
                            ApplyLoanInfoZhuDai record = new ApplyLoanInfoZhuDai();
                            BeanUtils.copyProperties(applyLoanInfo, record);
                            record.setRecordId(UUID.randomUUID().toString().replaceAll("-", ""));
                            applyLoanInfoZhuDaiMapper.saveApplyLoanInfoZhuDaiLog(record);
                        } catch (Exception e) {
                            log.error("量化派助贷TO百行报送(A1)-贷款申请信息保存记录失败", e);
                        }
                        //邮箱有错误的就不传
                        if (!TuoMinUtils.checkEmail(applyLoanInfo.getEmailAddress())) {
                            applyLoanInfo.setEmailAddress(null);
                        }
                        applyLoanInfo.setName(sensitiveFilter(applyLoanInfo.getName()));
                        applyLoanInfo.setMobile(sensitiveFilter(applyLoanInfo.getMobile()));
                        applyLoanInfo.setPid(sensitiveFilter(applyLoanInfo.getPid()));
                        reportList.add(JSON.toJSONString(applyLoanInfo));
                        atomicInteger.getAndIncrement();
                    } else {
                        log.error("量化派助贷TO百行报送(A1)-贷款申请信息重复报送, applyLoanInfo: {} ", JSON.toJSONString(applyLoanInfo));
                    }
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(A1)-贷款申请信息异常, reqId_log: {} ", reqId_log, e);
                }
            }
            File file = new File(HOST_PATH+"量子数科科技有限公司_A1_"+fileNameReplaceAll(starTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(A1)-贷款申请信息为空,加密文件不生成!starTime={},endTime={}",starTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(A1)-贷款申请信息完成, 实际大小: {} , 报送成功大小: {} , 耗时: {} ", applyLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(A1)-异常", e);
        }
        return encryptFileName;
    }

    private String reportA1Refuse(String starTime, String endTime){
        String encryptFileName = null;
        List<ApplyLoanInfoZhuDai> recordList = new ArrayList<>(A1_INITIAL_CAPACITY);
        List<ApplyLoanInfoZhuDai> zhuDaiLogList = null,applyLoanInfos = null;
        List<String> reportList = new ArrayList<>();
        try {
            Stopwatch queryWatch1 = Stopwatch.createStarted();
            BaiHangTimeRecord timeRecord = BaiHangTimeRecord.builder().startTime(starTime).endTime(endTime).build();
            applyLoanInfos = applyLoanInfoMapper.findApplyInfoOfRefuse(timeRecord);
            log.info("量化派助贷TO百行报送(A1Refuse)-非联合贷申请查询完成, 大小: {} , 耗时: {} ", (applyLoanInfos != null ? applyLoanInfos.size() : 0), (queryWatch1.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            if (applyLoanInfos == null) {
                applyLoanInfos = new ArrayList<ApplyLoanInfoZhuDai>();
            }
            AtomicInteger atomicInteger = new AtomicInteger();
            Stopwatch sendWatch = Stopwatch.createStarted();
            reportList.add("#applyInfo");
            String reqId_log = "";
            ApplyLoanInfoZhuDai record = null;
            Stopwatch checkQueryWatch = null;
            for (int i = 0; i < applyLoanInfos.size(); i++) {
                try {
                    ApplyLoanInfoZhuDai applyLoanInfo = applyLoanInfos.get(i);
                    log.info("量化派助贷TO百行报送(A1Refuse)-正在处理 {} 数据... i={}",applyLoanInfo.getApplyId(),i);
                    reqId_log = applyLoanInfo.getReqID();
                    checkQueryWatch = Stopwatch.createStarted();
                    zhuDaiLogList = applyLoanInfoZhuDaiMapper.findByApplyId(applyLoanInfo.getApplyId());
                    log.info("量化派助贷TO百行报送(A1Refuse)-校验查询完成, 耗时: {} ",checkQueryWatch.stop().elapsed(TimeUnit.MILLISECONDS));
                    if (CollectionUtils.isEmpty(zhuDaiLogList)) {
                        record = new ApplyLoanInfoZhuDai();
                        BeanUtils.copyProperties(applyLoanInfo, record);
                        record.setRecordId(UUID.randomUUID().toString().replaceAll("-", ""));
                        recordList.add(record);
                        if (recordList.size() >= A1_INITIAL_CAPACITY) {
                            saveApplyLoanInfoZhuDai(recordList,starTime,endTime);
                            clear(recordList);
                            recordList = new ArrayList<>(A1_INITIAL_CAPACITY);
                        }
                        //邮箱有错误的就不传
                        if (!TuoMinUtils.checkEmail(applyLoanInfo.getEmailAddress())) {
                            applyLoanInfo.setEmailAddress(null);
                        }
                        applyLoanInfo.setName(sensitiveFilter(applyLoanInfo.getName()));
                        applyLoanInfo.setMobile(sensitiveFilter(applyLoanInfo.getMobile()));
                        applyLoanInfo.setPid(sensitiveFilter(applyLoanInfo.getPid()));
                        reportList.add(JSON.toJSONString(applyLoanInfo));
                        atomicInteger.getAndIncrement();
                    } else {
                        log.error("量化派助贷TO百行报送(A1Refuse)-贷款申请信息重复报送, applyLoanInfo: {} ", JSON.toJSONString(applyLoanInfo));
                    }
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(A1Refuse)-贷款申请信息异常, reqId_log: {} ", reqId_log, e);
                }
            }
            if (!CollectionUtils.isEmpty(recordList) && recordList.size() > 0) {
                saveApplyLoanInfoZhuDai(recordList,starTime,endTime);
            }
            File file = new File(HOST_PATH+"量子数科科技有限公司_A1Re_"+fileNameReplaceAll(starTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(A1Refuse)-贷款申请信息为空,加密文件不生成!starTime={},endTime={}",starTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(A1Refuse)-贷款申请信息完成, 实际大小: {} , 报送成功大小: {} , 耗时: {} ", applyLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(A1Refuse)-异常", e);
        } finally {
            clear(applyLoanInfos,reportList,recordList);
        }
        return encryptFileName;
    }

    /**
     * 非循环贷款账户数据信息（D2）
     * @param starTime 开始时间
     * @param endTime 截至时间
     */
    private String reportD2(String starTime, String endTime, String prefix) {
        String encryptFileName = null;
        List<LoanInfoZhuDai> loanInfozdList = null,recordList = new ArrayList<>(D2_INITIAL_CAPACITY);
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            loanInfozdList = loanInfoMapper.queryStockLoanInfoZhuDai(BaiHangTimeRecord.builder().startTime(starTime).endTime(endTime).build());
            int listSize = loanInfozdList != null ? loanInfozdList.size() : 0;
            log.info("百行-助贷放款数据查询结束, listSize : {} , startTime: {} , endTime: {} , 耗时 : {} ", listSize, starTime, endTime, stopwatch.elapsed(TimeUnit.MILLISECONDS));
            AtomicInteger atomicInteger = new AtomicInteger();
            Stopwatch sendWatch = Stopwatch.createStarted();
            List<String> reportList = new ArrayList<>();
            reportList.add("#singleLoanAccountInfo");
            for (int i = 0; i < loanInfozdList.size(); i++) {
                LoanInfoZhuDai loanInfo = loanInfozdList.get(i);
                String id = "";
                try {
                    String loanInfoReqId = loanInfo.getReqID();
                    List<String> targetRepaymentDayList = loanInfoMapper.findTargetRepaymentDayList(loanInfoReqId);
                    if (targetRepaymentDayList != null && targetRepaymentDayList.size() > 0) {
                        loanInfo.setTargetRepayDateList(String.join(",", targetRepaymentDayList));
                        log.info("比较还款总期数&账单日列表totalTerm=" + loanInfo.getTotalTerm() + ",targetRepaymentDayList size=" + targetRepaymentDayList.size());
                    } else {
                        log.warn("比较还款总期数&账单日列表totalTerm=" + loanInfo.getTotalTerm() + ",targetRepaymentDayList is null.");
                    }
                    UUID loanInfoId = UUID.randomUUID();
                    id = loanInfoId.toString().replaceAll("-", "");
                    LoanInfoZhuDai record = new LoanInfoZhuDai();
                    BeanUtils.copyProperties(loanInfo, record);
                    record.setRecordId(id);
                    recordList.add(record);
                    String jsonStr = JSONObject.toJSONString(loanInfo);
                    LoanInfoZhuDaiVo loanInfoVo = JSONObject.parseObject(jsonStr, LoanInfoZhuDaiVo.class);
                    FinTechAgencyBusinessZhuDai finTechAgencyBusiness = JSONObject.parseObject(jsonStr, FinTechAgencyBusinessZhuDai.class);
                    loanInfoVo.setFinTechAgencyBusiness(finTechAgencyBusiness);
                    loanInfoVo.setReqID(id);
                    loanInfoVo.setName(sensitiveFilter(loanInfo.getName()));
                    loanInfoVo.setPid(sensitiveFilter(loanInfo.getPid()));
                    loanInfoVo.setMobile(sensitiveFilter(loanInfo.getMobile()));
                    try {
                        if (DateUtils.parseDate("yyyy-MM-dd'T'HH:mm:ss",loanInfoVo.getIssueDate()).getTime() < DateUtils.parseDate("yyyy-MM-dd'T'HH:mm:ss",loanInfoVo.getAccountOpenDate()).getTime()){
                            loanInfoVo.setAccountOpenDate(loanInfoVo.getIssueDate());
                            loanInfoVo.setApplyDate(loanInfoVo.getIssueDate());
                        }
                    } catch (Exception e) {
                        log.error("百行-助贷放款数据组装异常,loan_application_history_id: {}",loanInfoReqId,e);
                        //处理个别的老数据 loan_application_manifest_history表没数据的情况
                        List<RepairDataOfLoanAccountExt> repairDataList = loanInfoMapper.findRepairDataOfLoanAccountExt(loanInfoReqId);
                        if (!CollectionUtils.isEmpty(repairDataList)) {
                            loanInfoVo.setAccountOpenDate(repairDataList.get(0).getApplyTime());
                            loanInfoVo.setApplyDate(repairDataList.get(0).getApplyTime());
                            log.info("百行-助贷放款数据组装异常已修复,loan_application_history_id: {}",loanInfoReqId);
                        }
                    }
                    if (isDebug && StringUtils.isNotBlank(prefix)) {
                        loanInfoVo.setReqID(prefix+loanInfoVo.getReqID());
                        loanInfoVo.setLoanId(prefix+loanInfoVo.getLoanId());
                        finTechAgencyBusiness.setRelationID(prefix+finTechAgencyBusiness.getRelationID());
                        loanInfoVo.setFinTechAgencyBusiness(finTechAgencyBusiness);
                    }
                    reportList.add(JSON.toJSONString(loanInfoVo));
                    atomicInteger.getAndIncrement();
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送-放款信息异常, recordId: {} , loanId: {} ", id, loanInfo.getLoanId(), e);
                }
            }
            saveLoanInfoRecordLog(recordList,starTime,endTime);
            File file = new File(HOST_PATH+"量子数科科技有限公司_D2_"+fileNameReplaceAll(starTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送-放款信息为空,加密文件不生成!starTime={},endTime={}",starTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送-放款申请完成, 实际大小: {} , 报送成功大小: {} , 耗时: {} ", loanInfozdList.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送-异常", e);
        } finally {
            clear(loanInfozdList,recordList);
        }
        return encryptFileName;
    }

    /**
     * 非循环贷款贷后数据信息（D3）
     * @param startTime 开始时间
     * @param endTime 截止时间
     */
    private String reportD3( String startTime, String endTime, String prefix) {
        String encryptFileName = null;
        for (int j = 0; j < 2; j++) {
            List<String> reportList = new ArrayList<>();
            reportList.add("#singleLoanRepayInfo");
            Stopwatch sendWatch = Stopwatch.createStarted();
            try {
                List<RepaymentInfoZhuDai> repaymentLoanInfos = null;
                if (j == 0) {
                    Stopwatch stopwatch = Stopwatch.createStarted();
                    repaymentLoanInfos = repaymentLoanInfoMapper.queryStockRepayMentInfoZhuDai(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
                    log.info("百行-助贷还款查询结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (stopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
                } else {
                    Stopwatch stopwatch = Stopwatch.createStarted();
                    repaymentLoanInfos = repaymentLoanInfoMapper.queryStockRepayMentInfoOverdueZhuDai(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
                    log.info("百行-助贷还款查询逾期结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (stopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");

                }
                AtomicInteger atomicInteger = new AtomicInteger();
                for (int i = 0; i < repaymentLoanInfos.size(); i++) {
                    RepaymentInfoZhuDai repaymentLoanInfo = repaymentLoanInfos.get(i);
                    String id = "";
                    try {
                        RepaymentInfoZhuDai repaymentLoanInfo1 = repaymentLoanInfoDbMapper.findLastOne(BaiHangRepayment.builder().loanId(repaymentLoanInfo.getLoanId()).termNo(repaymentLoanInfo.getTermNo()).build());
                        if (j > 0 && Objects.nonNull(repaymentLoanInfo1)) {
                            log.info("量化派助贷TO百行报送-实时还款逾期跳过报送, startTime: {} , endTime: {} , bean: {} ", startTime, endTime, JSON.toJSONString(repaymentLoanInfo1));
                            continue;
                        }
                        id = UUID.randomUUID().toString().replaceAll("-", "");
                        try {
                            RepaymentInfoZhuDai record = new RepaymentInfoZhuDai();
                            BeanUtils.copyProperties(repaymentLoanInfo, record);
                            record.setRecordId(id);
                            repaymentLoanInfoDbMapper.saveRepaymentLoanInfoLog(record);
                        } catch (Exception e) {
                            log.error("量化派助贷TO百行报送-实时还款保存记录异常", e);
                        }
                        repaymentLoanInfo.setReqID(id);

                        repaymentLoanInfo.setName(sensitiveFilter(repaymentLoanInfo.getName()));
                        repaymentLoanInfo.setPid(sensitiveFilter(repaymentLoanInfo.getPid()));
                        repaymentLoanInfo.setMobile(sensitiveFilter(repaymentLoanInfo.getMobile()));
                        if (isDebug && StringUtils.isNotBlank(prefix)) {
                            repaymentLoanInfo.setReqID(prefix+repaymentLoanInfo.getReqID());
                            repaymentLoanInfo.setLoanId(prefix+repaymentLoanInfo.getLoanId());
                        }
                        atomicInteger.getAndIncrement();
                        reportList.add(JSON.toJSONString(repaymentLoanInfo));
                    } catch (Exception e) {
                        log.error("量化派助贷TO百行报送-还款信息异常, recordId: {} , loanId: {} ", id, repaymentLoanInfo.getLoanId(), e);
                    }
                }
                File file = new File(HOST_PATH+"量子数科科技有限公司_D3_"+endTime.split("T")[0].replaceAll("-","")+"_"+"0003"+".txt");
                FileUtils.writeLines(file,reportList);
                encryptFileName = createFile(file.getName());
                log.info("量化派助贷TO百行报送-还款申请完成 J: {} , 开始时间: {} , 结束时间: {} , 实际大小: {} , 报送成功大小: {} , 耗时: {} ", j, startTime, endTime, repaymentLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
            } catch (Exception e) {
                log.error("量化派助贷TO百行报送-异常 J: {} , 开始时间: {} , 结束时间: {} ", j, startTime, endTime, e);
            }
        }
        return encryptFileName;
    }

    /**
     * 根据传入的文件路径将加密后的文件生成到 {@link BaiHangFileReportService#HOST_PATH}中
     * @param filePath 需要加密的文件路径
     */
    private static String createFile(String filePath){
        String encryptFileName = null;
        log.info("创建加密文件,明文文件路径 {}",filePath);
        File host = new File(HOST_PATH);
        if (!host.exists()){
            host.mkdir();
        }
        //AES密钥（机构自行设置）
        String aesKey = "fc4c1cbca5a46840f60";
        FileUploadRequest req = new FileUploadRequest();
        //设置未作压缩加密前的原始数据文件路径
        req.setDataFile(filePath);
        //设置压缩加密后的密文文件输出路径(为空表示与原始数据文件同目录)
        req.setTargetFilePath(HOST_PATH);
        BhFileCreditApiClient client = new BhFileCreditApiClient();
        //初始化设置AES密钥和RSA公钥
        client.init(BaiHangZhuDaiService.getBaihangZDPublickey(),aesKey);
        //执行压缩加密操作
        FileUploadResponse response = client.execute(req);
        log.info("执行压缩加密操作出参{}",JSONObject.toJSONString(response));
        if(response.isSuccess){
            encryptFileName = response.getEncryptFileName();
            System.out.println("zip And encrypt success;fileName = "+response.getEncryptFilePath() + response.getEncryptFileName());
        }else{
            System.out.println("zip And encrypt fail;errorMessage = "+response.getErrorMessage());
        }
        return encryptFileName;
    }

    private String sensitiveFilter(String s){
        if (isDebug){
            if (StringUtils.isNumeric(s) || s.contains("X") || s.contains("x")){
                return s.substring(0,s.length()-4)+"0000";
            }else {
                return s.charAt(0)+"零";
            }
        }else {
            return s;
        }
    }

    private String fileNameReplaceAll(String str) {
        if (str.length()<=0) {
            return "";
        }
        return str.replaceAll("-","").replaceAll(":","").replaceAll(" ","");
    }

    public void encryptFile(String filePath){
        createFile(filePath);
    }

    /**
     * 非循环贷款贷后数据信息（D3）逾期记录
     * @param startTime 开始时间
     * @param endTime 截止时间
     */
    private String reportD3O(String startTime, String endTime, String prefix) {
        String encryptFileName = null;
        List<RepaymentInfoZhuDai> repaymentLoanInfos = null,recordList = new ArrayList<>(D3_INITIAL_CAPACITY);
        List<String> reportList = new ArrayList<>(D3_INITIAL_CAPACITY);
        reportList.add("#singleLoanRepayInfo");
        Stopwatch sendWatch = Stopwatch.createStarted();
        try {
            Stopwatch queryStopwatch = Stopwatch.createStarted();
            repaymentLoanInfos = repaymentLoanInfoMapper.queryOverdueRecordOfD3(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
            log.info("量化派助贷TO百行报送(D3)-逾期记录查询结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (queryStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            AtomicInteger atomicInteger = new AtomicInteger();
            String id = "";
            for (int i = 0; i < repaymentLoanInfos.size(); i++) {
                RepaymentInfoZhuDai repaymentLoanInfo = repaymentLoanInfos.get(i);
                try {
                    id = UUID.randomUUID().toString().replaceAll("-", "");
                    RepaymentInfoZhuDai record = new RepaymentInfoZhuDai();
                    BeanUtils.copyProperties(repaymentLoanInfo, record);
                    record.setRecordId(id);
                    recordList.add(record);
                    repaymentLoanInfo.setReqID(id);
                    repaymentLoanInfo.setName(sensitiveFilter(repaymentLoanInfo.getName()));
                    repaymentLoanInfo.setPid(sensitiveFilter(repaymentLoanInfo.getPid()));
                    repaymentLoanInfo.setMobile(sensitiveFilter(repaymentLoanInfo.getMobile()));
                    if (isDebug && StringUtils.isNotBlank(prefix)) {
                        repaymentLoanInfo.setReqID(prefix+repaymentLoanInfo.getReqID());
                        repaymentLoanInfo.setLoanId(prefix+repaymentLoanInfo.getLoanId());
                    }
                    atomicInteger.getAndIncrement();
                    reportList.add(JSON.toJSONString(repaymentLoanInfo));
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(D3)-逾期记录处理异常, recordId: {} , loanId: {} ", id, repaymentLoanInfo.getLoanId(), e);
                }
            }
            saveRepaymentLoanInfoLog(recordList,startTime,endTime);
            File file = new File(HOST_PATH+"量子数科科技有限公司_D3O_"+fileNameReplaceAll(startTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(D3)-逾期记录为空,加密文件不生成!starTime={},endTime={}",startTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(D3)-逾期记录完成,开始时间: {} , 结束时间: {} , 实际大小: {} , 报送成功大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(D3)-逾期记录处理异常,开始时间: {} , 结束时间: {} ", startTime, endTime, e);
        } finally {
            clear(repaymentLoanInfos,recordList,reportList);
        }
        return encryptFileName;
    }

    /**
     * 应还款日次日起，每30天的次日报送一次逾期记录，直至结清或转出
     */
    private String reportD3OFor30Day(String startTime, String endTime, String prefix) {
        String encryptFileName = null;
        List<LoanApplicationHistoryIdInfo> idInfos = null;
        LoanApplicationHistoryIdInfo idInfo = null;
        List<RepaymentInfoZhuDai> repaymentLoanInfos = new ArrayList<>(D3_INITIAL_CAPACITY),tempRepaymentLoanInfos = null,recordList = new ArrayList<>(D3_INITIAL_CAPACITY);
        Date stepEndDate = null;
        Integer step = null;
        String id = "";
        AtomicInteger atomicInteger = new AtomicInteger();
        List<String> reportList = new ArrayList<>(D3_INITIAL_CAPACITY);
        reportList.add("#singleLoanRepayInfo");
        Stopwatch sendWatch = Stopwatch.createStarted();
        try {
            Stopwatch queryStopwatch = Stopwatch.createStarted();
            idInfos = repaymentLoanInfoMapper.queryLoanApplicationHistoryIdInfo(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
            log.info("量化派助贷TO百行报送(D3)-每30天逾期记录总数查询结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, CollectionUtils.isEmpty(idInfos) ? 0 : idInfos.size(), (queryStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            for(int i = 0; i < idInfos.size(); i++) {
                step = 1;
                idInfo = idInfos.get(i);
                if (Objects.isNull(idInfo.getRepaymentReceivedAt()) || "2000-01-01".equals(DateUtils.formatDate(idInfo.getRepaymentReceivedAt(),"yyyy-MM-dd"))) {
                    idInfo.setRepaymentReceivedAt(new Date());
                }
                while (Boolean.TRUE) {
                    stepEndDate = org.apache.commons.lang3.time.DateUtils.addDays(idInfo.getDeadline(),30*step);
                    if (stepEndDate.compareTo(idInfo.getRepaymentReceivedAt())>0) {
                        break;
                    }
                    tempRepaymentLoanInfos = repaymentLoanInfoMapper.queryD3OverdueRecordOf30Day(BaiHangTimeRecord.builder().loanApplicationHistoryId(idInfo.getLoanApplicationHistoryId()).endTime(DateUtils.formatDate(stepEndDate,"yyyy-MM-dd HH:mm:ss")).build());
                    repaymentLoanInfos.addAll(tempRepaymentLoanInfos);
                    step++;
                }
            }
            for (int i = 0; i < repaymentLoanInfos.size(); i++) {
                RepaymentInfoZhuDai repaymentLoanInfo = repaymentLoanInfos.get(i);
                try {
                    id = UUID.randomUUID().toString().replaceAll("-", "");
                    RepaymentInfoZhuDai record = new RepaymentInfoZhuDai();
                    BeanUtils.copyProperties(repaymentLoanInfo, record);
                    record.setRecordId(id);
                    recordList.add(record);
                    repaymentLoanInfo.setReqID(id);
                    repaymentLoanInfo.setName(sensitiveFilter(repaymentLoanInfo.getName()));
                    repaymentLoanInfo.setPid(sensitiveFilter(repaymentLoanInfo.getPid()));
                    repaymentLoanInfo.setMobile(sensitiveFilter(repaymentLoanInfo.getMobile()));
                    if (isDebug && StringUtils.isNotBlank(prefix)) {
                        repaymentLoanInfo.setReqID(prefix+repaymentLoanInfo.getReqID());
                        repaymentLoanInfo.setLoanId(prefix+repaymentLoanInfo.getLoanId());
                    }
                    atomicInteger.getAndIncrement();
                    reportList.add(JSON.toJSONString(repaymentLoanInfo));
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(D3)-每30天逾期记录处理异常, recordId: {} , loanId: {} ", id, repaymentLoanInfo.getLoanId(), e);
                }
            }
            saveRepaymentLoanInfoLog(recordList,startTime,endTime);
            File file = new File(HOST_PATH+"量子数科科技有限公司_D3O30_"+fileNameReplaceAll(startTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(D3)-每30天逾期记录为空,加密文件不生成!starTime={},endTime={}",startTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(D3)-每30天逾期记录完成,开始时间: {} , 结束时间: {} , 实际大小: {} , 报送成功大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(D3)-每30天逾期记录处理异常,开始时间: {} , 结束时间: {} ", startTime, endTime, e);
        } finally {
            clear(idInfos,repaymentLoanInfos,tempRepaymentLoanInfos,recordList,reportList);
        }
        return encryptFileName;
    }

    /**
     * 非循环贷款贷后数据信息（D3）还款记录
     * @param startTime 开始时间
     * @param endTime 截止时间
     */
    private String reportD3R(String startTime, String endTime, String loanStartTime, String loanEndTime, String prefix) {
        String encryptFileName = null;
        List<RepaymentInfoZhuDai> repaymentLoanInfos = null,recordList = new ArrayList<>(D3_INITIAL_CAPACITY);
        List<String> reportList = new ArrayList<>(D3_INITIAL_CAPACITY);
        reportList.add("#singleLoanRepayInfo");
        //修复数据类型 : 还款状态确认时间取错,导致晚于下一期的还款状态确认时间
        List<String> repairDataList_01 = Arrays.asList("AN000000192289736004493312");
        Stopwatch sendWatch = Stopwatch.createStarted();
        try {
            Stopwatch queryStopwatch = Stopwatch.createStarted();
            repaymentLoanInfos = repaymentLoanInfoMapper.queryRepayMentRecordOfD3(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).loanStartTime(loanStartTime).loanEndTime(loanEndTime).build());
            log.info("量化派助贷TO百行报送(D3)-还款记录查询结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (queryStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            AtomicInteger atomicInteger = new AtomicInteger();
            String id = "";
            for (int i = 0; i < repaymentLoanInfos.size(); i++) {
                RepaymentInfoZhuDai repaymentLoanInfo = repaymentLoanInfos.get(i);
                try {
                    if (repairDataList_01.contains(repaymentLoanInfo.getLoanId())) {
                        log.error("量化派助贷TO百行报送(D3)-还款记录处理脏数据: {}",repaymentLoanInfo.getLoanId());
                        //脏数据特殊处理,不修改业务库数据
                        Date realRepaymentDate = DateUtils.parseDate("yyyy-MM-dd'T'HH:mm:ss",repaymentLoanInfo.getRealRepaymentDate());
                        Date statusConfirmAtDate = org.apache.commons.lang3.time.DateUtils.addSeconds(realRepaymentDate,5);
                        repaymentLoanInfo.setStatusConfirmAt(DateUtils.formatDate(statusConfirmAtDate,"yyyy-MM-dd'T'HH:mm:ss"));
                    }
                    id = UUID.randomUUID().toString().replaceAll("-", "");
                    RepaymentInfoZhuDai record = new RepaymentInfoZhuDai();
                    BeanUtils.copyProperties(repaymentLoanInfo, record);
                    record.setRecordId(id);
                    recordList.add(record);
                    repaymentLoanInfo.setReqID(id);
                    repaymentLoanInfo.setName(sensitiveFilter(repaymentLoanInfo.getName()));
                    repaymentLoanInfo.setPid(sensitiveFilter(repaymentLoanInfo.getPid()));
                    repaymentLoanInfo.setMobile(sensitiveFilter(repaymentLoanInfo.getMobile()));
                    if (isDebug && StringUtils.isNotBlank(prefix)) {
                        repaymentLoanInfo.setReqID(prefix+repaymentLoanInfo.getReqID());
                        repaymentLoanInfo.setLoanId(prefix+repaymentLoanInfo.getLoanId());
                    }
                    atomicInteger.getAndIncrement();
                    reportList.add(JSON.toJSONString(repaymentLoanInfo));
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(D3)-还款记录处理异常, recordId: {} , loanId: {} ", id, repaymentLoanInfo.getLoanId(), e);
                }
            }
            saveRepaymentLoanInfoLog(recordList,startTime,endTime);
            File file = new File(HOST_PATH+"量子数科科技有限公司_D3R_"+fileNameReplaceAll(startTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+String.format("%06d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(D3)-还款记录为空,加密文件不生成!starTime={},endTime={}",startTime,endTime);
            } else {
                encryptFileName = createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(D3)-还款记录完成,开始时间: {} , 结束时间: {} , 实际大小: {} , 报送成功大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(D3)-还款记录处理异常,开始时间: {} , 结束时间: {} ", startTime, endTime, e);
        } finally {
            clear(repaymentLoanInfos,recordList,reportList);
        }
        return encryptFileName;
    }

    private void clear(Collection ... collections) {
        for (Collection col : collections) {
            if (!CollectionUtils.isEmpty(col)) {
                col.clear();
            }
        }
    }

    private void saveRepaymentLoanInfoLog(List<RepaymentInfoZhuDai> recordList,String startTime, String endTime) {
        if (!CollectionUtils.isEmpty(recordList)) {
            List<List<RepaymentInfoZhuDai>> allList = null;
            try {
                allList = Lists.partition(recordList, LISTS_PARTITION_SIZE);
                Stopwatch saveStopwatch = Stopwatch.createStarted();
                allList.stream().forEach(pRecordList -> {
                    repaymentLoanInfoDbMapper.batchSaveRepaymentLoanInfoLog(pRecordList);
                });
                log.info("量化派助贷TO百行报送(D3)批量插入结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, recordList.size(), (saveStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            } catch (Exception e) {
                log.error("量化派助贷TO百行报送(D3)保存异常", e);
            } finally {
                clear(recordList,allList);
            }
        }
    }

    private void saveLoanInfoRecordLog(List<LoanInfoZhuDai> recordList,String startTime, String endTime) {
        if (!CollectionUtils.isEmpty(recordList)) {
            List<List<LoanInfoZhuDai>> allList = null;
            try {
                allList = Lists.partition(recordList, LISTS_PARTITION_SIZE);
                Stopwatch saveStopwatch = Stopwatch.createStarted();
                allList.stream().forEach(pRecordList -> {
                    loanInfoDbMapper.batchSaveLoanInfoRecordLog(pRecordList);
                });
                log.info("量化派助贷TO百行报送(D2)批量插入结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, recordList.size(), (saveStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            } catch (Exception e) {
                log.error("量化派助贷TO百行报送(D2)保存异常", e);
            } finally {
                clear(recordList,allList);
            }
        }
    }

    private void saveApplyLoanInfoZhuDai(List<ApplyLoanInfoZhuDai> recordList,String startTime, String endTime) {
        if (!CollectionUtils.isEmpty(recordList)) {
            List<List<ApplyLoanInfoZhuDai>> allList = null;
            try {
                allList = Lists.partition(recordList, LISTS_PARTITION_SIZE);
                Stopwatch saveStopwatch = Stopwatch.createStarted();
                allList.stream().forEach(pRecordList -> {
                    applyLoanInfoZhuDaiMapper.batchSaveLog(pRecordList);
                });
                log.info("量化派助贷TO百行报送(A1)批量插入结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, recordList.size(), (saveStopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            } catch (Exception e) {
                log.error("量化派助贷TO百行报送(A1)保存异常", e);
            } finally {
                clear(recordList,allList);
            }
        }
    }

    private Boolean increment(String key) {
        Long increment = redisTemplate.opsForValue().increment(key, 1);
        return increment <= 1;
    }

    public void deleteReportRedisKey(String keyStr) {
        if (redisTemplate.hasKey(keyStr)) {
            redisTemplate.delete(keyStr);
        }
    }

    public String getFundNameBySplitSymbol(List<Integer> fundingCorpIdList,String splitSymbol) {
        StringBuffer fundNameBuffer = new StringBuffer();
        List<FundingCorpInfo> fundingCorpInfoList = applyLoanInfoMapper.findFundingCorpInfo(fundingCorpIdList);
        if (!CollectionUtils.isEmpty(fundingCorpInfoList)) {
            fundNameBuffer.append(fundingCorpInfoList.stream().map(fundingCorpInfo -> fundingCorpInfo.getName()).collect(Collectors.joining(splitSymbol)));
        }
        return fundNameBuffer.toString();
    }

    @Async
    public void fileToBaiHangFtp(String collectFileName) throws Exception{
        if (StringUtils.isBlank(collectFileName)) {
            return;
        }
        log.info("文件开始上传到百行ftp服务器...");
        File collectFile = new File(HOST_PATH + collectFileName);
        File subFile = null;
        if (collectFile.exists()) {
            List<String> fileNameList = FileUtils.readLines(collectFile, "utf-8");
            SftpUtil sftp = new SftpUtil(baiHangFtpUsername,baiHangFtpPassword,baiHangFtpHost,baiHangFtpPort);
            for (String fileName : fileNameList) {
                if (StringUtils.isNotBlank(fileName)) {
                    subFile = new File(HOST_PATH + fileName);
                    if (subFile.exists()) {
                        log.info("开始上传文件 {} 到百行ftp服务器...",fileName);
                        try {
                            SftpUtil.uploadFile(sftp,baiHangFtpDataDirectory,HOST_PATH + fileName);
                            log.info("文件 {} 已上传到百行ftp服务器",fileName);
                        } catch (Exception e) {
                            log.error("文件 {} 上传百行ftp服务器失败,错误信息: {}",fileName,e);
                        }
                    } else {
                        log.error("文件 {} 不存在",fileName);
                    }
                }
            }
            sftp.logout();
        } else {
            log.error("文件 {} 不存在",collectFileName);
        }
    }

    @Async
    public void fileCollectDataCheck(String type,String collectFileName,String fieldNames) throws Exception{
        log.info("{}集合文件数据校验开始...",collectFileName);
        if (StringUtils.isBlank(collectFileName)) {
            return;
        }
        File collectFile = new File(HOST_PATH + collectFileName);
        if (collectFile.exists()) {
            List<String> fileNameList = FileUtils.readLines(collectFile, "utf-8");
            for (String fileName : fileNameList) {
                if (StringUtils.isNotBlank(fileName)) {
                    singleFileDataCheck(type,fileName,fieldNames);
                }
            }
        }
        log.info("{}集合文件数据校验结束",collectFileName);
    }

    @Async
    public void singleFileDataCheck(String type,String singleFileName,String checkFieldNames) throws Exception{
        log.info("{}文件数据校验开始...",singleFileName);
        List<String> checkResult = new ArrayList();
        Boolean noError = null;
        StringBuffer sBuffer = null;
        String dataStr = null;
        BufferedReader bis = null;
        try {
            List<String> checkFieldNameList = Arrays.asList(checkFieldNames.split(","));
            bis = new BufferedReader(new FileReader(HOST_PATH + singleFileName.replace("-R.cry",".txt")));
            while(null != (dataStr = bis.readLine())) {
                if (StringUtils.isBlank(dataStr) || StringUtils.equals(dataStr,"#applyInfo") || StringUtils.equals(dataStr,"#singleLoanAccountInfo") || StringUtils.equals(dataStr,"#singleLoanRepayInfo")) {

                } else {
                    sBuffer = new StringBuffer(singleFileName).append(",");
                    noError = Boolean.TRUE;
                    if ("A1".equalsIgnoreCase(type) || "A1Refuse".equalsIgnoreCase(type)) {
                        ApplyLoanInfoZhuDai applyLoanInfoZhuDai = JSONUtil.toBean(dataStr, ApplyLoanInfoZhuDai.class);
                        log.info("数据 {} 正在校验...",applyLoanInfoZhuDai.getApplyId());
                        sBuffer.append(applyLoanInfoZhuDai.getApplyId()).append(",");
                        for (String checkFieldName : checkFieldNameList) {
                            Field declaredField = ApplyLoanInfoZhuDai.class.getDeclaredField(checkFieldName);
                            Object property = BeanUtil.getProperty(applyLoanInfoZhuDai, checkFieldName);
                            if (Objects.isNull(property) || Objects.isNull(declaredField) || StringUtils.isBlank(property.toString())) {
                                noError = Boolean.FALSE;
                                sBuffer.append(checkFieldName).append(",");
                            }
                        }
                    }
                    if (!noError) {
                        checkResult.add(sBuffer.append(System.getProperty("line.separator")).toString());
                    }
                }
            }
            if (!CollectionUtils.isEmpty(checkResult)) {
                File file = new File(HOST_PATH + type + "_fileDataCheck_" + DateUtils.formatDate(new Date(),"yyyyMMddHHmmsss") + ".txt");
                FileUtils.writeLines(file,checkResult,Boolean.TRUE);
            }
        } catch (Exception e) {
            log.error("{}文件数据校验异常",singleFileName,e);
        } finally {
            IoUtil.close(bis);
        }
        log.info("{}文件数据校验结束",singleFileName);
    }


    @Async
    public void fileDataCheckCopy(String sourceFileName,String targetFileName,String checkFieldNames) throws Exception{
        log.info("sourceFileName:{} , targetFileName:{} 文件数据校验拷贝开始...",sourceFileName,targetFileName);
        List<String> checkFieldNameList = Arrays.asList(checkFieldNames.split(","));
        BufferedReader bis = null;
        FileWriter bos = null;
        String dataStr = null;
        try {
            bis = new BufferedReader(new FileReader(HOST_PATH + sourceFileName));
            bos = new FileWriter(HOST_PATH + targetFileName);
            while(null != (dataStr = bis.readLine())) {
                if (StringUtils.isBlank(dataStr) || StringUtils.equals(dataStr,"#applyInfo") || StringUtils.equals(dataStr,"#singleLoanAccountInfo") || StringUtils.equals(dataStr,"#singleLoanRepayInfo")) {
                    bos.write(dataStr + System.getProperty("line.separator"));
                    log.error("文件数据可能异常:{}",dataStr);
                } else {
                    Boolean noError = Boolean.TRUE;
                    ApplyLoanInfoZhuDai applyLoanInfoZhuDai = JSONUtil.toBean(dataStr, ApplyLoanInfoZhuDai.class);
                    for (String fieldName : checkFieldNameList) {
                        Field declaredField = ApplyLoanInfoZhuDai.class.getDeclaredField(fieldName);
                        Object property = BeanUtil.getProperty(applyLoanInfoZhuDai, fieldName);
                        if (Objects.isNull(property) || Objects.isNull(declaredField)) {
                            noError = Boolean.FALSE;
                            break;
                        }
                        if (Objects.nonNull(declaredField) && declaredField.getType().equals(String.class)) {
                            if (StringUtils.isBlank(property.toString())) {
                                noError = Boolean.FALSE;
                                break;
                            }
                        }
                    }
                    if (noError) {
                        bos.write(dataStr + System.getProperty("line.separator"));
                    } else {
                        log.error("文件数据校验异常信息:{}",dataStr);
                    }
                }
            }
        } catch (Exception e) {
            log.error("sourceFileName:{} , targetFileName:{} 文件数据校验拷贝异常",sourceFileName,targetFileName,e);
        } finally {
            IoUtil.close(bos);
            IoUtil.close(bis);
            dataStr = null;
        }
        log.info("sourceFileName:{} , targetFileName:{} 文件数据校验拷贝结束",sourceFileName,targetFileName);
    }

}
