package cn.quantgroup.report.service.thirdpartymonitor;

import cn.quantgroup.report.cmpt.CommonAlarmCmpt;
import cn.quantgroup.report.constant.TransactionCodeEnum;
import cn.quantgroup.report.domain.master.*;
import cn.quantgroup.report.enums.RequestUrlType;
import cn.quantgroup.report.mapper.master.*;
import cn.quantgroup.report.service.usersdk.IUserCenterService;
import com.google.common.base.Stopwatch;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.FastDateFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class ThirdpartyApiMonitorService  { //extends DataService

  private static final FastDateFormat dateFormat = FastDateFormat.getInstance("yyyyMM", TimeZone.getDefault(), Locale.getDefault());
  private static final String DATA_PRICE_CACHE = "MONITOR.DATA_PRICE_CACHE";
  private static final String DATA_QRY_RECORD = "MONITOR.DATA_QRY_RECORD.";
  private static final String DATA_ACCOUNT = "MONITOR.ACCOUNT.";


  @Autowired
  private UnionPayApiRequestEventMapper unionPayApiRequestEventMapper;

  @Autowired
  private CommonApiReqestEventMapper commonApiReqestEventMapper;
  @Autowired
  private ApiRequestLogPOMapper apiRequestLogPOMapper;
  @Autowired
  private TransactionLogPOMapper transactionLogPOMapper;
  @Autowired
  private CostRecordPOMapper costRecordPOMapper;
  @Autowired
  private DataPriceConfigPOMapper dataPriceConfigPOMapper;
  @Autowired
  private AccountPOMapper accountPOMapper;
  @Autowired
  private CommonAlarmCmpt commonAlarmCmpt;
  @Autowired
  private IUserCenterService userCenterService;



  @Autowired
  private RedisTemplate redisTemplate;

  private static Map<String, DataPriceConfigPO> dataPriceMap = null;

  @Value("${isDebug}")
  private Boolean isDebug;


  /**
   *
   * @param id    用户标示
   * @param urlType   数据源类型
   * @param isBilling  是否收费（0，收费     1 ，不收费）
   * @param responseMessage  返回的信息（成功，权限不足，参数错误。。。等）
   * @param millis      该次请求耗时
   * @param businessCode      返回码，与返回信息对应
   *  修改主要是isBilling  之后将增加是否计费字段，废弃之前httpcode都为200的字段
   */
  public void create(String id, RequestUrlType urlType, String isBilling, String responseMessage, int millis, String businessCode, String...args) {
    Stopwatch stopwatch = Stopwatch.createStarted();
    switch (urlType) {


      case CreditCardRecord:
        unionPayApiRequestEventMapper.insert(UnionPayApiRequestEvent.builder()
                .bankCardNo(id)
                .mills(millis)
                .requestUrlType(urlType.name())
                .businessCode(businessCode)
                .timeCreated(new Date())
                .build());
        break;


      //量化派数据产品输出黑名等级
      case LhpBlackListLevel:
            commonApiReqestEventMapper.insert(CommonWithBillingApiRequestEvent.builder().userId(id)
                .isBilling(isBilling)
                .mills(millis)
                .requestUrlType(urlType.name())
                .responseCode(businessCode)
                .responseMessage(responseMessage)
                .timeCreated(new Date()).build());
            break;
      default:
        log.error("暂时未监控{}", urlType.name());
        break;
    }
    log.info("保存调用记录完成, id: {} , urlType: {} , 耗时: {} ", id, urlType, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS));
  }

  public void saveLog(RequestUrlType urlType, String code, String msg, String flag, int millis) {
    try {
      apiRequestLogPOMapper.insert(ApiRequestLogPO.builder()
              .urlType(urlType.name())
              .resCode(code)
              .resMsg(msg)
              .hitFlag(flag)
              .timeElapse(millis)
              .createTime(new Date())
              .suffix(dateFormat.format(new Date())).build());
    }catch (Exception e){
      log.error("保存接口请求记录失败", e);
    }
  }

  public void saveTransactionLog(RequestUrlType urlType, String transactionId, String uuid, TransactionCodeEnum code) {
    try {
      transactionLogPOMapper.insert(TransactionLogPO.builder()
              .transactionId(transactionId)
              .uuid(uuid)
              .urlType(urlType.name())
              .code(code.getCode())
              .timeCreated(new Date()).build());
    }catch (Exception e){
      log.error("保存接口请求记录TransactionLog失败", e);
    }
  }

  public void saveTransactionLogByPhone(RequestUrlType urlType, String transactionId, String phone, TransactionCodeEnum code) {
    try {
      String uuid = userCenterService.getUuidByPhoneNumber(phone);
      log.info("TransactionLog 查询 phone: {}， uuid: {}", phone, uuid);
      transactionLogPOMapper.insert(TransactionLogPO.builder()
              .transactionId(transactionId)
              .uuid(uuid)
              .urlType(urlType.name())
              .code(code.getCode())
              .timeCreated(new Date()).build());
    }catch (Exception e){
      log.error("保存接口请求记录TransactionLog失败", e);
    }
  }

  public void saveTransactionLogByIdentifyNumber(RequestUrlType urlType, String transactionId, String identifyNumber, TransactionCodeEnum code) {
    try {
      String uuid = userCenterService.getUuidByIdentityNumber(identifyNumber);
      log.info("TransactionLog 查询 identifyNumber: {}， uuid: {}", identifyNumber, uuid);
      transactionLogPOMapper.insert(TransactionLogPO.builder()
              .transactionId(transactionId)
              .uuid(uuid)
              .urlType(urlType.name())
              .code(code.getCode())
              .timeCreated(new Date()).build());
    }catch (Exception e){
      log.error("保存接口请求记录TransactionLog失败", e);
    }
  }

  public void recordCostInDb(RequestUrlType urlType, String sourceName, Integer invokeTimes, BigDecimal cost, BigDecimal balance) {
    costRecordPOMapper.insert(CostRecordPO.builder()
            .urlType(urlType.name())
            .sourceName(sourceName)
            .invokeTimes(invokeTimes)
            .cost(cost)
            .balance(balance)
            .createTime(new Date())
            .suffix(dateFormat.format(new Date())).build());
  }

  public DataPriceConfigPO qryDataPrice(RequestUrlType urlType) {
    try {
      dataPriceMap = redisTemplate.opsForHash().entries(DATA_PRICE_CACHE);
      if (dataPriceMap == null || dataPriceMap.size() == 0) {
        synchronized (ThirdpartyApiMonitorService.class) {
          if (dataPriceMap == null || dataPriceMap.size() == 0) {
            List<DataPriceConfigPO> dataPriceConfigPOS = dataPriceConfigPOMapper.queryByParams(DataPriceConfigPO.builder().status("1").build());
            dataPriceMap = new ConcurrentHashMap<>();
            for (DataPriceConfigPO dataPriceConfigPO : dataPriceConfigPOS) {
              dataPriceMap.put(dataPriceConfigPO.getUrlType(), dataPriceConfigPO);
            }
            redisTemplate.opsForHash().putAll(DATA_PRICE_CACHE, dataPriceMap);
          }
        }
      }
    }catch (Exception e){
      log.error("获取配置数据失败", e);
      return null;
    }
    return dataPriceMap.get(urlType.name());
  }

  public AccountPO qryAccount(String sourceName) {
    List<AccountPO> accountPOS = accountPOMapper.selectByExample(AccountPO.builder().sourceName(sourceName).status("1").build());
    return accountPOS.get(0);
  }

  private void updateAccount(AccountPO accountPO) {
    accountPOMapper.updateByParams(accountPO);
  }

  public void recordCost(RequestUrlType urlType, String sourceName){
    Long times = redisTemplate.opsForValue().increment(DATA_QRY_RECORD + urlType.name(), 1l);
    DataPriceConfigPO dataPriceConfigPO = qryDataPrice(urlType);
    if(times >= dataPriceConfigPO.getFrequency()){
      String lockKey = DATA_ACCOUNT + sourceName;
      boolean result = lock(lockKey, 60);
      if(!result) return;
      try{
        AccountPO accountPO = qryAccount(sourceName);

        BigDecimal blance = accountPO.getBlance();
        BigDecimal cost = dataPriceConfigPO.getPrice().multiply(BigDecimal.valueOf(times));
        blance = blance.subtract(cost);
        accountPO.setBlance(blance);
        if(blance.compareTo(accountPO.getThershold()) < 0){
          commonAlarmCmpt.alarm("Warn", "三方数据源告警", "数据源【"+ sourceName +"】 余额已不足"+accountPO.getThershold()+", 目前账户余额为"+blance);
        }
        updateAccount(accountPO);
        recordCostInDb(urlType, sourceName, times.intValue(), cost, blance);
        redisTemplate.delete(DATA_QRY_RECORD + urlType.name());
      }catch (Exception e){
        log.error("扣费异常， requestUrlType: {} ", urlType, e);
      }finally {
        unlock(lockKey);
      }
    }
  }

  public void saveInHBase(RequestUrlType urlType, String id, String requestJson, String responseJson) {

  }
  /*public void saveInHBase(RequestUrlType urlType, String id, String requestJson, String responseJson) {
    Stopwatch stopwatch = Stopwatch.createStarted();
    if (BooleanUtils.isTrue(isDebug)) {
      log.info("测试环境下不写入数据,urlType={},id={},requestJson={},responseJson={}", urlType, MaskedUtils.buildMaskedCommonString(id), requestJson, responseJson);
      return;
    }
    Table table = null;
    try {
      table = connection.getTable(ThirdPartyBean.TABLE_NAME);
      ThirdPartyBean thirdPartyDataRow = new ThirdPartyBean(id, urlType.name(), requestJson, responseJson);
      Put put = thirdPartyDataRow.generatePut();
      List<Get> rows = Lists.newArrayList(new Get(thirdPartyDataRow.findRowKey()));
      List<String> contents = Lists.newArrayList(JSON.toJSONString(thirdPartyDataRow));
      KafkaService.getInstance.process(table.existsAll(rows), ThirdPartyDataRow.TABLE_NAME.toString(), contents);
      table.put(put);
    } catch (Exception ex) {
      log.error("Put the data error! urlType:{},id:{},requestJson:{},responseJson:{}", urlType, id, requestJson, responseJson, ex);
    } finally {
      IOUtils.closeQuietly(table);
    }
    log.info("数据源保存hbase完毕, id: {} , urlType: {} , 耗时: {} ", id, urlType, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS));
  }*/

 /* public String loadInHBase(RequestUrlType urlType, String id) {
    return loadInHBaseAndCheckTimeOut(urlType, id, 60);
  }

  public <T> T loadObjectInHBase(RequestUrlType requestUrlType, String id, TypeReference<T> clazz) {
    String responseInHBase = loadInHBase(requestUrlType, id);
    if (StringUtils.isNotBlank(responseInHBase)) {
      T result = JSON.parseObject(responseInHBase, clazz);
      if (result != null) {
        log.info("直接从hbase拿到数据,responseInHBase:{}", responseInHBase);
      }
      return result;
    }
    return null;
  }*/

 /* public <T> T loadObjectInHBase(RequestUrlType requestUrlType, String id, Class<T> clazz) {
    String responseInHBase = loadInHBase(requestUrlType, id);
    if (StringUtils.isNotBlank(responseInHBase)) {
      T result = JSON.parseObject(responseInHBase, clazz);
      return result;
    }
    return null;
  }*/

  /*public JSONObject loadObjectInHbaseTimeOut(RequestUrlType urlType, String id, Integer days){
    String data = loadInHBaseAndCheckTimeOut(urlType, id, days);
    if(StringUtils.isNotBlank(data)){
      return JSON.parseObject(data, JSONObject.class);
    }
    return null;
  }*/

  public String loadInHBaseAndCheckTimeOut(RequestUrlType urlType, String id, Integer days) {
    return null;
  }
  /*public String loadInHBaseAndCheckTimeOut(RequestUrlType urlType, String id, Integer days) {
    if (!HBaseConfig.readHBaseFirst.get().booleanValue()) {
      log.info("不读取hbase哦噢噢噢噢");
      return null;
    }
    QGPreconditions.checkArgument(StringUtils.isNotEmpty(id), QGExceptionType.COMMON_STRING_PARAM_IS_NULL);
    QGPreconditions.checkArgument(urlType != null, QGExceptionType.COMMON_STRING_PARAM_IS_NULL);
    Table table = null;
    try {
      table = connection.getTable(ThirdPartyDataRow.TABLE_NAME);
      byte[] prefix = RowKeyTool.getMD5Prefix(id, 4);
      if (prefix == null) {
        log.error("The uuid is is error! urlType:{},id:{}", urlType, id);
        return null;
      }

      Get get = new Get(Bytes.add(prefix, RowKeyTool.createRowKeyField(id), Bytes.toBytes("|" + urlType.name())));
      Result result = table.get(get);
      if (result == null || result.isEmpty()) {
        return null;
      }
      byte[] responseMessage = result.getValue(ThirdPartyDataRow.COLUMN_FAMILY, ThirdPartyDataRow.RESPONSE_MESSAGE);
      byte[] requestMessage = result.getValue(ThirdPartyDataRow.COLUMN_FAMILY, ThirdPartyDataRow.REQUEST_MESSAGE);

      if (days != null && days.intValue() > 0) {
        byte[] requestTime = result.getValue(ThirdPartyDataRow.COLUMN_FAMILY, ThirdPartyDataRow.REQUEST_TIME);
        if (requestTime == null || requestTime.length == 0) {
          return null;
        }
        long timeCreated = ClockUtils.hBaseParseTimestampFromDateTimeString(Bytes.toString(requestTime));
        if (ClockUtils.getDaysBetween(timeCreated, ClockUtils.now()) > days) {
          return null;
        }
      }
      if (urlType.equals(HaoAnIdentityCheck)){
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("requestMessage",Bytes.toString(requestMessage));
        jsonObject.put("responseMessage",Bytes.toString(responseMessage));
        return jsonObject.toJSONString();
      }
      log.info("从habse拿到数据 urlType:{},id:{}", urlType, id);
      return Bytes.toString(responseMessage);
    } catch (Exception ex) {
      log.error("Get the data error! urlType:{},id:{}", urlType, id, ex);
      return null;
    } finally {
      IOUtils.closeQuietly(table);
    }
  }
*/
  /**
   * lock
   *
   */
  public boolean lock(String lockKey, long timeOutSecond) {
    long nano = System.nanoTime();
    final Random r = new Random();

    try {
      while ((System.nanoTime() - nano) < 8000000000L) {
        RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
        boolean exits = connection.setNX(lockKey.getBytes(), new byte[0]);
        connection.close();
        //获取锁
        if (exits) {
          redisTemplate.expire(lockKey, timeOutSecond, TimeUnit.SECONDS);
          return true;
        }
        try {
          Thread.sleep(3, r.nextInt(500));
        } catch (InterruptedException e) {
          log.error("{}", e.getMessage(), e);
        }
      }
    }catch (Exception e){
      unlock(lockKey);
    }
    return false;

  }

  /**
   * unlock
   *
   * @param lockKey
   */
  public void unlock(String lockKey) {
    //防止死锁
    try {
      RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
      connection.del(lockKey.getBytes());
      connection.close();
    }catch (Exception e){
      log.error("{}", e.getMessage(), e);
    }
  }

}
