package cn.quantgroup.report.service.baihang;

import cn.quantgroup.report.domain.baihang.*;
import cn.quantgroup.report.error.QGException;
import cn.quantgroup.report.mapper.baihang.ApplyLoanInfoMapper;
import cn.quantgroup.report.mapper.baihang.LoanInfoMapper;
import cn.quantgroup.report.mapper.baihang.RepaymentLoanInfoMapper;
import cn.quantgroup.report.mapper.master.ApplyLoanInfoZhuDaiMapper;
import cn.quantgroup.report.mapper.master.LoanInfoDbMapper;
import cn.quantgroup.report.mapper.master.RepaymentLoanInfoDbMapper;
import cn.quantgroup.report.service.baihang.client.BhFileCreditApiClient;
import cn.quantgroup.report.service.baihang.request.FileUploadRequest;
import cn.quantgroup.report.service.baihang.response.FileUploadResponse;
import cn.quantgroup.report.service.baihang.util.TuoMinUtils;
import cn.quantgroup.report.service.http.IHttpService;
import cn.quantgroup.report.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Stopwatch;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.File;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import static cn.quantgroup.report.error.QGExceptionType.COMMON_ILLEGAL_PARAM_TOAST;

/**
 * @author jian.zheng
 * @date 2021/6/4 - 15:23
 **/
@Slf4j
@Service
public class BaiHangFileReportService {


    @Value("${isDebug}")
    private Boolean isDebug;
    @Autowired
    public ApplyLoanInfoMapper applyLoanInfoMapper;
    @Autowired
    public IHttpService iHttpService;
    @Autowired
    public LoanInfoMapper loanInfoMapper;
    @Autowired
    public RepaymentLoanInfoMapper repaymentLoanInfoMapper;
    @Autowired
    public LoanInfoDbMapper loanInfoDbMapper;
    @Autowired
    public RepaymentLoanInfoDbMapper repaymentLoanInfoDbMapper;
    @Autowired
    public ApplyLoanInfoZhuDaiMapper applyLoanInfoZhuDaiMapper;

    private static final String HOST_PATH = "/home/quant_group/reportFile/";

//    private static final String HOST_PATH = "C:\\Users\\musterHunter\\Desktop\\TEST_REPORT\\";


    /**
     * 创建报送文件 通常用于存量报送
     * @param type 报送类型 A1:申请信息 D2:放款信息 D3:还款信息
     * @param start 报送区间的开始时间 格式:"2019-05-07T00:00:00"
     * @param end 报送区间的截止时间 格式:"2019-05-07T00:00:00"
     */
    @Async
    public void createReportFile(String type, String start, String end,int daySplitCount) {
        if (StringUtils.isAnyBlank(type, start, end)) {
            throw new QGException(COMMON_ILLEGAL_PARAM_TOAST,"存在为空参数");
        }
        String dateFormat = "yyyy-MM-dd";
        Date stepStartDate = DateUtils.parseDate(dateFormat,start);
        Date stepEndDate = org.apache.commons.lang3.time.DateUtils.addDays(stepStartDate,daySplitCount);
        Date endDate = DateUtils.parseDate(dateFormat,end);
        Stopwatch stopwatch = Stopwatch.createStarted();
        while (stepEndDate.getTime() <= endDate.getTime()){
            createReportFile(type,stepStartDate,stepEndDate);
            stepStartDate = new Date(stepEndDate.getTime());
            stepEndDate = org.apache.commons.lang3.time.DateUtils.addDays(stepEndDate,daySplitCount);
        }
        log.info("当前 {} -- {}--{} 任务执行结束 总耗时 : {}",type,start,end,stopwatch.elapsed(TimeUnit.MILLISECONDS));
        if (stepEndDate.getTime() > endDate.getTime()){
            createReportFile(type,stepStartDate,endDate);
        }
    }

    private void createReportFile(String type, Date startDate, Date endDate) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        String starTime = DateUtils.formatDate(startDate,"yyyy-MM-dd HH:mm:ss");
        String endTime = DateUtils.formatDate(endDate,"yyyy-MM-dd HH:mm:ss");
        log.info("开始生成报送文件{}----- {} -- {}",type,starTime,endTime);
        if ("A1".equalsIgnoreCase(type)) {
            reportA1(starTime,endTime);
        } else if ("D2".equalsIgnoreCase(type)) {
            reportD2(starTime,endTime);
        } else if ("D3".equalsIgnoreCase(type)) {
            reportD3(starTime,endTime);
        } else {
            throw new QGException(COMMON_ILLEGAL_PARAM_TOAST,"报送类型不正确");
        }
        log.info("生成报送文件结束{}----- {} -- {},耗时 {}",type,starTime,endTime,stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    /**
     * 贷款申请信息（A1）
     * @param starTime 开始时间
     * @param endTime 截止时间
     */
    private void reportA1(String starTime, String endTime){
        try {
            Stopwatch queryWatch1 = Stopwatch.createStarted();
            BaiHangTimeRecord timeRecord = BaiHangTimeRecord.builder().startTime(starTime).endTime(endTime).build();
            List<ApplyLoanInfoZhuDai> applyLoanInfos = applyLoanInfoMapper.findRealTimeApplyLoanZhuDai(timeRecord);
            log.info("量化派助贷TO百行报送(A1)-非联合贷申请查询完成, 大小: {} , 耗时: {} ", (applyLoanInfos != null ? applyLoanInfos.size() : 0), (queryWatch1.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
            if (applyLoanInfos == null) {
                applyLoanInfos = new ArrayList<ApplyLoanInfoZhuDai>();
            }
            AtomicInteger atomicInteger = new AtomicInteger();
            Stopwatch sendWatch = Stopwatch.createStarted();
            List<String> reportList = new ArrayList<>();
            reportList.add("#applyInfo");
            String reqId_log = "";
            for (int i = 0; i < applyLoanInfos.size(); i++) {
                try {
                    ApplyLoanInfoZhuDai applyLoanInfo = applyLoanInfos.get(i);
                    try {
                        ApplyLoanInfoZhuDai record = new ApplyLoanInfoZhuDai();
                        BeanUtils.copyProperties(applyLoanInfo, record);
                        record.setRecordId(UUID.randomUUID().toString().replaceAll("-", ""));
                        applyLoanInfoZhuDaiMapper.saveApplyLoanInfoZhuDaiLog(record);
                    } catch (Exception e) {
                        log.error("量化派助贷TO百行报送(A1)-贷款申请信息保存记录失败", e);
                    }
                    reqId_log = applyLoanInfo.getReqID();
                    //邮箱有错误的就不传
                    if (!TuoMinUtils.checkEmail(applyLoanInfo.getEmailAddress())) {
                        applyLoanInfo.setEmailAddress(null);
                    }
                    applyLoanInfo.setName(sensitiveFilter(applyLoanInfo.getName()));
                    applyLoanInfo.setMobile(sensitiveFilter(applyLoanInfo.getMobile()));
                    applyLoanInfo.setPid(sensitiveFilter(applyLoanInfo.getPid()));
                    reportList.add(JSON.toJSONString(applyLoanInfo));
                    atomicInteger.getAndIncrement();
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送(A1)-贷款申请信息异常, reqId_log: {} ", reqId_log, e);
                }
            }
            File file = new File("量子数科科技有限公司_A1_"+fileNameReplaceAll(starTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+String.format("%04d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(A1)-贷款申请信息为空,加密文件不生成!starTime={},endTime={}",starTime,endTime);
            } else {
                createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送(A1)-贷款申请信息完成, 实际大小: {} , 报送成功大小: {} , 耗时: {} ", applyLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送(A1)-异常", e);
        }
    }

    /**
     * 非循环贷款账户数据信息（D2）
     * @param starTime 开始时间
     * @param endTime 截至时间
     */
    private void reportD2(String starTime, String endTime) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            List<LoanInfoZhuDai> loanInfozdList = loanInfoMapper.queryStockLoanInfoZhuDai(BaiHangTimeRecord.builder().startTime(starTime).endTime(endTime).build());
            int listSize = loanInfozdList != null ? loanInfozdList.size() : 0;
            log.info("百行-助贷放款数据查询结束, listSize : {} , startTime: {} , endTime: {} , 耗时 : {} ", listSize, starTime, endTime, stopwatch.elapsed(TimeUnit.MILLISECONDS));
            AtomicInteger atomicInteger = new AtomicInteger();
            Stopwatch sendWatch = Stopwatch.createStarted();
            List<String> reportList = new ArrayList<>();
            reportList.add("#singleLoanAccountInfo");
            for (int i = 0; i < loanInfozdList.size(); i++) {
                LoanInfoZhuDai loanInfo = loanInfozdList.get(i);
                String id = "";
                try {
                    String loanInfoReqId = loanInfo.getReqID();
                    List<String> targetRepaymentDayList = loanInfoMapper.findTargetRepaymentDayList(loanInfoReqId);
                    if (targetRepaymentDayList != null && targetRepaymentDayList.size() > 0) {
                        loanInfo.setTargetRepayDateList(String.join(",", targetRepaymentDayList));
                        log.info("比较还款总期数&账单日列表totalTerm=" + loanInfo.getTotalTerm() + ",targetRepaymentDayList size=" + targetRepaymentDayList.size());
                    } else {
                        log.warn("比较还款总期数&账单日列表totalTerm=" + loanInfo.getTotalTerm() + ",targetRepaymentDayList is null.");
                    }
                    UUID loanInfoId = UUID.randomUUID();
                    id = loanInfoId.toString().replaceAll("-", "");
                    try {
                        LoanInfoZhuDai record = new LoanInfoZhuDai();
                        BeanUtils.copyProperties(loanInfo, record);
                        record.setRecordId(id);
                        loanInfoDbMapper.saveLoanInfoRecordLog(record);
                    } catch (Exception e) {
                        log.error("量化派助贷TO百行报送-放款申请保存记录失败", e);
                    }
                    String jsonStr = JSONObject.toJSONString(loanInfo);
                    LoanInfoZhuDaiVo loanInfoVo = JSONObject.parseObject(jsonStr, LoanInfoZhuDaiVo.class);
                    FinTechAgencyBusinessZhuDai finTechAgencyBusiness = JSONObject.parseObject(jsonStr, FinTechAgencyBusinessZhuDai.class);
                    loanInfoVo.setFinTechAgencyBusiness(finTechAgencyBusiness);
                    loanInfoVo.setReqID(id);
                    loanInfoVo.setName(sensitiveFilter(loanInfo.getName()));
                    loanInfoVo.setPid(sensitiveFilter(loanInfo.getPid()));
                    loanInfoVo.setMobile(sensitiveFilter(loanInfo.getMobile()));
                    if (DateUtils.parseDate("yyyy-MM-dd'T'HH:mm:ss",loanInfoVo.getIssueDate()).getTime() < DateUtils.parseDate("yyyy-MM-dd'T'HH:mm:ss",loanInfoVo.getAccountOpenDate()).getTime()){
                        loanInfoVo.setAccountOpenDate(loanInfoVo.getIssueDate());
                        loanInfoVo.setApplyDate(loanInfoVo.getIssueDate());
                    }
                    reportList.add(JSON.toJSONString(loanInfoVo));
                    atomicInteger.getAndIncrement();
                } catch (Exception e) {
                    log.error("量化派助贷TO百行报送-放款信息异常, recordId: {} , loanId: {} ", id, loanInfo.getLoanId(), e);
                }
            }
            File file = new File("量子数科科技有限公司_D2_"+fileNameReplaceAll(starTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+String.format("%04d",new Random().nextInt(999999))+".txt");
            FileUtils.writeLines(file,reportList);
            if (reportList.size()<=1) {
                log.info("量化派助贷TO百行报送(D2)-放款信息为空,加密文件不生成!starTime={},endTime={}",starTime,endTime);
            } else {
                createFile(file.getAbsolutePath());
            }
            log.info("量化派助贷TO百行报送-放款申请完成, 实际大小: {} , 报送成功大小: {} , 耗时: {} ", loanInfozdList.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
        } catch (Exception e) {
            log.error("量化派助贷TO百行报送-异常", e);
        }
    }

    /**
     * 非循环贷款贷后数据信息（D3）
     * @param startTime 开始时间
     * @param endTime 截止时间
     */
    private void reportD3( String startTime, String endTime) {
        for (int j = 0; j < 2; j++) {
            List<String> reportList = new ArrayList<>();
            reportList.add("#singleLoanRepayInfo");
            Stopwatch sendWatch = Stopwatch.createStarted();
            try {
                List<RepaymentInfoZhuDai> repaymentLoanInfos = null;
                if (j == 0) {
                    Stopwatch stopwatch = Stopwatch.createStarted();
                    repaymentLoanInfos = repaymentLoanInfoMapper.queryStockRepayMentInfoZhuDai(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
                    log.info("百行-助贷还款查询结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (stopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");
                } else {
                    Stopwatch stopwatch = Stopwatch.createStarted();
                    repaymentLoanInfos = repaymentLoanInfoMapper.queryStockRepayMentInfoOverdueZhuDai(BaiHangTimeRecord.builder().startTime(startTime).endTime(endTime).build());
                    log.info("百行-助贷还款查询逾期结束, startTime: {} , endTime: {} , 大小: {} , 耗时: {} ", startTime, endTime, repaymentLoanInfos.size(), (stopwatch.stop().elapsed(TimeUnit.MILLISECONDS) / 1000) + ".s");

                }
                AtomicInteger atomicInteger = new AtomicInteger();
                for (int i = 0; i < repaymentLoanInfos.size(); i++) {
                    RepaymentInfoZhuDai repaymentLoanInfo = repaymentLoanInfos.get(i);
                    String id = "";
                    try {
                        RepaymentInfoZhuDai repaymentLoanInfo1 = repaymentLoanInfoDbMapper.findLastOne(BaiHangRepayment.builder().loanId(repaymentLoanInfo.getLoanId()).termNo(repaymentLoanInfo.getTermNo()).build());
                        if (j > 0 && Objects.nonNull(repaymentLoanInfo1)) {
                            log.info("量化派助贷TO百行报送-实时还款逾期跳过报送, startTime: {} , endTime: {} , bean: {} ", startTime, endTime, JSON.toJSONString(repaymentLoanInfo1));
                            continue;
                        }
                        id = UUID.randomUUID().toString().replaceAll("-", "");
                        try {
                            RepaymentInfoZhuDai record = new RepaymentInfoZhuDai();
                            BeanUtils.copyProperties(repaymentLoanInfo, record);
                            record.setRecordId(id);
                            repaymentLoanInfoDbMapper.saveRepaymentLoanInfoLog(record);
                        } catch (Exception e) {
                            log.error("量化派助贷TO百行报送-实时还款保存记录异常", e);
                        }
                        repaymentLoanInfo.setReqID(id);

                        repaymentLoanInfo.setName(sensitiveFilter(repaymentLoanInfo.getName()));
                        repaymentLoanInfo.setPid(sensitiveFilter(repaymentLoanInfo.getPid()));
                        repaymentLoanInfo.setMobile(sensitiveFilter(repaymentLoanInfo.getMobile()));
                        atomicInteger.getAndIncrement();
                        reportList.add(JSON.toJSONString(repaymentLoanInfo));
                    } catch (Exception e) {
                        log.error("量化派助贷TO百行报送-还款信息异常, recordId: {} , loanId: {} ", id, repaymentLoanInfo.getLoanId(), e);
                    }
                }
                File file = new File("量子数科科技有限公司_D3_"+fileNameReplaceAll(startTime.split("T")[0])+"_"+fileNameReplaceAll(endTime.split("T")[0])+"_"+j+"_"+String.format("%04d",new Random().nextInt(999999))+".txt");
                FileUtils.writeLines(file,reportList);
                if (reportList.size()<=1) {
                    log.info("量化派助贷TO百行报送(D3)-还款信息为空,加密文件不生成!starTime={},endTime={}",startTime,endTime);
                } else {
                    createFile(file.getAbsolutePath());
                }
                log.info("量化派助贷TO百行报送-还款申请完成 J: {} , 开始时间: {} , 结束时间: {} , 实际大小: {} , 报送成功大小: {} , 耗时: {} ", j, startTime, endTime, repaymentLoanInfos.size(), atomicInteger.get(), sendWatch.stop().elapsed(TimeUnit.MILLISECONDS));
            } catch (Exception e) {
                log.error("量化派助贷TO百行报送-异常 J: {} , 开始时间: {} , 结束时间: {} ", j, startTime, endTime, e);
            }
        }
    }

    /**
     * 根据传入的文件路径将加密后的文件生成到 {@link BaiHangFileReportService#HOST_PATH}中
     * @param filePath 需要加密的文件路径
     */
    private static void createFile(String filePath){
        log.info("创建加密文件,明文文件路径 {}",filePath);
        File host = new File(HOST_PATH);
        if (!host.exists()){
            host.mkdir();
        }
        //AES密钥（机构自行设置）
        String aesKey = "fc4c1cbca5a46840f60";
        FileUploadRequest req = new FileUploadRequest();
        //设置未作压缩加密前的原始数据文件路径
        req.setDataFile(filePath);
        //设置压缩加密后的密文文件输出路径(为空表示与原始数据文件同目录)
        req.setTargetFilePath(HOST_PATH);
        BhFileCreditApiClient client = new BhFileCreditApiClient();
        //初始化设置AES密钥和RSA公钥
        client.init(BaiHangZhuDaiService.getBaihangZDPublickey(),aesKey);
        //执行压缩加密操作
        FileUploadResponse response = client.execute(req);
        log.info("执行压缩加密操作出参{}",JSONObject.toJSONString(response));
        if(response.isSuccess){
            System.out.println("zip And encrypt success;fileName = "+response.getEncryptFilePath() + response.getEncryptFileName());
        }else{
            System.out.println("zip And encrypt fail;errorMessage = "+response.getErrorMessage());
        }
    }

    private String sensitiveFilter(String s){
        if (isDebug){
            if (StringUtils.isNumeric(s) || s.contains("X") || s.contains("x")){
                return s.substring(0,s.length()-4)+"0000";
            }else {
                return s.charAt(0)+"零";
            }
        }else {
            return s;
        }
    }

    private String fileNameReplaceAll(String str) {
        if (str.length()<=0) {
            return "";
        }
        return str.replaceAll("-","").replaceAll(":","").replaceAll(" ","");
    }
}
