package com.quantgroup.asset.distribution.service.funding.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Stopwatch;
import com.quantgroup.asset.distribution.constant.AidFundStatus;
import com.quantgroup.asset.distribution.model.form.AssetForm;
import com.quantgroup.asset.distribution.model.response.GlobalResponse;
import com.quantgroup.asset.distribution.service.funding.*;
import com.quantgroup.asset.distribution.service.jpa.entity.AidLoanFundAuditOrder;
import com.quantgroup.asset.distribution.service.jpa.entity.AidLoanFundConfig;
import com.quantgroup.asset.distribution.service.jpa.entity.AidLoanFundRouteRecord;
import com.quantgroup.asset.distribution.service.jpa.entity.Asset;
import com.quantgroup.asset.distribution.service.jpa.repository.ICustomerTypeRuleConfigRepository;
import com.quantgroup.asset.distribution.service.niwodai.INiwodaiAssetService;
import com.quantgroup.asset.distribution.service.redis.IRedisService;
import com.quantgroup.asset.distribution.service.rule.IRuleService;
import com.quantgroup.asset.distribution.util.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.quantgroup.asset.distribution.constant.RedisKeyConstants.*;

/**
 * Created by renfeng on 2019/7/19.
 */
@Service
@Slf4j
public class AidFundRouteServiceImpl implements IAidFundRouteService {


    @Autowired
    private IAidLoanFundConfigService iAidLoanFundConfigService;
    @Autowired
    private IAidFundRouteRecordService iAidFundRouteRecordService;
    @Autowired
    private IRedisService<String> redisService;
    @Autowired
    private IRedisService<AidLoanFundRouteRecord> redisRouteRecord;
    @Autowired
    private IRuleService ruleService;
    @Autowired
    private ICustomerTypeRuleConfigRepository iCustomerTypeRuleConfigRepository;
    @Autowired
    private IAidFundAssetService aidFundAssetService;
    @Autowired
    private IAidFundAuditOrderService aidFundAuditOrderService;

    /**
     * 助贷资金路由
     *
     * @param assetForm
     * @param userLoanType
     * @return
     */
    @Override
    public boolean aidFundRoute(AssetForm assetForm, Asset asset, Map<String, Object> data) {


        Stopwatch started = Stopwatch.createStarted();
        //第一步 查询所有助贷资金
        List<AidLoanFundConfig> aidLoanFundConfigList = iAidLoanFundConfigService.findAll();
        log.info("助贷资金路由开始, uuid : {} , bizNo : {} ,助贷资金池 : {} ",assetForm.getUuid(),assetForm.getBizNo(),JSON.toJSONString(aidLoanFundConfigList));

        //第二步 开关筛选
        if(CollectionUtils.isNotEmpty(aidLoanFundConfigList))
            aidLoanFundConfigList = aidLoanFundConfigList.parallelStream().filter(aidLoanFundConfig->aidLoanFundConfig.getFundSwitchState()==1).collect(Collectors.toList());
        //第三步 限制次数筛选
        if(CollectionUtils.isNotEmpty(aidLoanFundConfigList)){
            // AID_LOAN_COUNT_LIMIT_KEY + DateUtil.getDay() //TODO 放redis
            aidLoanFundConfigList = aidLoanFundConfigList.parallelStream()
                    .filter(aidLoanFundConfig->aidLoanFundConfig.getFundCountLimit() >=redisService.setIncr(AID_LOAN_COUNT_LIMIT_KEY + DateUtil.getDay(), 0, 3, TimeUnit.DAYS))
                    .collect(Collectors.toList());
        }
        //第四步 限制总额度筛选
        if(CollectionUtils.isNotEmpty(aidLoanFundConfigList)){
            // 获取已放款总金额  AID_LOAN_ALL_AMOUNT_LIMIT_KEY + DateUtil.getDay() //TODO 放redis
            String allAmount = redisService.getString(AID_LOAN_ALL_AMOUNT_LIMIT_KEY + DateUtil.getDay());
            aidLoanFundConfigList = aidLoanFundConfigList.parallelStream()
                    .filter(aidLoanFundConfig->new BigDecimal(aidLoanFundConfig.getFundAllAmountLimit()).compareTo(new BigDecimal(StringUtils.isEmpty(allAmount)?"0":allAmount))>0)
                    .collect(Collectors.toList());
        }
        //第五步 开始路由
        if(CollectionUtils.isNotEmpty(aidLoanFundConfigList)){

            log.info("助贷资金优先级筛选开始, uuid : {} , bizNo : {} ,助贷资金池 : {} ",assetForm.getUuid(),assetForm.getBizNo(),JSON.toJSONString(aidLoanFundConfigList));
            //按优先级排序
            TreeSet<AidLoanFundConfig> aidLoanFundConfigSet = new TreeSet<AidLoanFundConfig>(Comparator.comparing(AidLoanFundConfig::getFundPriority));
            aidLoanFundConfigSet.addAll(aidLoanFundConfigList);

            for(AidLoanFundConfig aidLoanFundConfig : aidLoanFundConfigSet){

                //此助贷资方如果已经被这笔订单路由过  就跳过
                AidLoanFundRouteRecord aidLoanFundRouteRecord = iAidFundRouteRecordService.findByBizNoAndFundId(assetForm.getBizNo(), aidLoanFundConfig.getFundId());
                int status = aidLoanFundRouteRecord == null ? -1 : aidLoanFundRouteRecord.getAidFundRouteStatus().intValue();
                if (status == AidFundStatus.Route.INCOMING_COMPLETE || status == AidFundStatus.Route.PRE_REJECT) {
                    // 进件完成和准入拒绝的直接跳过
                    continue;
                }
                //助贷资金分配规则校验
                if(ruleService.valid(aidLoanFundConfig.getFundRuleEl(), data)){
                    // 助贷资金准入接口调用
                    boolean accessResult = false;
                    if (status == AidFundStatus.Route.PRE_PASS) {
                        accessResult = true;
                    } else {
                        accessResult = aidFundAssetService.preAudit(assetForm, aidLoanFundConfig.getFundId());
                        // 保存主贷资金路由记录
                        aidLoanFundRouteRecord = new AidLoanFundRouteRecord();
                        aidLoanFundRouteRecord.setAidFundRouteStatus(accessResult ? AidFundStatus.Route.PRE_PASS : AidFundStatus.Route.PRE_REJECT);//1-准入成功 2-准入失败 3-进件完成
                        aidLoanFundRouteRecord.setAssetNo(assetForm.getAssetNo());
                        aidLoanFundRouteRecord.setUuid(assetForm.getUuid());
                        aidLoanFundRouteRecord.setBizChannel(assetForm.getBizChannel());
                        aidLoanFundRouteRecord.setBizNo(assetForm.getBizNo());
                        aidLoanFundRouteRecord.setFinanceProductType(Integer.parseInt(assetForm.getBizType()));
                        aidLoanFundRouteRecord.setUserLoanType(asset.getUserLoanType());
                        aidLoanFundRouteRecord.setFundNo(aidLoanFundConfig.getFundNo());
                        aidLoanFundRouteRecord.setFundId(aidLoanFundConfig.getFundId());
                        aidLoanFundRouteRecord.setEnable(true);
                        aidLoanFundRouteRecord = iAidFundRouteRecordService.saveAidLoanFundRouteRecord(aidLoanFundRouteRecord);

                        log.info("助贷资金路由-准入完成, bizChannel : {} , uuid : {} , bizNo : {} , fundId : {} , 准入结果 : {} , 耗时 : {}",assetForm.getBizChannel(),assetForm.getUuid(),assetForm.getBizNo(),aidLoanFundRouteRecord.getFundId(),accessResult,started.elapsed(TimeUnit.MILLISECONDS));
                    }

                    if(accessResult){
                        //客户类别区分 1-自然流量 2-拒绝流量   留在这里 你我贷暂时不用   等以后要用时 随时可用
//                        int customerType = 0 ;
//                        List<CustomerTypeRuleConfig> byEnableTrue = iCustomerTypeRuleConfigRepository.findByEnableTrue();
//                        if(CollectionUtils.isNotEmpty(byEnableTrue)) {
//                            for (CustomerTypeRuleConfig customerTypeRuleConfig : byEnableTrue) {
//                                if (ruleService.valid(customerTypeRuleConfig.getCustomerTypeRuleEl(), data)) {
//                                    customerType = customerTypeRuleConfig.getCustomerType();
//                                }
//                            }
//                        }
                        // 先创建进件订单记录
                        AidLoanFundAuditOrder aidLoanFundAuditOrder = createAidFundAuditOrder(aidLoanFundRouteRecord, assetForm, asset, data);
                        // 调用助贷资方进件接口 异步
                        boolean incomingResult = aidFundAssetService.audit(assetForm, data, asset.getUserLoanType(), aidLoanFundConfig.getFundId());
                        // 将助贷路由记录状态改为进件完成
                        iAidFundRouteRecordService.updateAidLoanFundRouteRecordStatus(aidLoanFundRouteRecord, AidFundStatus.Route.INCOMING_COMPLETE);
                        log.info("助贷资金路由-进件完成, bizChannel : {} , uuid : {} , bizNo : {} , fundId : {} , 进件结果 : {}, 耗时 : {} ",assetForm.getBizChannel(),assetForm.getUuid(),assetForm.getBizNo(),aidLoanFundRouteRecord.getFundId(),incomingResult,started.stop().elapsed(TimeUnit.MILLISECONDS));

                        if (incomingResult) {
                            aidFundAuditOrderService.updateOrderStatus(aidLoanFundAuditOrder, AidFundStatus.Incoming.WAIT);
                            return true;
                        } else {
                            aidFundAuditOrderService.updateOrderStatus(aidLoanFundAuditOrder, AidFundStatus.Incoming.REJECT);
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * 创建资方审核订单
     * @param aidLoanFundRouteRecord
     * @param assetForm
     * @param asset
     * @param data
     */
    private AidLoanFundAuditOrder createAidFundAuditOrder(AidLoanFundRouteRecord aidLoanFundRouteRecord, AssetForm assetForm, Asset asset, Map<String, Object> data) {
        AidLoanFundAuditOrder aidLoanFundAuditOrder = aidFundAuditOrderService.findByBizNoAndFundId(assetForm.getBizNo(), aidLoanFundRouteRecord.getFundId());
        if (aidLoanFundAuditOrder != null) {
            log.info("助贷资方审核订单已存在, 直接返回订单, uuid : {}, bizNo : {}, assetNo : {}, bizChannel : {}, orderStatus : {}", assetForm.getUuid(), assetForm.getBizNo(),
                    assetForm.getAssetNo(), assetForm.getBizChannel(), aidLoanFundAuditOrder.getAuditResult());
            return aidLoanFundAuditOrder;
        }
        aidLoanFundAuditOrder = new AidLoanFundAuditOrder();
        aidLoanFundAuditOrder.setAssetNo(asset.getAssetNo());
        aidLoanFundAuditOrder.setFundNo(aidLoanFundRouteRecord.getFundNo());
        aidLoanFundAuditOrder.setUuid(assetForm.getUuid());
        aidLoanFundAuditOrder.setBizNo(assetForm.getBizNo());
        aidLoanFundAuditOrder.setFundId(aidLoanFundRouteRecord.getFundId());
        aidLoanFundAuditOrder.setAssetFormText(JSON.toJSONString(assetForm));
        aidLoanFundAuditOrder.setAssetText(JSON.toJSONString(asset));
        aidLoanFundAuditOrder.setDataText(JSON.toJSONString(data));
        aidLoanFundAuditOrder.setAuditResult(AidFundStatus.Incoming.PRE);
        aidLoanFundAuditOrder.setEnable(true);
        return aidFundAuditOrderService.saveAidFundAuditOrder(aidLoanFundAuditOrder);
    }
}
