package com.quantgroup.asset.distribution.service.asset.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Stopwatch;
import com.quantgroup.asset.distribution.constant.AssetAttributeConstants;
import com.quantgroup.asset.distribution.exception.QGException;
import com.quantgroup.asset.distribution.exception.QGExceptionType;
import com.quantgroup.asset.distribution.exception.QGPreconditions;
import com.quantgroup.asset.distribution.model.form.AssetForm;
import com.quantgroup.asset.distribution.service.asset.IAssetAttributeService;
import com.quantgroup.asset.distribution.service.httpclient.IHttpService;
import com.quantgroup.asset.distribution.service.jpa.entity.Asset;
import com.quantgroup.asset.distribution.service.jpa.entity.AssetAttributeExtend;
import com.quantgroup.asset.distribution.service.jpa.entity.AssetAttributeExtendConfig;
import com.quantgroup.asset.distribution.service.jpa.repository.IAssetAttributeExtendRepository;
import com.quantgroup.asset.distribution.service.jpa.repository.IAssetRepository;

import lombok.extern.slf4j.Slf4j;

/**
 * 资产扩展属性Service
 * 
 * @author liwenbin
 *
 */
@Slf4j
@Service
public class AssetAttributeServiceImpl implements IAssetAttributeService {
	
	@Autowired
	private IHttpService httpService;
	@Autowired
	private IAssetRepository assetRepository;
	@Autowired
	private IAssetAttributeExtendRepository assetAttributeExtendRepository;
	
	@Value("${rule.engine.url}")
	private String ruleEngineURL;

	/**
	 * 获取所有资产扩展属性value
	 */
	@Override
	public Map<String, Object> getAllAssetAttributeValue(List<AssetAttributeExtendConfig> assetAttributeExtendConfigList, AssetForm assetForm) {
		Stopwatch stopwatch = Stopwatch.createStarted();
		Map<String, Object> data = new HashMap<>();
		// 决策特征Key
		Set<String> decKeys = new HashSet<>();
		decKeys.add(AssetAttributeConstants.USER_LOAN_TYPE);
		if (assetAttributeExtendConfigList != null && assetAttributeExtendConfigList.size() > 0) {
			for (AssetAttributeExtendConfig config : assetAttributeExtendConfigList) {
				if (config.getAssetAttributeType() == 1) { decKeys.add(config.getAssetAttributeCode()); }
			}
		}
		// 请求决策特征
		Map<String, Object> decAttributeValue = getDecAttributeValue(decKeys, assetForm);
		data.putAll(decAttributeValue);
		log.info("用户所有资产扩展属性获取完成, uuid : {}, assetNo : {}, bizNo : {}, bizType : {}, 耗时 : {}", assetForm.getUuid(), assetForm.getAssetNo(), assetForm.getBizNo(), assetForm.getBizType(), stopwatch.stop().elapsed(TimeUnit.MILLISECONDS));
		return data;
	}
	
	/**
	 * 获取所有决策特征属性值
	 * @param decKeys
	 * @return
	 */
	public Map<String, Object> getDecAttributeValue(Set<String> decKeys, AssetForm assetForm) {
		if (CollectionUtils.isEmpty(decKeys)) { return MapUtils.EMPTY_MAP; }
		Stopwatch stopwatch = Stopwatch.createStarted();
		String result = httpService.post(ruleEngineURL + "/feature/get", new HashMap<String, String>(){{
			put("uuid", assetForm.getUuid());
			put("bizChannel", assetForm.getBizChannel());
			put("bizNo", assetForm.getBizNo());
			put("bizType", assetForm.getBizType());
			put("keys", StringUtils.join(decKeys, ","));
			put("method", "0");
		}});
		JSONObject resultJSON = null;
		QGPreconditions.checkArgument(StringUtils.isNotEmpty(result) && (resultJSON = JSON.parseObject(result)).getInteger("code") == 0, QGExceptionType.GET_DEC_ATTRIBUTE_VALUE_ERROR, assetForm.getUuid(), JSON.toJSONString(decKeys));
		Map<String, Object > data = resultJSON.getJSONObject("body");
		log.info("决策特征属性获取完成, uuid : {}, assetNo : {}, bizChannel : {}, bizNo : {}, bizType : {}, data : {}, 耗时 : {}", assetForm.getUuid(), assetForm.getAssetNo(), assetForm.getBizChannel(), assetForm.getBizNo(), assetForm.getBizType(), JSON.toJSONString(data), stopwatch.stop().elapsed(TimeUnit.MILLISECONDS));
		return data;
	}

	@Transactional(rollbackFor=Exception.class)
	@Override
	public void saveAssetAttrubite(Asset asset, List<AssetAttributeExtendConfig> assetAttributeExtendConfigList,
			Map<String, Object> data) {
		// 用户类型特殊处理
		QGPreconditions.checkArgument(data.get(AssetAttributeConstants.USER_LOAN_TYPE) != null, QGExceptionType.ASSET_ATTRIBUTE_IS_EMPTY, AssetAttributeConstants.USER_LOAN_TYPE);
		asset.setUserLoanType((Integer)data.get(AssetAttributeConstants.USER_LOAN_TYPE));
		assetRepository.save(asset);
		
		if (assetAttributeExtendConfigList != null && assetAttributeExtendConfigList.size() > 0) {
			for (AssetAttributeExtendConfig config : assetAttributeExtendConfigList) {
				QGPreconditions.checkArgument(data.get(config.getAssetAttributeCode()) != null, QGExceptionType.ASSET_ATTRIBUTE_IS_EMPTY, config.getAssetAttributeCode());
				AssetAttributeExtend attributeExtend = new AssetAttributeExtend();
				attributeExtend.setAssetNo(asset.getAssetNo());
				attributeExtend.setAssetAttributeName(config.getAssetAttributeName());
				attributeExtend.setAssetAttributeCode(config.getAssetAttributeCode());
				attributeExtend.setAssetAttributeType(config.getAssetAttributeType());
				attributeExtend.setAssetAttributeValue(String.valueOf(data.get(config.getAssetAttributeCode())));
				attributeExtend.setEnable(true);
				assetAttributeExtendRepository.save(attributeExtend);
			}
		}
	}
}
