package com.quantgroup.asset.distribution.controller;

import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Stopwatch;
import com.quantgroup.asset.distribution.constant.FundModuleConstants;
import com.quantgroup.asset.distribution.enums.response.FundModuleResponse;
import com.quantgroup.asset.distribution.model.response.GlobalResponse;
import com.quantgroup.asset.distribution.service.approval.IApprovalLogService;
import com.quantgroup.asset.distribution.service.funding.IFundModuleService;
import com.quantgroup.asset.distribution.util.fund.module.ChannelFundConfigUtil;

import lombok.extern.slf4j.Slf4j;

/**
 * 资方模块Controller
 * @author liwenbin
 *
 */
@RestController
@Slf4j
@RequestMapping("/fund")
public class FundModuleController {
	
	@Autowired
	private IFundModuleService fundModuleService;
	
	@RequestMapping("/get_all_funds")
	public GlobalResponse getAllFunds() {
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.getAllFundsInfo();
		log.info("资方模块接口, 获取所有资方信息完成, 耗时 : {}, response : {}", stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("/get_all_limit_type")
	public GlobalResponse getAllLimitInfo() {
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.getAllLimitType();
		log.info("资方模块接口, 获取所有条件限制类型完成, 耗时 : {}, response : {}", stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("save_channel_funds_config")
	public GlobalResponse saveChannelFundsConfig(Integer type, Long id, String bizChannel, String funds, String remarks, 
			String auditor, String proposer) {
		log.info("资方模块接口, 新增或修改资方配置开始, type : {}, id : {}, bizChannel : {}, funds : {}, remarks : {}, auditor : {}, proposer : {}", type, id, bizChannel, funds, remarks, auditor, proposer);
		// 参数校验
		if (type == null) {
			return GlobalResponse.create(FundModuleResponse.TYPE_IS_EMPTY);
		}
		if (type != FundModuleConstants.CHANNEL_FUNDS_OPERAOTR_TYPE_ADD && type != FundModuleConstants.CHANNEL_FUNDS_OPERATOR_TYPE_UPDATE) {
			return GlobalResponse.create(FundModuleResponse.UNKNOW_TYPE);
		}
		if (type == 2 && id == null) {
			return GlobalResponse.create(FundModuleResponse.ID_IS_EMPTY);
		}
		if (StringUtils.isEmpty(bizChannel)) {
			return GlobalResponse.create(FundModuleResponse.BIZ_CHANNEL_IS_EMPTY);
		}
		if (StringUtils.isEmpty(funds)) {
			return GlobalResponse.create(FundModuleResponse.FUNDS_INFO_IS_EMPTY);
		}
		if (!ChannelFundConfigUtil.checkFunds(funds)) {
			return GlobalResponse.create(FundModuleResponse.FUNDS_INFO_ERROR);
		}
		if (StringUtils.isEmpty(auditor) || StringUtils.isEmpty(proposer)) {
			return GlobalResponse.create(FundModuleResponse.AUDITOR_OR_PROPOSER_IS_EMPTY);
		}
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.saveChannelFundConfig(type, id, bizChannel, funds, remarks, auditor, proposer);
		log.info("资方模块接口, 新增或修改资方配置结束, type : {}, id : {}, bizChannel : {}, funds : {}, remarks : {}, auditor : {}, proposer : {}, 耗时 : {}, reponse : {}", type, id, bizChannel, funds, remarks, auditor, proposer, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("/get_channel_fund_configs")
	public GlobalResponse getChannelFundConfigs(String bizChannel, Long fundId, Integer pageNum, Integer pageSize) {
		log.info("资方模块接口, 查询渠道资方配置信息开始, bizChannel : {}, fundId : {}, pageNum : {}, pageSize : {}", bizChannel, fundId, pageNum, pageSize);
		if (pageNum == null || pageSize == null) {
			return GlobalResponse.create(FundModuleResponse.PAGEING_CONDITIONS_IS_EMPTY);
		}
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.getChannelFundConfigs(bizChannel, fundId, pageNum, pageSize);
		log.info("资方模块接口, 查询渠道资方配置信息结束, bizChannel : {}, fundId : {}, pageNum : {}, pageSize : {}, 耗时 : {}, response : {}", bizChannel, fundId, pageNum, pageSize, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("/get_audit_infos")
	public GlobalResponse getAuditInfos(String targetName, Integer auditStatus, Integer auditType, Integer auditTarget, String applyStartTime, String applyEndTime, String user, Integer pageNum, Integer pageSize) {
		log.info("资方模块接口, 获取审批列表开始, targetName : {}, auditStatus : {}, auditType : {}, auditTarget : {}， applyStartTime : {}, applyEndTime : {}, user : {}, pageNum : {}, pageSize : {}", targetName, auditStatus, auditType, auditTarget, applyStartTime, applyEndTime, user, pageNum, pageSize);
		if (pageNum == null || pageSize == null) {
			return GlobalResponse.create(FundModuleResponse.PAGEING_CONDITIONS_IS_EMPTY);
		}
		if (StringUtils.isEmpty(user)) {
			return GlobalResponse.create(FundModuleResponse.USER_IS_EMPTY);
		}
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.getAuditInfos(targetName, auditStatus, auditType, auditTarget, applyStartTime, applyEndTime, user, pageNum, pageSize);
		log.info("资方模块接口, 获取审批列表结束, targetName : {}, auditStatus : {}, auditType : {}, auditTarget : {}， applyStartTime : {}, applyEndTime : {}, user : {}, pageNum : {}, pageSize : {}， 耗时 : {}, response : {}", targetName, auditStatus, auditType, auditTarget, applyStartTime, applyEndTime, user, pageNum, pageSize, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("/find_channel_fund_config")
	public GlobalResponse findChannelFundConfig(Long configId) {
		log.info("资方模块接口，根据id获取资方配置, configId : {}", configId);
		if (configId == null) {
			return GlobalResponse.create(FundModuleResponse.ID_IS_EMPTY);
		}
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.findChannelFundConfigById(configId);
		log.info("资方模块接口, 根据id获取资方配置, configId : {} 耗时 : {}, response : {}", configId, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
	
	@RequestMapping("/audit")
	public GlobalResponse audit(Long id, Integer auditStatus) {
		log.info("资方模块接口，审批执行开始, id : {}, auditStatus : {}", id, auditStatus);
		if (id == null) {
			return GlobalResponse.create(FundModuleResponse.ID_IS_EMPTY);
		}
		if (auditStatus == null) {
			return GlobalResponse.create(FundModuleResponse.AUDIT_STATUS_IS_EMPTY);
		}
		Stopwatch stopwatch = Stopwatch.createStarted();
		GlobalResponse response = fundModuleService.audit(id, auditStatus);
		log.info("资方模块接口，审批执行开始, id : {}, auditStatus : {}, 耗时 : {}, response : {}", id, auditStatus, stopwatch.stop().elapsed(TimeUnit.MILLISECONDS), JSON.toJSONString(response));
		return response;
	}
}
