package com.quantgroup.asset.distribution.service.rule.vo;

import com.alibaba.fastjson.annotation.JSONField;
import com.alibaba.fastjson.annotation.JSONType;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import com.google.common.primitives.Doubles;
import com.quantgroup.asset.distribution.enums.RuleOperator;
import com.quantgroup.asset.distribution.exception.QGException;
import com.quantgroup.asset.distribution.exception.QGExceptionType;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;

@Data
@AllArgsConstructor
@NoArgsConstructor
@JSONType(typeName = "base", serialzeFeatures = SerializerFeature.WriteEnumUsingToString)
@Slf4j
public class BaseRuleVO implements IRuleVO, Serializable {
    private static final long serialVersionUID = -1L;
    public static final String IN_FORMAT = "%s in (%s)";
    public static final String NOT_IN_FORMAT = "%s not in (%s)";
    private String key;
    private String operator;
    @JSONField(name = "value")
    private String value;

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BaseRuleVO that = (BaseRuleVO) o;
        return Objects.equal(key, that.key) &&
                Objects.equal(operator, that.operator) &&
                Objects.equal(value, that.value);
    }

    @Override
    public String format() {
        switch (RuleOperator.fromCode(operator)) {
            case In:
                return String.format(IN_FORMAT, key, value);
            case NotIn:
                return String.format(NOT_IN_FORMAT, key, value);
            default:
                return key + operator + value;
        }
    }

    @Override
    public boolean valid(Map<String, Object> data) {
        if(data == null || !data.containsKey(key)){
            log.info("规则数据不存在,key:{},operator:{},value:{}",key,operator,value);
            throw new QGException(QGExceptionType.DATA_NOT_EXIST_FOR_RULE,key,operator,value);
        }
        if(operator == null){
            log.info("规则符为空,key:{},value:{}",key,value);
            throw new QGException(QGExceptionType.RULE_OPERATOR_IS_EMPTY);
        }
        Object object = data.get(key);
        if(object == null){
            log.info("规则数据为空,key:{},operator:{},value:{}",key,operator,value);
            throw new QGException(QGExceptionType.RULE_DATA_IS_EMPTY,key,operator,value);
        }
        return compareCommon(object);
    }

    @Override
    public Map<String, Boolean> valid2(Map<String, Object> data) {
        return ImmutableMap.of(this.format(),compareCommon(data.get(key)));
    }

    /**
     * 比较
     * @param object
     * @return
     */
    private boolean compareCommon(Object object){
        try {
            if (object instanceof Integer)
                return Doubles.tryParse(value) == null ? operateInt((Integer) object) : operateBigDecimal(BigDecimal.valueOf(((Integer) object)));
            if (object instanceof Float)
                return operateBigDecimal(BigDecimal.valueOf(((Float) object)));
            if (object instanceof Double)
                return operateBigDecimal(BigDecimal.valueOf(((Double) object)));
            if (object instanceof BigDecimal)
                return operateBigDecimal((BigDecimal) object);

            //防止 仿真把数字传成字符串
            try {
                return operateBigDecimal(BigDecimal.valueOf((Double.parseDouble(String.valueOf(object)))));
            } catch (NumberFormatException e) {
                return operateString(String.valueOf(object));
            }
        } catch (Exception e) {
            log.error("数据比较出现错误,key:{}, 配置数据:{}, 待比较的数据:{}", key, value, object, e);
            throw new QGException(QGExceptionType.DATA_COMPARE_ERROR);
        }
    }


    @Override
    public String getId() {
        return null;
    }

    private boolean operateInt(int data) {
        switch (RuleOperator.fromCode(operator)) {
            case Equal:
                return Integer.parseInt(value) == data;
            case In:
                return Arrays.stream(StringUtils.split(value, ',')).filter(StringUtils::isNumeric).map(Integer::parseInt).anyMatch(i -> data == i);
            case NotIn:
                return Arrays.stream(StringUtils.split(value, ',')).filter(StringUtils::isNumeric).map(Integer::parseInt).noneMatch(i -> data == i);
            case LessThan:
                return data < Integer.parseInt(value);
            case NotEqual:
                return data != Integer.parseInt(value);
            case GreaterThan:
                return data > Integer.parseInt(value);
            case LessThanOrEqual:
                return data <= Integer.parseInt(value);
            case GreaterThanOrEqual:
                return data >= Integer.parseInt(value);
            default:
                throw new QGException(QGExceptionType.RULE_OPERATOR_NOT_EXIST);
        }

    }

    private boolean operateBigDecimal(BigDecimal data) {
        switch (RuleOperator.fromCode(operator)) {
            case Equal:
                return new BigDecimal(value).compareTo(data) == 0;
            case In:
                return Arrays.stream(StringUtils.split(value, ',')).filter(StringUtils::isNumeric).map(BigDecimal::new).anyMatch(o -> o.compareTo(data) == 0);
            case NotIn:
                return Arrays.stream(StringUtils.split(value, ',')).filter(StringUtils::isNumeric).map(BigDecimal::new).noneMatch(o -> o.compareTo(data) == 0);
            case LessThan:
                return new BigDecimal(value).compareTo(data) > 0;
            case NotEqual:
                return new BigDecimal(value).compareTo(data) != 0;
            case GreaterThan:
                return new BigDecimal(value).compareTo(data) < 0;
            case LessThanOrEqual:
                return new BigDecimal(value).compareTo(data) >= 0;
            case GreaterThanOrEqual:
                return new BigDecimal(value).compareTo(data) <= 0;
            default:
                throw new QGException(QGExceptionType.RULE_OPERATOR_NOT_EXIST);
        }

    }

    private boolean operateString(String data) {
        switch (RuleOperator.fromCode(operator)) {
            case Equal:
                return StringUtils.equals(StringUtils.replaceAll(value, "\"", ""), data);
            case In:
                return Arrays.stream(StringUtils.split(StringUtils.replaceAll(value, "\"", ""), ',')).anyMatch(data::equals);
            case NotIn:
                return Arrays.stream(StringUtils.split(StringUtils.replaceAll(value, "\"", ""), ',')).noneMatch(data::equals);
            case LessThan:
                return StringUtils.compare(data, StringUtils.replaceAll(value, "\"", "")) < 0;
            case NotEqual:
                return !StringUtils.equals(data, StringUtils.replaceAll(value, "\"", ""));
            case GreaterThan:
                return StringUtils.compare(data, StringUtils.replaceAll(value, "\"", "")) > 0;
            case LessThanOrEqual:
                return StringUtils.compare(data, StringUtils.replaceAll(value, "\"", "")) <= 0;
            case GreaterThanOrEqual:
                return StringUtils.compare(data, StringUtils.replaceAll(value, "\"", "")) >= 0;
            default:
                throw new QGException(QGExceptionType.RULE_OPERATOR_NOT_EXIST);
        }

    }

    @Override
    public void checkParams() {

    }

    @Override
    public Set<String> getParamNames() {
        return Sets.newHashSet(key);
    }
}