package com.quantgroup.asset.distribution.service.distribute.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Stopwatch;
import com.quantgroup.asset.distribution.constant.DistributeConstants;
import com.quantgroup.asset.distribution.constant.RedisKeyConstants;
import com.quantgroup.asset.distribution.constant.StatusConstants;
import com.quantgroup.asset.distribution.enums.funding.FundingResult;
import com.quantgroup.asset.distribution.exception.QGException;
import com.quantgroup.asset.distribution.exception.QGExceptionType;
import com.quantgroup.asset.distribution.model.entity.DistributeRecord;
import com.quantgroup.asset.distribution.model.form.AssetForm;
import com.quantgroup.asset.distribution.model.response.GlobalResponse;
import com.quantgroup.asset.distribution.service.alarm.IAlarmService;
import com.quantgroup.asset.distribution.service.distribute.IAssetDistributeRecordService;
import com.quantgroup.asset.distribution.service.distribute.IAssetDistributeRuleConfigService;
import com.quantgroup.asset.distribution.service.distribute.IAssetDistributeService;
import com.quantgroup.asset.distribution.service.funding.IAidFundRouteService;
import com.quantgroup.asset.distribution.service.jpa.entity.Asset;
import com.quantgroup.asset.distribution.service.jpa.entity.AssetDistributeRecord;
import com.quantgroup.asset.distribution.service.jpa.entity.AssetDistributeRuleConfig;
import com.quantgroup.asset.distribution.service.notify.INotifyService;
import com.quantgroup.asset.distribution.service.redis.IRedisService;
import com.quantgroup.asset.distribution.service.rule.IRuleService;

import lombok.extern.slf4j.Slf4j;

/**
 * 资产分发Service
 * @author liwenbin
 *
 */
@Slf4j
@Service
public class AssetDistributeServiceImpl implements IAssetDistributeService{
	
	/**
	 * 分发总开关
	 */
	@Value("${is.distribute}")
    private Boolean isDistribute;
	@Autowired
	private IAssetDistributeRuleConfigService assetDistributeRuleConfigService;
	@Autowired
	private IRedisService<Integer> redisService;
	@Autowired
	private IRuleService ruleService;
	@Autowired
	private IAidFundRouteService aidFundRouteService;
	@Autowired
	private INotifyService notifyService;
	@Autowired
	private IAssetDistributeRecordService assetDistributeRecordService;
	@Autowired
	private IAlarmService alarmService;
	
	/**
	 * 分发
	 */
	@Override
	public void distribute(AssetForm assetForm, Asset asset, Map<String, Object> data) {
		if (isDistribute) {
			List<AssetDistributeRuleConfig> assetDistributeRuleConfigList = assetDistributeRuleConfigService.getAllRuleConfigOrderByPriorityAsc();
			// 检查规则配置
			checkDistributeRuleConfig(assetDistributeRuleConfigList);
			// 获取该资产所有已经尝试分发过的类型
			Set<Integer> hasDistributedTypeSet = getAllDistributeRecordType(asset.getAssetNo());
			boolean success = false;
			DistributeRecord record = new DistributeRecord(data); 
			for (AssetDistributeRuleConfig ruleConfig : assetDistributeRuleConfigList) {
				Integer ruleType = ruleConfig.getAssetDistributeRuleType();
				// 如果配置被关闭或者该类型被分发过，就默认往下走，主要避免priority调换顺序的情况
				if (ruleConfig.getAssetDistributeRuleSwitch() == 0 || hasDistributedTypeSet.contains(ruleType)) { continue; } 
				String el = ruleConfig.getAssetDistributeRuleEl();
				boolean valid = ruleService.valid(el, data);
				record.addRecords(ruleConfig.getId(), el, ruleType, ruleConfig.getAssetDistributeRuleName(), valid);
				if (valid) {
					// 先创建记录，避免mq里消息来了就还没创建
					AssetDistributeRecord assetDistributeRecord = assetDistributeRecordService.saveDistributeRecord(asset, record, StatusConstants.WAIT, ruleType);
					int distributeStatus = beginDistribute(assetForm, asset, ruleType, assetDistributeRecord, data);
					// 分发成功
					if (distributeStatus == StatusConstants.SUCCESS || distributeStatus == StatusConstants.WAIT) {
						log.info("用户执行分发节点, uuid : {}, bizNo : {}, assetNo : {}, bizChannel : {}, ruleId : {}, ruleName : {}, distributeStatus : {}", 
								assetForm.getUuid(), assetForm.getBizNo(), assetForm.getAssetNo(), assetForm.getBizChannel(), ruleConfig.getId(), ruleConfig.getAssetDistributeRuleName(),
								distributeStatus);
						success = true;
					}
				}
				// 保存节点尝试记录
				saveAssetAttemptDistributeNode(asset.getAssetNo(), ruleType);
				if (success) { break; }
			}
			if (!success) {
				// 如果一个分配节点都没找到，报错
				throw new QGException(QGExceptionType.NO_DISTRIBUTE_NODE, asset.getUuid(), asset.getAssetNo(), JSON.toJSONString(record));
			}
		} else {
			log.info("资产分发开关关闭，直接通知资金系统, uuid : {}, bizNo : {}, assetNo : {}, bizType : {}, bizChannel : {}", 
					assetForm.getUuid(), assetForm.getBizNo(), assetForm.getAssetNo(), assetForm.getBizType(), assetForm.getBizChannel());
			notifyService.notifyFundServer(assetForm, data);
		}
	}
	
	/**
	 * 开始进行分发
	 * @param assetForm
	 * @param ruleType
	 * @return 1:成功，2:失败， 3:分配中
	 */
	public int beginDistribute(AssetForm assetForm, Asset asset, int ruleType, AssetDistributeRecord assetDistributeRecord, Map<String, Object> data) {
		try {
			switch (ruleType) {
				case DistributeConstants.RuleType.FUND_ROUTE:
					notifyService.notifyFundServer(assetForm, data);
					return StatusConstants.WAIT;
				case DistributeConstants.RuleType.AID_FUND_ROUTE: {
					GlobalResponse response = aidFundRouteService.aidFundRoute(assetForm, asset.getUserLoanType(), data);
					int status = response.getCode() == 0 ? StatusConstants.SUCCESS : StatusConstants.FAIL;
					// 助贷资金路由目前不从mq里接受终态
					assetDistributeRecord.setAssetDistributeStatus(status);
					assetDistributeRecordService.updateAssetDistribute(assetDistributeRecord);
					return status;
				}
				case DistributeConstants.RuleType.DIVERSION:
					notifyService.notifyFundServer(assetForm, data);
					return StatusConstants.WAIT;
				default:
					throw new QGException(QGExceptionType.UNKNOW_RULE_TYPE, ruleType);
			}
		} catch (Exception e) {
			// 如果报错，把保存的分发记录置为失效
			assetDistributeRecord.setEnable(false);
			assetDistributeRecordService.updateAssetDistribute(assetDistributeRecord);
			throw e;
		}
	}
	  

	/**
	 * 分发配置检查
	 * @param assetDistributeRuleConfigList
	 */
	public void checkDistributeRuleConfig(List<AssetDistributeRuleConfig> assetDistributeRuleConfigList) {
		int prePriority = -1;
		for (AssetDistributeRuleConfig assetDistributeRuleConfig : assetDistributeRuleConfigList) {
			if (assetDistributeRuleConfig.getAssetDistributeRulePriority().equals(prePriority)) {
				throw new QGException(QGExceptionType.DISTRIBUTE_RULE_CONFIG_PRIORITY_IS_NOT_UNIQUE);
			}
			prePriority = assetDistributeRuleConfig.getAssetDistributeRulePriority();
		}
	}
	
	/**
	 * 获取该用户所有分发记录Type,如果用户初次分发或者异常重试返回空
	 * @param assetNo
	 * @return
	 */
	public Set<Integer> getAllDistributeRecordType(String assetNo) {
		Set<Integer> sets = new HashSet<>();
		List<Integer> list = redisService.getList(RedisKeyConstants.DISTRIBUTE_FAIL_TYPE_RECORD + assetNo);
		if (list == null || list.size() == 0) {
			// 如果缓存没有去查库，把库里执行过的都拿出来
			List<AssetDistributeRecord> assetDistributeRecords = assetDistributeRecordService.getDistributeRecord(assetNo);
			if (assetDistributeRecords == null || assetDistributeRecords.size() == 0) { return sets; }
			list = new ArrayList<>();
			for (AssetDistributeRecord assetDistributeRecord : assetDistributeRecords) {
				JSONArray array = JSON.parseObject(assetDistributeRecord.getAssetDistributeTravel()).getJSONArray("records");
				for (int i = 0, len = array.size(); i < len; ++i) {
					JSONObject record = array.getJSONObject(i);
					list.add(record.getInteger("type"));
				}
			}
		}
		sets.addAll(list);
		if (sets.size() != list.size()) {
			list = list.stream().distinct().collect(Collectors.toList());
			redisService.del(RedisKeyConstants.DISTRIBUTE_FAIL_TYPE_RECORD + assetNo);
			redisService.setListEx(RedisKeyConstants.DISTRIBUTE_FAIL_TYPE_RECORD + assetNo, list, 3, TimeUnit.DAYS);
		}
		return sets;
	}
	
	/**
	 * 保存资产尝试分发记录节点
	 */
	public void saveAssetAttemptDistributeNode(String assetNo, Integer ruleType) {
		redisService.rightPushEx(RedisKeyConstants.DISTRIBUTE_FAIL_TYPE_RECORD + assetNo, ruleType, 3, TimeUnit.DAYS);
	}
	
	/**
	 * 接收fundingResult消息
	 */
	@Override
	public void receiveFundingResult(String bizNo, FundingResult fundingResult, String nextOperateDate) {
		try {
			Stopwatch stopwatch = Stopwatch.createStarted();
			// 1、更改分配记录状态
			assetDistributeRecordService.updateAssetDistributeStatus(bizNo, fundingResult == FundingResult.FUAD_ASSIGN_SUCC ? StatusConstants.SUCCESS : StatusConstants.FAIL);
			// 2、通知业务流系统
			notifyService.notifyBusinessFlow(bizNo, fundingResult, nextOperateDate);
			// 3、重新进行分发, 目前还没有，等接了助贷资金路由以后再增加
			log.info("资产分发接收资金结果处理结束, bizNo : {}, fundingResult : {}, 耗时 : {}", bizNo, fundingResult.getCode(), stopwatch.stop().elapsed(TimeUnit.MILLISECONDS));
		} catch (QGException qe) {
			log.error("资产分发接收资金结果处理出现错误 : {}, bizNo : {}, fundingResult : {}", qe.qgExceptionType.code + "->" + qe.detail, bizNo, fundingResult.getCode()) ;
			alarmService.dingtalkAlarm("Warn", "资产分发接收资金结果处理出现错误", "bizNo : " + bizNo + " , fundingResult : " + fundingResult.getCode() + 
					" , 错误信息 : " + qe.qgExceptionType.code + "->" + qe.detail);
		} catch (Exception e) {
			log.error("资产分发接收资金结果处理异常, bizNo : {}, fundingResult : {}", bizNo, fundingResult.getCode(), e);
			alarmService.dingtalkAlarm("Warn", "资产分发接收资金结果处理出现异常", "bizNo : " + bizNo + " , fundingResult : " + fundingResult.getCode() + 
					" , 错误信息 : 未知错误");
		}
	}
}
