package com.quantgroup.asset.distribution.service.approval.impl;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.criteria.*;

import com.quantgroup.asset.distribution.service.funding.IFundModuleChannelFundConfigNewService;
import com.quantgroup.asset.distribution.service.jpa.entity.FundModuleChannelFundConfigNew;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.jpa.criteria.predicate.InPredicate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.quantgroup.asset.distribution.enums.funding.AuditStatusEnum;
import com.quantgroup.asset.distribution.enums.funding.AuditTargetEnum;
import com.quantgroup.asset.distribution.enums.funding.AuditTypeEnum;
import com.quantgroup.asset.distribution.service.approval.IApprovalLogService;
import com.quantgroup.asset.distribution.service.funding.IFundModuleChannelFundConfigService;
import com.quantgroup.asset.distribution.service.jpa.entity.ApprovalLog;
import com.quantgroup.asset.distribution.service.jpa.entity.FundModuleChannelFundConfig;
import com.quantgroup.asset.distribution.service.jpa.repository.IApprovalLogRepository;
import com.quantgroup.asset.distribution.util.DateUtil;

@Service
public class ApprovalLogServiceImpl implements IApprovalLogService{
	
	@Autowired
	private IApprovalLogRepository approvalLogRepository;
	@Autowired
	private IFundModuleChannelFundConfigNewService fundModuleChannelFundConfigNewService;

	@Override
	public void createApprovalLog(AuditTypeEnum auditType, AuditTargetEnum auditTarget, String targetName,
			String proposer, String auditor, Long preConfigId, Long auditConfigId) {
		ApprovalLog approvalLog = new ApprovalLog();
		approvalLog.setAuditType(auditType.getCode());
		approvalLog.setAuditTarget(auditTarget.getCode());
		approvalLog.setTargetName(targetName);
		approvalLog.setProposer(proposer);
		approvalLog.setApplyTime(new Timestamp(System.currentTimeMillis()));
		approvalLog.setAuditor(auditor);
		approvalLog.setAuditStatus(AuditStatusEnum.WAIT.getCode());
		approvalLog.setPreConfigId(preConfigId);
		approvalLog.setAuditConfigId(auditConfigId);
		approvalLog.setEnable(true);
		approvalLogRepository.save(approvalLog);
	}

	@Override
	public Map<String, Object> getApprovalLogs(String targetName, Integer auditStatus, Integer auditType,
			Integer auditTarget, String applyStartTime, String applyEndTime, String user, Integer pageNum,
			Integer pageSize,Integer sortMode) {
		// 分页条件
		Pageable pageable;
		if (sortMode == 1){
			Sort sort = new Sort(Sort.Direction.ASC,"audit_status").and(new Sort(Sort.Direction.DESC,"audit_time"));
			pageable = new PageRequest(pageNum < 0 ? 0 : pageNum, pageSize, sort);
		}else {
			pageable = new PageRequest(pageNum < 0 ? 0 : pageNum, pageSize);
		}
		Specification<ApprovalLog> specification = new Specification<ApprovalLog>() {
			@Override
			public Predicate toPredicate(Root<ApprovalLog> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				List<Predicate> predicatesAnd = new ArrayList<>();
				List<Predicate> predicatesOr = new ArrayList<>();
				predicatesAnd.add(cb.equal(root.get("enable"), true));
	            if(auditStatus != null){
	            	predicatesAnd.add(cb.equal(root.get("auditStatus"), auditStatus));
	            }else {
	            	if (sortMode == 1){
	            		predicatesAnd.add(cb.in(root.get("auditStatus")).value(0).value(1));
					}
				}
	            if(auditType != null){
	            	predicatesAnd.add(cb.equal(root.get("auditType"), auditType));
	            }
	            if(auditTarget != null){
	            	predicatesAnd.add(cb.equal(root.get("auditTarget"), auditTarget));
	            }
	            if(StringUtils.isNotEmpty(applyStartTime)){
	            	predicatesAnd.add(cb.greaterThanOrEqualTo(root.get("applyTime").as(Timestamp.class), Timestamp.valueOf(applyStartTime)));
	            }
	            if(StringUtils.isNotEmpty(applyEndTime)){
	            	predicatesAnd.add(cb.lessThan(root.get("applyTime").as(Timestamp.class), Timestamp.valueOf(applyEndTime)));
	            }
	            if(StringUtils.isNotEmpty(targetName)){
	            	predicatesAnd.add(cb.like(root.get("targetName"), "%" + targetName + "%"));
	            }
	            if (StringUtils.isNotEmpty(user)) {
	            	predicatesOr.add(cb.equal(root.get("proposer"), user));
	            	predicatesOr.add(cb.equal(root.get("auditor"), user));
	            }
	            List<Order> orderList = new ArrayList<>();
	            orderList.add(cb.asc(root.get("auditStatus")));
                return query.where(cb.and(predicatesAnd.toArray(new Predicate[predicatesAnd.size()])), cb.or(predicatesOr.toArray(new Predicate[predicatesOr.size()])))
                		    .orderBy(orderList).getRestriction();
			}
		};
		Page<ApprovalLog> channelFundConfigs = approvalLogRepository.findAll(specification, pageable);
		Map<String, Object> result = new HashMap<>();
		result.put("total", channelFundConfigs.getTotalElements());
		result.put("pages", channelFundConfigs.getTotalPages());
		result.put("pageSize", channelFundConfigs.getSize());
		result.put("pageNum", channelFundConfigs.getNumber());
		result.put("list", channelFundConfigs.getContent());
		return result;
	}

	@Override
	public List<Long> getFundConfigIds(String channel){
		if (StringUtils.isEmpty(channel)){
			return approvalLogRepository.findAuditConfigIds();
		}else {
			return approvalLogRepository.findAuditConfigIds(channel);
		}
	}

	@Transactional(rollbackFor=Exception.class)
	@Override
	public void audit(ApprovalLog approvalLog, Integer auditStatus) {
		if (auditStatus == AuditStatusEnum.PASS.getCode()) {
			// 先更改审核记录, 这里返回的是新配置
			FundModuleChannelFundConfigNew config = fundModuleChannelFundConfigNewService.auditPassConfig(approvalLog.getPreConfigId(), approvalLog.getAuditConfigId());
			// 根据新配置清楚渠道缓存
			fundModuleChannelFundConfigNewService.clearChannelFundConfigCache(config.getBizChannel());
		}
		updateApprovalLogAuditStatus(approvalLog, auditStatus);
	}

	@Override
	public ApprovalLog findById(Long id) {
		return approvalLogRepository.findByIdAndEnableIsTrue(id);
	}
	
	/**
	 * 更改审批记录状态和审批时间
	 * @param approvalLog
	 * @param auditStatus
	 */
	private void updateApprovalLogAuditStatus(ApprovalLog approvalLog, Integer auditStatus) {
		approvalLog.setAuditTime(DateUtil.getCurDateTime());
		approvalLog.setAuditStatus(auditStatus);
		approvalLogRepository.save(approvalLog);
	}

	@Override
	public Boolean isAuditing(String bizChannel) {
		return approvalLogRepository.findByAuditTypeAndAuditTargetAndTargetNameAndAuditStatusAndEnableIsTrue(AuditTypeEnum.ONLINE.getCode(),
				AuditTargetEnum.FUND_CONFIG.getCode(), bizChannel, AuditStatusEnum.WAIT.getCode()) != null;
	}
}
