package cn.quantgroup.financial.service.sys.impl;

import cn.quantgroup.financial.constant.EncodingConfig;
import cn.quantgroup.financial.dao.ISystemDao;
import cn.quantgroup.financial.model.MailInfo;
import cn.quantgroup.financial.service.sys.IMailService;
import cn.quantgroup.financial.util.DateUtil;
import cn.quantgroup.sms.IMailSendCallback;
import cn.quantgroup.sms.MailSendBean;
import cn.quantgroup.sms.SendCloudMailSender;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.AllowAllHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.cookie.Cookie;
import org.apache.http.cookie.CookieOrigin;
import org.apache.http.cookie.CookieSpecProvider;
import org.apache.http.cookie.MalformedCookieException;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.StandardHttpRequestRetryHandler;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.impl.cookie.BestMatchSpecFactory;
import org.apache.http.impl.cookie.BrowserCompatSpec;
import org.apache.http.impl.cookie.BrowserCompatSpecFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.activation.DataSource;
import javax.mail.util.ByteArrayDataSource;
import javax.net.ssl.SSLContext;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;


@Service
public class MailServiceImpl implements IMailService {


    private static final SendCloudMailSender sendCloudMailSender = new SendCloudMailSender(httpClient());

    @Autowired
    private ISystemDao systemDao;

    @Value("${hubei.history.filepath}")
    private String hubeiHistoryFilePath;


    private static final Logger logger = LoggerFactory.getLogger(MailServiceImpl.class);


    /**
     * Create HttpClient
     *
     * @return
     * @throws KeyStoreException
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static CloseableHttpClient httpClient(){
        try {
            //  socket factory
            ConnectionSocketFactory plainSocketFactory = new PlainConnectionSocketFactory();
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(
                    null, (x509Certificates, authType) -> true).build();

            LayeredConnectionSocketFactory sslSocketFactory =
                    new SSLConnectionSocketFactory(sslContext, new AllowAllHostnameVerifier());
            Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", plainSocketFactory)
                    .register("https", sslSocketFactory).build();

            // cookie specification
            Registry<CookieSpecProvider> cookieSpecProviderRegistry = RegistryBuilder.<CookieSpecProvider>create()
                    .register(CookieSpecs.BEST_MATCH, new BestMatchSpecFactory())
                    .register(CookieSpecs.BROWSER_COMPATIBILITY, new BrowserCompatSpecFactory())
                    .register("easy", httpContext -> new BrowserCompatSpec() {
                        public void validate(Cookie cookie, CookieOrigin origin) throws MalformedCookieException {
                        }
                    }).build();

            // connection manager
            PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);

            connectionManager.setMaxTotal(10000);
            connectionManager.setDefaultMaxPerRoute(1000);

            // retry handler
            HttpRequestRetryHandler retryHandler = new StandardHttpRequestRetryHandler(3, false);

            // keep alive strategy
            ConnectionKeepAliveStrategy keepAliveStrategy = new DefaultConnectionKeepAliveStrategy();

            // httpclient
            return HttpClients.custom()
                    .setConnectionManager(connectionManager)
                    .setRetryHandler(retryHandler)
                    .setKeepAliveStrategy(keepAliveStrategy)
                    .setDefaultCookieSpecRegistry(cookieSpecProviderRegistry).build();
        } catch (NoSuchAlgorithmException e) {
            logger.error(e.getMessage(),e);
        } catch (KeyManagementException e) {
            logger.error(e.getMessage(),e);
        } catch (KeyStoreException e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    @Override
    public void sendAttachmentMailAsync(String from, String[] sendTo, String[] ccTo, String subject, String text, ArrayList<String> attachedFileList, IMailSendCallback mailSendCallback) {

        try {
            if(!ArrayUtils.isEmpty(sendTo)){
                for(String to:sendTo){
                    HashMap<String,File> attachments = new HashMap<String,File>();
                    if (CollectionUtils.isNotEmpty(attachedFileList)) {
                        for (String filePath : attachedFileList) {
                            File attachedFile = hasValidAttachent(filePath);
                            attachments.put(attachedFile.getName(),attachedFile);
                        }
                    }
                    MailSendBean mailSendBean = MailSendBean.newDefaultBuilder()
                            .from(from)
                            .to(to)
                            .subject(subject)
                            .fromName("财务系统")
                            .labelId(0)
                            .plain(text)
                            .attachments(attachments)
                            .build();
                    if(attachments.isEmpty()){
                        sendCloudMailSender.sendSimpleMailAsync(mailSendBean,mailSendCallback);
                    }else {
                        sendCloudMailSender.sendAttachmentMailAsync(mailSendBean,mailSendCallback);
                    }
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }


    @Override
    public void sendAttachmentMailAsync(String from, String sendTo, String[] ccTo, String subject, String text, ArrayList<File> attachedFileList, IMailSendCallback mailSendCallback) {

        try {
            if(!org.apache.commons.lang.StringUtils.isEmpty(sendTo)){
                HashMap<String,File> attachments = new HashMap<String,File>();
                if (CollectionUtils.isNotEmpty(attachedFileList)) {
                    for (File attachedFile : attachedFileList) {
                        attachments.put(attachedFile.getName(),attachedFile);
                    }
                }
                MailSendBean mailSendBean = MailSendBean.newDefaultBuilder()
                        .from(from)
                        .to(sendTo)
                        .subject(subject)
                        .fromName("财务系统")
                        .labelId(0)
                        .plain(text)
                        .attachments(attachments)
                        .build();
                if(attachments.isEmpty()){
                    sendCloudMailSender.sendSimpleMailAsync(mailSendBean,mailSendCallback);
                }else {
                    sendCloudMailSender.sendAttachmentMailAsync(mailSendBean,mailSendCallback);
                }

            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }



    @Async
    @Override
    public void sendMailAsync(MailSendBean mailSendBean, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            sendCloudMailSender.sendSimpleMailAsync(mailSendBean);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }


    @Override
    public DataSource getDataSource(String content,String fileName){
        ByteArrayDataSource dataSource = new ByteArrayDataSource(content.getBytes(EncodingConfig.defaultCharset), "text/html;charset=UTF-8");
        dataSource.setName(fileName);
        return dataSource;
    }



    /**
     * 判断是否是有效附件
     *
     * @param attachedFileAbdPath 附件文件绝对路径
     * @return File or null
     */
    private File hasValidAttachent(String attachedFileAbdPath) {
        if (StringUtils.isBlank(attachedFileAbdPath)) {
            return null;
        }

        File file = new File(attachedFileAbdPath);
        return (file.exists() && file.isFile()) ? file : null;
    }

    /**
     * 保存邮件信息
     * @param mailInfo
     * @return
     */
    @Override
    public Long saveMailInfo(MailInfo mailInfo){
        return systemDao.saveMailInfo(mailInfo);
    }



    /**
     * 保存发送和接收的内容方便历史查询
     * @param fileContent
     * @param fileName
     */
    @Override
    public File createAttachMailFile(String fileContent, String fileName){
        if(StringUtils.isEmpty(fileContent)||StringUtils.isEmpty(fileName)){
            logger.info("createAttachMailFile fileContent or fileName is empty");
            return null;
        }
        try {
            createDir();
            File file = new File(hubeiHistoryFilePath+ DateUtil.sampleDateFormatFormat(new Date())+File.separatorChar+"mail"+File.separatorChar+fileName);
            file.createNewFile();
            try(BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(file))){
                outputStream.write(fileContent.getBytes(EncodingConfig.defaultCharset));
                outputStream.flush();
            }

            logger.info("File save successfully!");
            return file;
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
            logger.info("save fileContent fail,fileName={}, content=",fileName);
            logger.info("{}",fileContent);
        }
        return null;
    }

    public void createDir(){
        String dateFormat = DateUtil.sampleDateFormatFormat(new Date());
        File fileDir = new File(hubeiHistoryFilePath+dateFormat+File.separatorChar+"mail"+File.separatorChar);
        if(!fileDir.exists()||!fileDir.isDirectory()){
            fileDir.mkdir();
        }
    }

}
