package cn.quantgroup.financial.service.impl;

import cn.quantgroup.financial.constant.*;
import cn.quantgroup.financial.dao.IHuBeiCFCDao;
import cn.quantgroup.financial.dao.IPaymentDao;
import cn.quantgroup.financial.dao.IRepayOfflineRecordDao;
import cn.quantgroup.financial.dao.IRepayRecordDao;
import cn.quantgroup.financial.exception.ServerErrorException;
import cn.quantgroup.financial.handler.IHuBeiDispatcher;
import cn.quantgroup.financial.model.*;
import cn.quantgroup.financial.model.huibeicfc.*;
import cn.quantgroup.financial.service.IApiCommonService;
import cn.quantgroup.financial.service.IHuBeiService;
import cn.quantgroup.financial.service.sys.ICompensationDayService;
import cn.quantgroup.financial.service.sys.IMailService;
import cn.quantgroup.financial.service.sys.IScheduledJudgeService;
import cn.quantgroup.financial.util.DateUtil;
import cn.quantgroup.financial.util.HubeiCFCUtil;
import cn.quantgroup.financial.util.NetUtil;
import cn.quantgroup.sms.IMailSendCallback;
import cn.quantgroup.sms.MailSendBean;
import com.alibaba.fastjson.JSON;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.DocumentException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import java.io.*;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

import static cn.quantgroup.financial.constant.HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT;
import static cn.quantgroup.financial.constant.HubeiCFCDataType.RETURN_BATCH_COMPENSATION;
import static cn.quantgroup.financial.constant.HubeiCFCDataType.RETURN_BATCH_DEBIT;

/**
 * Created by WuKong on 2017/1/19.
 */
@Service
public class HuBeiServiceImpl implements IHuBeiService {

    @Value("${thirdparty.api.hubei}")
    private String hubeiApiUrl;

    @Value("${spring.mail.username}")
    private String sender;

    @Value("${hubei.history.filepath}")
    private String hubeiHistoryFilePath;



    @Autowired
    private IHuBeiDispatcher huBeiHandlerDispatcher;

    @Autowired
    private IHuBeiCFCDao huBeiCFCDao;

    @Autowired
    private IPaymentDao paymentDao;

    @Autowired
    private IRepayRecordDao repayRecordDao;

    @Autowired
    private IRepayOfflineRecordDao repayOfflineRecordDao;

    @Autowired
    private ICompensationDayService compensationDayService;

    @Autowired
    private IApiCommonService iApiCommonService;

    @Autowired
    private IMailService iMailService;

    @Autowired
    private IScheduledJudgeService scheduledJudgeService;

    private static final Logger logger = LoggerFactory.getLogger(HuBeiServiceImpl.class);



    /**
     * 所有类型统一处理入口
     * @param hubeiCFCDataType
     * @param docName
     */
    @Override
    public HuBeiCFCResponse handleDiscData(HubeiCFCDataType hubeiCFCDataType,String docName,Long docId,Byte seqNo){
        String applyDt = DateUtil.sampleDateFormatFormat(new Date());
//        applyDt = "20170216";//测试
        return handleDiscData(hubeiCFCDataType,docName,docId,seqNo,applyDt);
    }

    /**
     * 处理回盘送盘
     * @param hubeiCFCDataType
     * @param docName
     * @param docId
     * @param seqNo
     * @param applyDt
     * @return
     */
    @Override
    public HuBeiCFCResponse handleDiscData(HubeiCFCDataType hubeiCFCDataType,String docName,Long docId,Byte seqNo,String applyDt){
        HuBeiCFCResponse response = null;
        switch (hubeiCFCDataType){
            case SEND_DEBIT:
                response = querySendDiscData(applyDt);
                break;
            case SEND_ADVANCE_REPAYMENT_CHECK:
                response = checkSendDiscData(docName,applyDt);
                break;
            default:
                response = pushReturnDiscData(hubeiCFCDataType,applyDt,docId,seqNo);
        }
        return response;
    }

    /**
     * 回盘文件先发邮件。
     * @param huBeiCFCRequest
     */
    public void returnMailDisc(HuBeiCFCRequest huBeiCFCRequest){
        try {
            HubeiCFCDataType dataType = huBeiCFCRequest.getHubeiCFCDataType();
            switch (dataType){
                case RETURN_ADVANCE_REPAYMENT:
                case RETURN_BATCH_DEBIT:
                case RETURN_BATCH_COMPENSATION:
                    //发送回盘 邮件
                    File file = iMailService.createAttachMailFile(new String(Base64.decodeBase64(huBeiCFCRequest.getFileConent())),huBeiCFCRequest.getFileName());
                    sendMailAttachment(dataType,new ArrayList<File>(Arrays.asList(file)));
                    break;
                default:
            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }
    /**
     * 构建请求数据
     * @param huBeiCFCRequest
     * @return
     * @throws IOException
     * @throws TransformerException
     * @throws ParserConfigurationException
     * @throws DocumentException
     * @throws SAXException
     * @throws ServerErrorException
     */
    private HuBeiCFCResponse requestDiscData(HuBeiCFCRequest huBeiCFCRequest) throws IOException, TransformerException, ParserConfigurationException, DocumentException, SAXException, ServerErrorException {


        HubeiCFCDataType dataType = huBeiCFCRequest.getHubeiCFCDataType();
        if(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT.equals(dataType)
                ||HubeiCFCDataType.RETURN_BATCH_COMPENSATION.equals(dataType)
                ||HubeiCFCDataType.RETURN_BATCH_DEBIT.equals(dataType)){
            returnMailDisc(huBeiCFCRequest);
            //回盘文件需要检测是否开启了请求开关
            if(!scheduledJudgeService.isOpenHuBeiRequest()){
                logger.info("did not push hubei return disc");
                HuBeiCFCResponse response = new HuBeiCFCResponse();
                response.setEc(HubeiCFCField.EC_SUCCESS_CODE);
                return response;
            }
        }

        String docName = huBeiCFCRequest.getFileName();
        String fileContent = huBeiCFCRequest.getFileConent();
        String dateFormat = huBeiCFCRequest.getApplyDt();
        Map<String,String> argsMap = new HashMap<>();
        Map<String,String> headerMap = new HashMap<>();
        Map<String,String> requestMap = new HashMap<>();
        headerMap.put("Content-Type","text/html;charset=UTF-8");
        headerMap.put("Connection","close");

        argsMap.put(HubeiCFCField.flag,dataType.getFlag());
        argsMap.put(HubeiCFCField.applyDt, dateFormat);
        if(!StringUtils.isEmpty(docName)){
            argsMap.put(HubeiCFCField.docName, docName);
        }
        if(!StringUtils.isEmpty(fileContent)){
            argsMap.put(HubeiCFCField.picUploadFile,fileContent);
        }

        String requestBody = HubeiCFCUtil.buildRequest(argsMap, HubeiCFCField.businessMap.get(dataType));
        logger.info("apiUrl={}",hubeiApiUrl);
        if(logger.isDebugEnabled()){
            logger.info("apiUrl={}, requestBody={}",hubeiApiUrl,requestBody);
        }
        logger.info("apiUrl={}, requestBody={}",hubeiApiUrl,requestBody);
        requestMap.put(HubeiCFCField.xmlString,requestBody);
        String filePre = docName==null?dataType.get()+"_":dataType.get()+"_"+docName;
        saveHistoryFile(requestBody,filePre+"_request");
        HttpResult httpResult = null;
        try {
            httpResult = NetUtil.postRequestUrl(hubeiApiUrl,requestMap,headerMap, SysConstant.retryOneNums);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        if(httpResult!=null&&httpResult.getStatus()==NetUtil.RequestStatus.SUCCESS_STATUS){
            String responseBody = httpResult.getResult();
            saveHistoryFile(responseBody,filePre+"_response");
            if(logger.isDebugEnabled()){
                logger.info("responseBody={}",responseBody);
            }
            HuBeiCFCResponse response = HubeiCFCUtil.parseDiscResponse(new ByteArrayInputStream(responseBody.getBytes(EncodingConfig.defaultCharset)));
            //响应成功
            if(response!=null&&HubeiCFCField.EC_SUCCESS_CODE.equals(response.getEc())){
                //需要保存返回数据
                if(huBeiCFCRequest.getSaveRepData()){
                    saveResponseData(response,dataType);
                }else {
                    logger.info("do not need save data");
                }
            }
            switch (dataType){
                case SEND_DEBIT:
                case SEND_ADVANCE_REPAYMENT_CHECK:
                    if(response!=null&&HubeiCFCField.EC_SUCCESS_CODE.equals(response.getEc())){
                        //发送送盘文件 邮件
                        File file = iMailService.createAttachMailFile(new String(Base64.decodeBase64(response.getFileConent())),response.getFileName());
                        sendMailAttachment(dataType,new ArrayList<File>(Arrays.asList(file)));
                        response.setFileConent(null);
                    }
                    break;

                default:
            }
            return response;
        }else {
            logger.info("response is not success");
        }

        logger.info("requestDiscData method end");
        return null;
    }

    /**
     * 送盘文件
     * @param applyDt
     */
    private HuBeiCFCResponse querySendDiscData(String applyDt){
        try {
            return requestDiscData(HuBeiCFCRequest.build(HubeiCFCDataType.SEND_DEBIT,null,null,applyDt,Boolean.TRUE));
        }catch (Exception e){
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    /**
     * 送盘检查
     * @param docName
     */
    private HuBeiCFCResponse checkSendDiscData(String docName,String applyDt){
        try {
            return requestDiscData(HuBeiCFCRequest.build(HubeiCFCDataType.SEND_ADVANCE_REPAYMENT_CHECK,docName,null,applyDt,Boolean.TRUE));
        }catch (Exception e){
            logger.error(e.getMessage(),e);
        }
        return null;
    }
    /**
     * 回盘文件发送
     * @param hubeiCFCDataType
     */
    private HuBeiCFCResponse pushReturnDiscData(HubeiCFCDataType hubeiCFCDataType,String applyDt,Long docId,Byte seqNo){

        try {
            String docName = HubeiCFCUtil.getDocName(hubeiCFCDataType,seqNo);
            HuBeiDocName huBeiDocName =huBeiCFCDao.getDocNameById(docId);
            String fileContent = buildFileContentData(hubeiCFCDataType,docId,seqNo);
            if(StringUtils.isEmpty(fileContent)){
                logger.info("hubeiCFCDataType={},applyDt={} fileContent is empty",hubeiCFCDataType.get(),applyDt);
                HuBeiCFCResponse response = new HuBeiCFCResponse();
                response.setEc(HubeiCFCField.EC_SUCCESS_CODE);
                return response;
            }
            logger.info("pushReturnDiscData hubeiCFCDataType={},applyDt={}",hubeiCFCDataType.get(),applyDt);
            HuBeiCFCRequest huBeiCFCRequest = HuBeiCFCRequest.build(hubeiCFCDataType,docName,fileContent,applyDt,Boolean.FALSE);

            HuBeiCFCResponse response = requestDiscData(huBeiCFCRequest);

            //推送数据 记录返回结果码
            if(response!=null){
                huBeiDocName.setEcCode(response.getEc());
                huBeiDocName.setEmMsg(response.getEm());
            }else {
                huBeiDocName.setEcCode(HubeiCFCField.EC_SUCCESS_CODE);
            }
            huBeiCFCDao.updateDocName(huBeiDocName);
            if(response==null||!HubeiCFCField.EC_SUCCESS_CODE.equals(response.getEc())){
                logger.info(new StringBuilder("ec:").append(response==null?"null":response.getEc())
                        .append("|em:").append(response==null?"":response.getEm())
                        .append("|dateFormat=").append(huBeiCFCRequest.getApplyDt()).toString());
            }
            return response;
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }
    /**
     * 构建推送的数据内容 base64编码
     * @param hubeiCFCDataType
     * @return
     */
    private String buildFileContentData(HubeiCFCDataType hubeiCFCDataType,Long docId,Byte seqNo){
        List<HuBeiHistory> huBeiHistoryList = huBeiCFCDao.getListByDocNameIdAndSeqNo(docId,seqNo);
        //直接放在内存中，后续数据量大了写入文件
        StringBuilder stringBuilder = new StringBuilder();
        String base64String = "";
        List<HuBeiHistory> errorHuBeiHistoryList = new ArrayList<>();
        Date deadline = HubeiCFCUtil.getTodayDeadLine();
        int cursor = 0;
        if(huBeiHistoryList!=null&&huBeiHistoryList.size()>0){
            int size = huBeiHistoryList.size();
            for(int index=0;index<size;index++){
                HuBeiHistory history = huBeiHistoryList.get(index);

                HuBeiJsonBean jsonBean = history.getData();
                //提前还款文件 要过滤应还日小于等于当天
                if(hubeiCFCDataType.get().compareTo(RETURN_ADVANCE_REPAYMENT.get())==0){
                    Date repayDate = DateUtil.dateParse(jsonBean.getRepayDate());
                    if(repayDate!=null&&repayDate.before(deadline)){
                        continue;
                    }
                    BigDecimal computeRepayAmount = HubeiCFCUtil.getShouldRepay(jsonBean.getLoanAmount(),jsonBean.getTotalTerm(),history.getCurrTermNo());
                    if(computeRepayAmount!=null&&computeRepayAmount.subtract(jsonBean.getRepayTotalAmount()).abs().compareTo(new BigDecimal("0.01"))==0){
                        jsonBean.setRepayTotalAmount(computeRepayAmount);
                    }
                }
                //应扣不等于实扣 需要发送异常邮件
                //提前还款没有实扣
                if(hubeiCFCDataType.get().compareTo(RETURN_ADVANCE_REPAYMENT.get())!=0){
                    if(jsonBean!=null&&jsonBean.getRepayAmount()!=null&&jsonBean.getReallyRepayAmount()!=null){
                        if(jsonBean.getReallyRepayAmount().compareTo(BigDecimal.ZERO)!=0&&jsonBean.getRepayAmount().compareTo(jsonBean.getReallyRepayAmount())!=0){
                            errorHuBeiHistoryList.add(history);
//                            if(jsonBean.getRepayAmount().subtract(jsonBean.getReallyRepayAmount()).abs().compareTo(new BigDecimal("0.01"))>0){
//                                continue;
//                            }else {
                            //需求改变 无论是否一分钱都放在邮件中
                            //差值在一分钱以内 则发错误邮件同时校准跟湖北保持一致 方便后续对账。
                            jsonBean.setReallyRepayAmount(jsonBean.getRepayAmount());
                            jsonBean.setTradeCode(HubeiCFCField.tradeCodeSuccess);
                            jsonBean.setTradeMsg(HubeiCFCField.codeMappingMsg.get(HubeiCFCField.tradeCodeSuccess));
//                            }
                        }
                    }else {
                        logger.info("jsonBean is null, history={}", JSON.toJSONString(history));
                    }
                }
                if(cursor==0){
                    stringBuilder.append(huBeiHandlerDispatcher.builder(hubeiCFCDataType,history));
                    cursor++;
                }else {
                    stringBuilder.append("\r").append(huBeiHandlerDispatcher.builder(hubeiCFCDataType,history));
                    cursor++;
                }
            }
            if(!CollectionUtils.isEmpty(errorHuBeiHistoryList)){
                sendErrorMailNotice(errorHuBeiHistoryList,"以下是应扣金额与实扣金额不符:\r\n");
            }

            logger.info("returnDisc fileContent={}",stringBuilder.toString());
            base64String = new String(Base64.encodeBase64Chunked(stringBuilder.toString().getBytes(EncodingConfig.defaultCharset)),EncodingConfig.defaultCharset);
        }

        return base64String;
    }

    /**
     * 保存响应数据
     * @param type
     */
    public void saveResponseData(HuBeiCFCResponse response, HubeiCFCDataType type) throws ServerErrorException {
        logger.info("saveResponseData method is called");
        String line = null;
        saveHistoryFile(response.getFileConent(),response.getFileName());
        try(BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(Base64.decodeBase64(response.getFileConent()))))) {
            HuBeiDocName docName = new HuBeiDocName();
            docName.setDocName(response.getFileName());
            docName.setDataType(type.get());
            huBeiCFCDao.saveDocName(docName);
            response.setFileId(docName.getId());
            while ((line=bufferedReader.readLine())!=null){
                if(!StringUtils.isEmpty(line)){
                    HuBeiHistory history = huBeiHandlerDispatcher.parse(type,line);
                    history.setDocNameId(docName.getId());
                    huBeiCFCDao.saveHistory(history);
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
            throw new ServerErrorException(e.getMessage());
        }
        logger.info("saveResponseData method is finished");
    }

    /**
     * 构建回盘文件信息
     * @return
     */
    @Override
    public List<HuBeiHistory> getReturnDisc(HuBeiDocName sendHuBeiDocName){
        /**
         * 当日送盘文件只有合同号 没有期数以及loanHistoryId
         * 需要先查到对应的loanHistoryId 同时按最近一期未代偿的作为需要查的期数
         */
        List<HuBeiHistory> sendHuBeiHistoryList = huBeiCFCDao.getListByDocNameId(sendHuBeiDocName.getId());
        Set<String>  contactNoSet = new HashSet<>();
        Map<String,HuBeiHistory> contractMapHuiBeiHistory = new HashMap<>();
        sendHuBeiHistoryList.forEach(huBeiHistroy ->{
            contactNoSet.add(huBeiHistroy.getContactNo());
            contractMapHuiBeiHistory.put(huBeiHistroy.getContactNo(),huBeiHistroy);
        });

        List<PaymentDetail> paymentDetailList = paymentDao.getListByMerchantContractNo(new ArrayList<>(contactNoSet));

        List<Long> loanHistoryIdList = new ArrayList<Long>();
        Map<Long,HuBeiHistory> loanIdMapHuiBeiHistory = new HashMap<>();
        Map<String,Long> contractNoMapLoanId = new HashMap<>();
        paymentDetailList.forEach(paymentDetail -> {
            loanHistoryIdList.add(paymentDetail.getLoanHistoryId());
            loanIdMapHuiBeiHistory.put(paymentDetail.getLoanHistoryId(),contractMapHuiBeiHistory.get(paymentDetail.getMerchantContractNo()));
            contractNoMapLoanId.put(paymentDetail.getMerchantContractNo(),paymentDetail.getLoanHistoryId());
        });
        //清空
        contactNoSet.clear();
        contractMapHuiBeiHistory.clear();

        /**
         * 通过loanHistoryId list 查询还款计划 最近一期
         */
        List<RepaymentPlanDetail> repaymentPlanDetailList = paymentDao.getRepaymentPlanListByLoanIds(loanHistoryIdList);
        repaymentPlanDetailList = repaymentPlanDetailList==null?new ArrayList<RepaymentPlanDetail>():repaymentPlanDetailList;
        for(int index=0;index<repaymentPlanDetailList.size();index++){
            RepaymentPlanDetail repaymentPlanDetail = repaymentPlanDetailList.get(index);
            HuBeiJsonBean jsonBean= loanIdMapHuiBeiHistory.get(repaymentPlanDetail.getLoanHistoryId()).getData();
            if(DateUtil.parse(jsonBean.getShouldRepayDate()).compareTo(repaymentPlanDetail.getDeadLine())!=0){
                repaymentPlanDetailList.remove(index--);
            }
        }
        //未还款
        Map<String,RepaymentPlanDetail> noYetContractMapRepayPlan = new HashMap<>();
        //设置具体哪一期
        repaymentPlanDetailList.forEach(repaymentPlanDetail -> {
            loanIdMapHuiBeiHistory.get(repaymentPlanDetail.getLoanHistoryId()).setCurrTermNo(repaymentPlanDetail.getCurrentTerm());
            noYetContractMapRepayPlan.put(loanIdMapHuiBeiHistory.get(repaymentPlanDetail.getLoanHistoryId()).getContactNo(),repaymentPlanDetail);
        });


        /**
         * 如果今天是代偿日  通过loanHistoryId list 查询是代偿的还款计划
         */
        Map<String,RepaymentPlanDetail> compensationContractMapRepayPlan = new HashMap<>();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(sendHuBeiDocName.getCreateTime());
        if(compensationDayService.isCompensationDay(calendar)){
            logger.info("ReturnDisc today is compenstaion");
            //代偿日
            int year = calendar.get(Calendar.YEAR);
            int month = calendar.get(Calendar.MONTH)+1;
            Date date = compensationDayService.getCompensationDay(year,month);
            List<RepaymentPlanDetail> compensationList = paymentDao.getListByCompensationDateAndLoanIds(loanHistoryIdList,date);
            compensationList.forEach(repaymentPlanDetail ->compensationContractMapRepayPlan.put(loanIdMapHuiBeiHistory.get(repaymentPlanDetail.getLoanHistoryId()).getContactNo(),repaymentPlanDetail) );
        }
        loanIdMapHuiBeiHistory.clear();

        /**
         * 通过loanHistory查询所有 关于的还款信息（包含所有还款的期数）
         */
        Map<Long,List<RepayXyqbDetail>> listRepayDetailMap = getRepayXyqbMap(loanHistoryIdList);
        /**
         * 通过loanHistory查询所有线下还款信息
         */
        Map<Long,List<RepayOfflineRecord>> listRepayOfflineMap = getOfflineRepayMap(loanHistoryIdList);
        List<HuBeiHistory> returnHuBeiHistoryList = new ArrayList<>();
        List<HuBeiHistory> errorHuBeiHistoryList = new ArrayList<>();
        returnHuBeiHistoryList.addAll(advanceRepayment(loanHistoryIdList));
        loanHistoryIdList.clear();

        int index = 0;


        for(; index< sendHuBeiHistoryList.size(); index++){
            HuBeiHistory history = sendHuBeiHistoryList.get(index);
            Long loanId = contractNoMapLoanId.get(history.getContactNo());
            List<RepayXyqbDetail> repayXyqbDetailList = loanId==null?null:(ArrayList<RepayXyqbDetail>) listRepayDetailMap.get(loanId);
            List<RepayOfflineRecord> repayOfflineRecordList = loanId==null?null:(ArrayList<RepayOfflineRecord>) listRepayOfflineMap.get(loanId);
            Boolean isContinue = Boolean.FALSE;
            //现金贷还款
            if(!CollectionUtils.isEmpty(repayXyqbDetailList)){
                //过滤掉要查找的期数之前的还款信息
                if(logger.isDebugEnabled()){
                    logger.info(history.getCurrTermNo()+"|"+JSON.toJSONString(repaymentPlanDetailList));
                }
                repayXyqbDetailList = repayXyqbDetailList.stream().filter(repayXyqbDetail -> {
                    if(history.getCurrTermNo()!=null&&repayXyqbDetail.getCurrTermNo().compareTo(history.getCurrTermNo())>=0){
                        return true;
                    }else {
                        return false;
                    }
                }).collect(Collectors.toCollection(ArrayList::new));
                Collections.sort(repayXyqbDetailList,new RepayXyqbCompartor());
                if(!CollectionUtils.isEmpty(repayXyqbDetailList)){
                    RepayXyqbDetail repayXyqbDetail = repayXyqbDetailList.get(0);
                    if(repayXyqbDetail.getCurrTermNo().compareTo(history.getCurrTermNo())==0){
                        RepaymentPlanDetail repaymentPlanDetail = noYetContractMapRepayPlan.get(history.getContactNo());
                        if(repaymentPlanDetail!=null){
                            //湖北罚息
                            BigDecimal hubeiOverdueInterest = HubeiCFCUtil.caculateOverDueFee(repaymentPlanDetail.getDeadLine(),repayXyqbDetail.getPayCenterRepayAt(),repaymentPlanDetail.getPrincipal());
                            //批扣
                            BigDecimal reallyRepayAmount = repaymentPlanDetail.getPrincipal().add(repaymentPlanDetail.getInterest()).add(hubeiOverdueInterest).setScale(2, RoundingMode.HALF_UP);
                            returnHuBeiHistoryList.add(deitHuiBeiHistory(history, sendHuBeiDocName.getCreateTime(),reallyRepayAmount));
                        }

                        repayXyqbDetailList.remove(0);
                    }

                    //逾期 提前还款 逾期的放在回盘，提前还款放在提前还款文件中
                    if(!CollectionUtils.isEmpty(repayXyqbDetailList)){
                        //这部分
                        List<HuBeiHistory> advanceHistoryList = advanceHuiBeiHistory(repayXyqbDetailList);
                        if(!CollectionUtils.isEmpty(advanceHistoryList)){
                            returnHuBeiHistoryList.addAll(advanceHistoryList);
                        }
                    }
                    isContinue = Boolean.TRUE;
                }
            }

            //线下还款
            if(!CollectionUtils.isEmpty(repayOfflineRecordList)){
                //过滤掉要查找的期数之前的还款信息
                if(logger.isDebugEnabled()){
                    logger.info(history.getCurrTermNo()+"|"+JSON.toJSONString(repayOfflineRecordList));
                }
                repayOfflineRecordList = repayOfflineRecordList.stream().filter(repayOfflineRecord -> {
                    if(history.getCurrTermNo()!=null&& repayOfflineRecord.getCurrTerm().compareTo(history.getCurrTermNo())>=0){
                        return true;
                    }else {
                        return false;
                    }
                }).collect(Collectors.toCollection(ArrayList::new));
                Collections.sort(repayOfflineRecordList,new RepayOfflineCompartor());
                if(!CollectionUtils.isEmpty(repayOfflineRecordList)){
                    RepayOfflineRecord repayOfflineRecord = repayOfflineRecordList.get(0);
                    //应还日或者逾期 湖北不能接收再提前还款
                    if(repayOfflineRecord.getCurrTerm().compareTo(history.getCurrTermNo())==0){
                        RepaymentPlanDetail repaymentPlanDetail = noYetContractMapRepayPlan.get(history.getContactNo());
                        if(repaymentPlanDetail!=null){
                            //湖北罚息
                            BigDecimal hubeiOverdueInterest = HubeiCFCUtil.caculateOverDueFee(repaymentPlanDetail.getDeadLine(), repayOfflineRecord.getUpdatedAt(),repaymentPlanDetail.getPrincipal());
                            //批扣
                            BigDecimal reallyRepayAmount = repaymentPlanDetail.getPrincipal().add(repaymentPlanDetail.getInterest()).add(hubeiOverdueInterest).setScale(2, RoundingMode.HALF_UP);
                            returnHuBeiHistoryList.add(deitHuiBeiHistory(history, sendHuBeiDocName.getCreateTime(),reallyRepayAmount));
                        }
                        if(repayOfflineRecordList.size()>1){
                            //应还日提前还款了多期 第一期放在批扣中 其他的放在异常文件中
                            repayOfflineRecordList.remove(0);
                            if(!CollectionUtils.isEmpty(repayOfflineRecordList)){
                                List<HuBeiHistory> advanceHistoryList = advanceOfflineHuiBeiHistory(repayOfflineRecordList);
                                if(!CollectionUtils.isEmpty(advanceHistoryList)){
                                    errorHuBeiHistoryList.addAll(advanceHistoryList);
                                }
                            }
                        }
                    }else {
                        //这部分提前还款的
                        List<HuBeiHistory> advanceHistoryList = advanceOfflineHuiBeiHistory(repayOfflineRecordList);
                        if(!CollectionUtils.isEmpty(advanceHistoryList)){
                            returnHuBeiHistoryList.addAll(advanceHistoryList);
                        }
                    }

                    isContinue = Boolean.TRUE;
                }
            }
            if(isContinue){
                //在这里做截断原因是，有可能客户同时发生了线上还款和线下还款 所以都要处理一遍逻辑。
                continue;
            }
            if(compensationContractMapRepayPlan.containsKey(history.getContactNo())){
                //当日代偿
                //湖北罚息
                RepaymentPlanDetail repaymentPlanDetail = compensationContractMapRepayPlan.get(history.getContactNo());
                BigDecimal hubeiOverdueInterest = HubeiCFCUtil.caculateOverDueFee(repaymentPlanDetail.getDeadLine(),new Date(),repaymentPlanDetail.getPrincipal());
                //批扣
                BigDecimal reallyRepayAmount = repaymentPlanDetail.getPrincipal().add(repaymentPlanDetail.getInterest()).add(hubeiOverdueInterest).setScale(2,RoundingMode.HALF_UP);
                returnHuBeiHistoryList.add(compensationHuiBeiHistory(history, sendHuBeiDocName.getCreateTime(),reallyRepayAmount));
                continue;
            }

            //未还款 批扣 实际还款金额零
            returnHuBeiHistoryList.add(deitHuiBeiHistory(history, sendHuBeiDocName.getCreateTime(),BigDecimal.ZERO));
        }


        //给相关人员发送异常的邮件
        if(!CollectionUtils.isEmpty(errorHuBeiHistoryList)){
            sendErrorMailNotice(errorHuBeiHistoryList,"异常的数据(应还日或逾期 提前还款造成)：");
        }


        return returnHuBeiHistoryList;
    }


    /**
     * 查询当日提前还款信息
     * @param loanIdList
     * @param
     * @param
     * @return
     */
    public List<HuBeiHistory> advanceRepayment(List<Long> loanIdList){
        //今天还款 送盘文件中不存在的 查询出来是按 curr_term_no 正序
        List<HuBeiHistory> returnHuBeiHistoryList = new ArrayList<>();
        Date[] dates = HubeiCFCUtil.getDateScope();
        Date[] offlineDates = HubeiCFCUtil.getOfflineDateScope();
        List<RepayXyqbDetail> repayXyqbDetailList = repayRecordDao.getListNotLoanIdBetweenDate(FundCorpType.HuBeiCFC.getId(),loanIdList,dates[0],dates[1]);
        List<RepayOfflineRecord> repayOfflineRecordList = repayOfflineRecordDao.getByNotInLoanListAndRepayStatusAndUpdateAt(loanIdList,OfflineRepayStatus.REPAY_BIZ_STATUS_OK,offlineDates[0],offlineDates[1]);
        Map<Long,List<RepayXyqbDetail>> loanIdMappingRepayXyqb = new HashMap<>();
        Map<Long,List<RepayOfflineRecord>> loanIdMappingRepayOffline = new HashMap<>();
        if(!CollectionUtils.isEmpty(repayXyqbDetailList)){
            for(RepayXyqbDetail repayXyqbDetail: repayXyqbDetailList){
                if(loanIdMappingRepayXyqb.containsKey(repayXyqbDetail.getLoanApplicationHistoryId())){
                    loanIdMappingRepayXyqb.get(repayXyqbDetail.getLoanApplicationHistoryId()).add(repayXyqbDetail);
                }else {
                    List<RepayXyqbDetail> list = new ArrayList<>();
                    list.add(repayXyqbDetail);
                    loanIdMappingRepayXyqb.put(repayXyqbDetail.getLoanApplicationHistoryId(),list);
                }
            }
        }

        if(!CollectionUtils.isEmpty(repayOfflineRecordList)){
            for(RepayOfflineRecord repayOfflineRecord: repayOfflineRecordList){
                if(repayOfflineRecord.getBizType()!=null&&repayOfflineRecord.getBizType().intValue()!=0){
                    //非现金贷 类型 过滤
                    logger.info("bizType ={}",repayOfflineRecord.getBizType());
                    continue;
                }
                if(!repayOfflineRecord.getDebtFundingCorp().contains("湖北")){
                    //不是湖北消金的 过滤
                    logger.info("debtFundingCorp ={}",repayOfflineRecord.getDebtFundingCorp());
                    continue;
                }
                Long loanId = Long.valueOf(repayOfflineRecord.getBizLoanNo());
                if(loanIdMappingRepayOffline.containsKey(loanId)){
                    loanIdMappingRepayOffline.get(loanId).add(repayOfflineRecord);
                }else {
                    List<RepayOfflineRecord> list = new ArrayList<>();
                    list.add(repayOfflineRecord);
                    loanIdMappingRepayOffline.put(loanId,list);
                }
            }
        }

        for(Long loanId:loanIdMappingRepayXyqb.keySet()){
            Collections.sort(loanIdMappingRepayXyqb.get(loanId),new RepayXyqbCompartor());
            returnHuBeiHistoryList.addAll(advanceHuiBeiHistory(loanIdMappingRepayXyqb.get(loanId)));
        }

        for(Long loanId:loanIdMappingRepayOffline.keySet()){
            Collections.sort(loanIdMappingRepayOffline.get(loanId),new RepayOfflineCompartor());
            returnHuBeiHistoryList.addAll(advanceOfflineHuiBeiHistory(loanIdMappingRepayOffline.get(loanId)));
        }
        return returnHuBeiHistoryList;
    }

    /**
     * 比较器
     */
    class RepayXyqbCompartor implements  Comparator<RepayXyqbDetail>{
        @Override
        public int compare(RepayXyqbDetail o1, RepayXyqbDetail o2) {
            return o1.getLoanApplicationHistoryId().compareTo(o2.getLoanApplicationHistoryId());
        }
    }

    class RepayOfflineCompartor implements  Comparator<RepayOfflineRecord>{

        @Override
        public int compare(RepayOfflineRecord o1, RepayOfflineRecord o2) {
            return o1.getBizLoanNo().compareTo(o2.getBizLoanNo());
        }
    }
    /**
     * 代偿
     * @param history
     * @param happenTime
     * @return
     */
    private HuBeiHistory compensationHuiBeiHistory(HuBeiHistory history,Date happenTime,BigDecimal reallyRepayAmount){
        HuBeiHistory compensationHuiBeiHistory = HubeiCFCUtil.replicateBase(history);
        compensationHuiBeiHistory.setDataType(RETURN_BATCH_COMPENSATION.get());
        HuBeiJsonBean huBeiJsonBean = new HuBeiJsonBean();
        HubeiCFCUtil.copy(compensationHuiBeiHistory,huBeiJsonBean);
        //应扣金额
        huBeiJsonBean.setRepayAmount(history.getData().getRepayAmount());
        //实扣金额
        huBeiJsonBean.setReallyRepayAmount(reallyRepayAmount.setScale(2,RoundingMode.HALF_UP));
        huBeiJsonBean.setTradeCode(HubeiCFCField.tradeCodeSuccess);
        huBeiJsonBean.setTradeMsg(HubeiCFCField.tradeMsgSuccess);
        compensationHuiBeiHistory.setHappenTime(happenTime);
        compensationHuiBeiHistory.setData(huBeiJsonBean);
        compensationHuiBeiHistory.setSeqNo(new Integer(1).byteValue());
        return compensationHuiBeiHistory;
    }


    /**
     * 线下还款 提前还款
     * @param repayOfflineRecordlList
     * @return
     */
    private List<HuBeiHistory> advanceOfflineHuiBeiHistory(List<RepayOfflineRecord> repayOfflineRecordlList){
        if(CollectionUtils.isEmpty(repayOfflineRecordlList)){
            return null;
        }
        List<HuBeiHistory> advanceHuiBeiList = new ArrayList<HuBeiHistory>();
        Integer seqNo = new Integer(0);
        Long loanId = Long.valueOf(repayOfflineRecordlList.get(0).getBizLoanNo());
        PaymentDetail paymentDetail = paymentDao.getByLoanId(loanId);
        if(paymentDetail==null){
            logger.warn("offline loanId={} cant`t get paymentdetail",loanId);
            iApiCommonService.queryData(loanId,null);
            paymentDetail = paymentDao.getByLoanId(loanId);
        }

        List<HuBeiHistory> flowHuBeiHistoryList = null;

        if(paymentDetail!=null){
            try {
                flowHuBeiHistoryList = huBeiCFCDao.getFlowByContractNoList(Arrays.asList(paymentDetail.getMerchantContractNo()));
            } catch (Exception e) {
                logger.error(e.getMessage(),e);
            }
            for(RepayOfflineRecord repayOfflineRecord : repayOfflineRecordlList){
                seqNo++;
                HuBeiHistory advanceHuiBeiHistory = new HuBeiHistory();
                if(CollectionUtils.isEmpty(flowHuBeiHistoryList)){
                    advanceHuiBeiHistory.setFlowNo(null);
                }else {
                    advanceHuiBeiHistory.setFlowNo(flowHuBeiHistoryList.get(0).getFlowNo());
                }
                advanceHuiBeiHistory.setContactNo(paymentDetail.getMerchantContractNo());
                advanceHuiBeiHistory.setCurrTermNo(repayOfflineRecord.getCurrTerm());
                advanceHuiBeiHistory.setUserIdType(new Integer(UserCardType.IDENTITY_CARD.get()).byteValue());
                advanceHuiBeiHistory.setUserIdNo(paymentDetail.getUserIdNo());
                advanceHuiBeiHistory.setUserId(paymentDetail.getUserId());
                advanceHuiBeiHistory.setUserName(paymentDetail.getUserName());
                advanceHuiBeiHistory.setHappenTime(new Date());
                //批次问题
                advanceHuiBeiHistory.setSeqNo(repayOfflineRecord.getCurrTerm().byteValue());
                advanceHuiBeiHistory.setDataType(RETURN_ADVANCE_REPAYMENT.get());
                HuBeiJsonBean huBeiJsonBean = new HuBeiJsonBean();
                HubeiCFCUtil.copy(advanceHuiBeiHistory,huBeiJsonBean);
                huBeiJsonBean.setRepayType(HubeiCFCField.repayType);
                Date repayDate = repayOfflineRecord.getUpdatedAt();
                //yyyy-MM-dd  还款日期
                Calendar deadLine = Calendar.getInstance();
                deadLine.setTime(repayOfflineRecord.getDeadLine());
                String dateFormat = DateUtil.dateFormat(deadLine.getTime());
                String applyRepayDateFormat = DateUtil.dateFormat(repayDate);
                huBeiJsonBean.setRepayDate(dateFormat);
                huBeiJsonBean.setShortenTerm(repayOfflineRecordlList.size());
                //本次总计还款金额
                //湖北罚息
                BigDecimal huBeiOverdueInterest = HubeiCFCUtil.caculateOverDueFee(deadLine.getTime(), repayOfflineRecord.getUpdatedAt(), repayOfflineRecord.getPrincipal());
                BigDecimal reallyPayment = repayOfflineRecord.getPrincipal().add(repayOfflineRecord.getInterest()).add(huBeiOverdueInterest);
                huBeiJsonBean.setReallyRepayAmount(reallyPayment.setScale(2,RoundingMode.HALF_UP));
                huBeiJsonBean.setChannel(HubeiCFCField.repayChannel);
                huBeiJsonBean.setApplyRepayDate(applyRepayDateFormat);
                huBeiJsonBean.setRepayTotalAmount(reallyPayment.setScale(2,RoundingMode.HALF_UP));
                huBeiJsonBean.setLoanAmount(paymentDetail.getTotalPrincipal());
                huBeiJsonBean.setTotalTerm(paymentDetail.getContractTotalTerm());
                advanceHuiBeiHistory.setHappenTime(new Date());
                advanceHuiBeiHistory.setData(huBeiJsonBean);
                advanceHuiBeiList.add(advanceHuiBeiHistory);
            }
        }else {
            logger.warn("offline paymentDetail is null !!!,loanId={}",loanId);
        }

        return advanceHuiBeiList;
    }

    /**
     * 提前还款 RepayXyqb转换器HuBeiHistory
     * @param repayXyqbDetailList
     * @return
     */
    private List<HuBeiHistory> advanceHuiBeiHistory(List<RepayXyqbDetail> repayXyqbDetailList){
        if(CollectionUtils.isEmpty(repayXyqbDetailList)){
            return null;
        }
        List<HuBeiHistory> advanceHuiBeiList = new ArrayList<HuBeiHistory>();
        Integer seqNo = new Integer(0);
        PaymentDetail paymentDetail = paymentDao.getByLoanId(repayXyqbDetailList.get(0).getLoanApplicationHistoryId());
        if(paymentDetail==null){
            logger.warn("repayXyqb loanId={} cant`t get paymentdetail",repayXyqbDetailList.get(0).getLoanApplicationHistoryId());
            iApiCommonService.queryData(repayXyqbDetailList.get(0).getLoanApplicationHistoryId(),null);
            paymentDetail = paymentDao.getByLoanId(repayXyqbDetailList.get(0).getLoanApplicationHistoryId());
        }

        List<HuBeiHistory> flowHuBeiHistoryList = null;

        if(paymentDetail!=null){
            try {
                flowHuBeiHistoryList = huBeiCFCDao.getFlowByContractNoList(Arrays.asList(paymentDetail.getMerchantContractNo()));
            } catch (Exception e) {
                logger.error(e.getMessage(),e);
            }
            for(RepayXyqbDetail repayXyqbDetail: repayXyqbDetailList){
                seqNo++;
                HuBeiHistory advanceHuiBeiHistory = new HuBeiHistory();
                if(CollectionUtils.isEmpty(flowHuBeiHistoryList)){
                    advanceHuiBeiHistory.setFlowNo(null);
                }else {
                    advanceHuiBeiHistory.setFlowNo(flowHuBeiHistoryList.get(0).getFlowNo());
                }
                advanceHuiBeiHistory.setContactNo(paymentDetail.getMerchantContractNo());
                advanceHuiBeiHistory.setCurrTermNo(repayXyqbDetail.getCurrTermNo());
                advanceHuiBeiHistory.setUserIdType(new Integer(UserCardType.IDENTITY_CARD.get()).byteValue());
                advanceHuiBeiHistory.setUserIdNo(paymentDetail.getUserIdNo());
                advanceHuiBeiHistory.setUserId(paymentDetail.getUserId());
                advanceHuiBeiHistory.setUserName(paymentDetail.getUserName());
                advanceHuiBeiHistory.setHappenTime(new Date());
                //批次问题
                advanceHuiBeiHistory.setSeqNo(repayXyqbDetail.getCurrTermNo().byteValue());
                advanceHuiBeiHistory.setDataType(RETURN_ADVANCE_REPAYMENT.get());
                HuBeiJsonBean huBeiJsonBean = new HuBeiJsonBean();
                HubeiCFCUtil.copy(advanceHuiBeiHistory,huBeiJsonBean);
                huBeiJsonBean.setRepayType(HubeiCFCField.repayType);
                huBeiJsonBean.setLoanAmount(paymentDetail.getTotalPrincipal());
                huBeiJsonBean.setTotalTerm(paymentDetail.getContractTotalTerm());
                Date repayDate = repayXyqbDetail.getPayCenterRepayAt();
                //yyyy-MM-dd  还款日期
                Calendar deadLine = Calendar.getInstance();
                deadLine.setTime(repayXyqbDetail.getDeadline());
                deadLine.set(Calendar.DATE, deadLine.get(Calendar.DATE) - 1);
                String dateFormat = DateUtil.dateFormat(deadLine.getTime());
                String applyRepayDateFormat = DateUtil.dateFormat(repayDate);
                huBeiJsonBean.setRepayDate(dateFormat);
                huBeiJsonBean.setShortenTerm(repayXyqbDetailList.size());
                //本次总计还款金额
                //湖北罚息
                BigDecimal huBeiOverdueInterest = HubeiCFCUtil.caculateOverDueFee(deadLine.getTime(),repayXyqbDetail.getPayCenterRepayAt(),repayXyqbDetail.getPrincipal());
                BigDecimal reallyPayment = repayXyqbDetail.getPrincipal().add(repayXyqbDetail.getInterest()).add(huBeiOverdueInterest);
                huBeiJsonBean.setReallyRepayAmount(reallyPayment.setScale(2,RoundingMode.HALF_UP));
                huBeiJsonBean.setChannel(HubeiCFCField.repayChannel);
                huBeiJsonBean.setApplyRepayDate(applyRepayDateFormat);
                huBeiJsonBean.setRepayTotalAmount(reallyPayment.setScale(2,RoundingMode.HALF_UP));
                advanceHuiBeiHistory.setHappenTime(new Date());
                advanceHuiBeiHistory.setData(huBeiJsonBean);
                advanceHuiBeiList.add(advanceHuiBeiHistory);
            }
        }else {
            logger.warn("repayXyqb paymentDetail is null !!!,loanId={}",repayXyqbDetailList.get(0).getLoanApplicationHistoryId());
        }

        return advanceHuiBeiList;
    }


    /**
     * 批扣
     * @param history
     * @param happenTime
     * @param reallyRepayAmount
     * @return
     */
    private HuBeiHistory deitHuiBeiHistory(HuBeiHistory history,Date happenTime,BigDecimal reallyRepayAmount){
        HuBeiHistory deitHuiBeiHistory = HubeiCFCUtil.replicateBase(history);
        deitHuiBeiHistory.setDataType(RETURN_BATCH_DEBIT.get());
        HuBeiJsonBean huBeiJsonBean = new HuBeiJsonBean();
        HubeiCFCUtil.copy(history,huBeiJsonBean);
        //应扣金额
        huBeiJsonBean.setRepayAmount(history.getData().getRepayAmount());
        //实扣金额 0
        huBeiJsonBean.setReallyRepayAmount(reallyRepayAmount.setScale(2,RoundingMode.HALF_UP));
        if(history.getData().getRepayAmount()!=null
                &&reallyRepayAmount.compareTo(BigDecimal.ZERO)!=0
                &&history.getData().getRepayAmount().compareTo(reallyRepayAmount)==0){
            huBeiJsonBean.setTradeCode(HubeiCFCField.tradeCodeSuccess);
            huBeiJsonBean.setTradeMsg(HubeiCFCField.codeMappingMsg.get(HubeiCFCField.tradeCodeSuccess));
        }else if(reallyRepayAmount.compareTo(BigDecimal.ZERO)==0){
            huBeiJsonBean.setTradeCode(HubeiCFCField.tradeCodeNoTrade);
            huBeiJsonBean.setTradeMsg(HubeiCFCField.codeMappingMsg.get(HubeiCFCField.tradeCodeNoTrade));
        }else {
            huBeiJsonBean.setTradeCode(HubeiCFCField.tradeCodeWrongPrice);
            huBeiJsonBean.setTradeMsg(HubeiCFCField.codeMappingMsg.get(HubeiCFCField.tradeCodeWrongPrice));
        }

        deitHuiBeiHistory.setHappenTime(happenTime);
        deitHuiBeiHistory.setData(huBeiJsonBean);
        deitHuiBeiHistory.setSeqNo(new Integer(1).byteValue());
        return deitHuiBeiHistory;
    }


    /**
     *  通过loanId list 查询线下还款信息
     * @param loanHistoryIdList
     * @return
     */
    private Map<Long,List<RepayOfflineRecord>> getOfflineRepayMap(List<Long> loanHistoryIdList){
        List<RepayOfflineRecord> repayOfflineRecordList = repayOfflineRecordDao.getByReapyStatusAndLoanList(OfflineRepayStatus.REPAY_BIZ_STATUS_OK,loanHistoryIdList);
        Map<Long,List<RepayOfflineRecord>> listOfflineRepayMap = new HashMap<>();
        for(RepayOfflineRecord repayOfflineRecord : repayOfflineRecordList){
            if(listOfflineRepayMap.containsKey(Long.valueOf(repayOfflineRecord.getBizLoanNo()))){
                listOfflineRepayMap.get(Long.valueOf(repayOfflineRecord.getBizLoanNo())).add(repayOfflineRecord);
            }else {
                List<RepayOfflineRecord> list = new ArrayList<>();
                list.add(repayOfflineRecord);
                listOfflineRepayMap.put(Long.valueOf(repayOfflineRecord.getBizLoanNo()),list);
            }
        }
        return listOfflineRepayMap;
    }

    /**
     * 通过loanId list 查询还款信息
     * @param loanHistoryIdList
     * @return
     */
    private Map<Long,List<RepayXyqbDetail>> getRepayXyqbMap(List<Long> loanHistoryIdList){
        List<RepayXyqbDetail> repayXyqbDetailList = repayRecordDao.getListByLoanHistroyIds(FundCorpType.HuBeiCFC.getId(),loanHistoryIdList);

        Map<Long,List<RepayXyqbDetail>> listRepayDetailMap = new HashMap<>();
        for(RepayXyqbDetail repayXyqbDetail: repayXyqbDetailList){
            if(listRepayDetailMap.containsKey(repayXyqbDetail.getLoanApplicationHistoryId())){
                listRepayDetailMap.get(repayXyqbDetail.getLoanApplicationHistoryId()).add(repayXyqbDetail);
            }else {
                List<RepayXyqbDetail> list = new ArrayList<>();
                list.add(repayXyqbDetail);
                listRepayDetailMap.put(repayXyqbDetail.getLoanApplicationHistoryId(),list);
            }
        }
        return listRepayDetailMap;
    }


    /**
     * 异常邮件 金额不等 提前还款文件检验失败  送盘文件接收失败 回盘文件发送失败
     * @param messageString
     */
    @Async
    @Override
    public void sendErrorMailNotice(String messageString){
        try {
            String[] toUserArray = getMailListByType(SysConstant.MailType.HUBEI_ERROR);
            if(toUserArray==null||toUserArray.length==0){
                return;
            }
            SimpleMailMessage message = new SimpleMailMessage();
            message.setSentDate(new Date());
            message.setTo(toUserArray);
            message.setFrom("财务系统<" + sender + ">");
            message.setSubject("湖北消金送回盘文件接口失败");
            StringBuilder sb= new StringBuilder();
            sb.append(messageString);
            message.setText(sb.toString());
            for(String to: toUserArray){
                MailSendBean mailSendBean = MailSendBean.newDefaultBuilder()
                        .to(to)
                        .from(sender)
                        .fromName("财务系统")
                        .subject("湖北消金送回盘文件接口失败")
                        .plain(sb.toString())
                        .build();
                iMailService.sendMailAsync(mailSendBean,new IMailSendCallback() {
                    @Override
                    public void onError(Throwable e) {
                        logger.error("发送通知邮件失败,to-->{}", Arrays.toString(toUserArray));
                        logger.error("发送邮件失败", e);
                    }
                    @Override
                    public void onSuccess() {
                        logger.info("发送邮件成功,to-->{}",Arrays.toString(toUserArray));
                    }
                });
            }

        } catch (Exception e1) {
            logger.error(e1.getMessage(),e1);
        }
    }

    @Async
    @Override
    public void sendErrorMailNotice(List<HuBeiHistory> huBeiHistoryList, String noticeMessage){
        try {
            if(CollectionUtils.isEmpty(huBeiHistoryList)){
                return;
            }
            String[] toUserArray = getMailListByType(SysConstant.MailType.HUBEI_ERROR);
            if(toUserArray==null||toUserArray.length==0){
                return;
            }
            StringBuilder errorStringBuilder = new StringBuilder(noticeMessage+"\r");
            for(HuBeiHistory huBeiHistory:huBeiHistoryList){
                if(RETURN_ADVANCE_REPAYMENT.get().equals(huBeiHistory.getDataType())){
                    errorStringBuilder.append(huBeiHistory.getFlowNo()).append(SysConstant.vertical).append(huBeiHandlerDispatcher.builder(HubeiCFCField.HubeiTypeMap.get(huBeiHistory.getDataType()),huBeiHistory)).append("\r");
                }else {
                    errorStringBuilder.append(huBeiHandlerDispatcher.builder(HubeiCFCField.HubeiTypeMap.get(huBeiHistory.getDataType()),huBeiHistory)).append("\r");
                }
            }
            String message = errorStringBuilder.toString();
            if(!StringUtils.isEmpty(message)){
                File file = iMailService.createAttachMailFile(message,"exception_file.txt");
                sendErrorMailAttachment(new ArrayList<File>(Arrays.asList(file)),toUserArray,noticeMessage,"湖北消金送回盘文件异常数据");
            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }

    private String[] getMailListByType(Byte type){
        List<MailInfo> mailInfoList = huBeiCFCDao.getListByType(type);
        if(CollectionUtils.isEmpty(mailInfoList)){
            logger.info("mailInfoList is empty");
            return null;
        }
        List<String> mailList = new ArrayList<>();
        for(MailInfo mailInfo: mailInfoList){
            mailList.add(mailInfo.getMail());
        }
        String[] toUserArray = mailList.toArray(new String[mailList.size()]);
        return toUserArray;
    }

    @Async
    @Override
    public void sendErrorMailAttachment(ArrayList<File> attachedBytes, String[] toUserArray, String text, String subject){
        try {
            if(toUserArray==null||toUserArray.length==0){
                return;
            }
            for(String to: toUserArray){
                iMailService.sendAttachmentMailAsync(sender,to,null,subject,text,attachedBytes,new IMailSendCallback() {
                    @Override
                    public void onError(Throwable e) {
                        logger.error("发送通知邮件失败,to-->{}", Arrays.toString(toUserArray));
                        logger.error("发送邮件失败", e);
                    }
                    @Override
                    public void onSuccess() {
                        logger.info("发送邮件成功,to-->{}",Arrays.toString(toUserArray));
                    }
                });
            }
        } catch (Exception e1) {
            logger.error(e1.getMessage(),e1);
        }
    }


    @Async
    @Override
    public void sendMailAttachment(HubeiCFCDataType hubeiCFCDataType, ArrayList<File> attachedBytes){
        try {
            String[] toUserArray = getMailListByType(SysConstant.MailType.HUBEI_FILE);
            if(toUserArray==null||toUserArray.length==0){
                return;
            }

            String subject = DateUtil.getNowDate()+"-湖北消金对账接口";
            if(hubeiCFCDataType!=null){
                switch (hubeiCFCDataType){
                    case SEND_DEBIT:
                        subject = subject+"送盘文件";
                        break;
                    case SEND_ADVANCE_REPAYMENT_CHECK:
                        subject = subject+"提前还款检验文件";
                        break;
                    case RETURN_BATCH_DEBIT:
                        subject = subject+"回盘批扣文件";
                        break;
                    case RETURN_BATCH_COMPENSATION:
                        subject = subject+"回盘代偿文件";
                        break;
                    case RETURN_ADVANCE_REPAYMENT:
                        subject = subject+"回盘提前还款文件";
                        break;
                }
            }else {
                subject = subject+"回盘文件";
            }

            String text = "请查收附件，系统自动发出不用回复。";
            sendErrorMailAttachment(attachedBytes,toUserArray,text,subject);
        } catch (Exception e1) {
            logger.error(e1.getMessage(),e1);
        }
    }

    @Override
    public HuBeiDocName generateHuBeiDoc(HubeiCFCDataType hubeiCFCDataType,Byte seqNo){
        String docName = HubeiCFCUtil.getDocName(hubeiCFCDataType,seqNo);
        HuBeiDocName huBeiDocName = new HuBeiDocName();
        huBeiDocName.setDocName(docName);
        huBeiDocName.setDataType(hubeiCFCDataType.get());
        Long docId = huBeiCFCDao.saveDocName(huBeiDocName);
        huBeiDocName.setId(docId);
        return huBeiDocName;
    }

    /**
     * 保存发送和接收的内容方便历史查询
     * @param fileContent
     * @param fileName
     */
    public void saveHistoryFile(String fileContent,String fileName){
        if(StringUtils.isEmpty(fileContent)||StringUtils.isEmpty(fileName)){
            logger.info("saveHistoryFile fileContent or fileName is empty");
            return;
        }
        try {
            fileName = fileName +"_"+System.currentTimeMillis();
            createDir();
            File file = new File(hubeiHistoryFilePath+DateUtil.sampleDateFormatFormat(new Date())+File.separatorChar+fileName);
            file.createNewFile();
            try(BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(file))){
                outputStream.write(fileContent.getBytes(EncodingConfig.defaultCharset));
                outputStream.flush();
            }
            logger.info("File save successfully!");
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
            logger.info("save fileContent fail,fileName={}, content=",fileName);
            logger.info("{}",fileContent);
        }
    }

    public void createDir(){
        String dateFormat = DateUtil.sampleDateFormatFormat(new Date());
        File fileDir = new File(hubeiHistoryFilePath+dateFormat+File.separatorChar);
        if(!fileDir.exists()||!fileDir.isDirectory()){
            fileDir.mkdir();
        }
    }



}
