package cn.quantgroup.financial.service.sys.impl;

import cn.quantgroup.financial.model.CompensationDayBean;
import cn.quantgroup.financial.service.sys.ICompensationDayService;
import cn.quantgroup.financial.util.GsonUtil;
import com.google.gson.reflect.TypeToken;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.InputStreamReader;
import java.lang.reflect.Type;
import java.util.*;

/**
 * 维护代偿日
 * Created by WuKong on 2017/1/6.
 */
@Service
public class CompensationDayServiceImpl implements ICompensationDayService {
    private static final Logger LOGGER = LoggerFactory.getLogger(CompensationDayServiceImpl.class);

    @Value("${compensationday.file.path}")
    private String compensationDayFilePath;


    Map<String, CompensationDayBean> compensationDayBeanMap = Collections.emptyMap();

    /**
     * 当天是否为代偿日
     * @return
     */
    @Override
    public boolean isCompensationDay() {
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        CompensationDayBean compensationDayBean = compensationDayBeanMap.get("y" + year);
        if(compensationDayBean==null){
            LOGGER.error("can`t find year={} compensation days information",year);
            return false;
        }
        return compensationDayBean.isCompensationDay(calendar);
    }

    @Override
    public boolean isCompensationDay(Calendar calendar) {
        int year = calendar.get(Calendar.YEAR);
        CompensationDayBean compensationDayBean = compensationDayBeanMap.get("y" + year);
        if(compensationDayBean==null){
            LOGGER.error("can`t find year={} compensation days information",year);
            return false;
        }
        return compensationDayBean.isCompensationDay(calendar);
    }
    /**
     * 通过月查询代偿日
     * @param month 1-12
     * @return
     */
    @Override
    public Date getCompensationDay(Integer year, Integer month){
        CompensationDayBean compensationDayBean = compensationDayBeanMap.get("y" + year);
        if(compensationDayBean==null){
            LOGGER.error("can`t find year={} compensation days information",year);
            return null;
        }
        return compensationDayBean.getCompensationDay(month);
    }

    /**
     * 通过传入date计算代偿日
     * @param date
     * @return
     */
    @Override
    public Date computeCompenstationDay(Date date) {
        if(date==null){
            LOGGER.info("date is null");
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        CompensationDayBean compensationDayBean = compensationDayBeanMap.get("y" + calendar.get(Calendar.YEAR));
        if(compensationDayBean==null){
            LOGGER.error("can`t find year={} compensation days information",calendar.get(Calendar.YEAR));
            return null;
        }
        Date compensationDayDate = compensationDayBean.getCompensationDay(calendar.get(Calendar.MONTH)+1);
        if(compensationDayDate==null){
            LOGGER.error("can`t find year={} month={} compensation days information",calendar.get(Calendar.YEAR),calendar.get(Calendar.MONTH)+1);
            return null;
        }
        if(date.after(compensationDayDate)){
            //如果是本月代偿日之后 则是下个月的
            calendar.add(Calendar.MONTH,1);
            compensationDayBean = compensationDayBeanMap.get("y" + calendar.get(Calendar.YEAR));
            if(compensationDayBean==null){
                LOGGER.error("can`t find year={} compensation days information",calendar.get(Calendar.YEAR));
                return null;
            }
            compensationDayDate = compensationDayBean.getCompensationDay(calendar.get(Calendar.MONTH)+1);
            if(compensationDayDate==null){
                LOGGER.error("can`t find year={} month={} compensation days information",calendar.get(Calendar.YEAR),calendar.get(Calendar.MONTH)+1);
                return null;
            }
        }
        return compensationDayDate;
    }

    @Override
    public void prepareCompenstationDaysData() throws Exception {

        // 读取本地假日数据文件
        InputStreamReader compenstationDaysFileReader = new InputStreamReader(this.getClass().getResourceAsStream("/"+compensationDayFilePath));
        Type type = new TypeToken<HashMap<String, CompensationDayBean>>() {}.getType();
        compensationDayBeanMap = GsonUtil.GSON.fromJson(compenstationDaysFileReader, type);
        if (compensationDayBeanMap == null || CollectionUtils.sizeIsEmpty(compensationDayBeanMap)) {
            throw new Exception("本地未提供有效的节假日数据");
        }
        int year = Calendar.getInstance().get(Calendar.YEAR);
        if (!compensationDayBeanMap.containsKey("y" + year)) {
            LOGGER.error("本地代偿日信息不正确, 没有本年(" + year + ")的代偿日信息 :: " + compensationDayFilePath);
            throw new Exception("本地代偿日信息不正确, 没有本年(" + year + ")的代偿日信息 :: " + compensationDayFilePath);
        }
        for(String key: compensationDayBeanMap.keySet()){
            compensationDayBeanMap.get(key).init();
        }
    }
}
