package cn.quantgroup.financial.util;

import cn.quantgroup.financial.model.HttpResult;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.*;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.*;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

/**
 * Created by WuKong on 2017/1/18.
 */
public class NetUtil {

    private static Logger logger = LoggerFactory.getLogger(NetUtil.class);
    private static int time_out = 120;
    private static int connect_out = 30;


    public static enum  RequestMethod {
        POST("POST"),
        POST_BODY("POST"),
        GET("GET");

        private String method;

        RequestMethod(String method) {
            this.method = method;
        }

        public String get() {
            return method;
        }
    }

    /**
     * 请求的状态
     */
    public static class  RequestStatus {

        public static final int SUCCESS_STATUS = 0;

        public static final int UNKNOWNHOST_STATUS = 1;

        public static final int SOCKET_STAUS = 2;

        public static final int UNKNOWN_STATUS = 3;

        public static final int SSLHANDSHAKE_STATUS = 4;
        public static final int FILENOTFOUND_STATUS = 5;

        public static final int UNAVAILABLE_STATUS = 6;
    }

    /**
     *
     * @param url
     * @param params
     * @param headerParams
     * @param requestMethod
     * @param retryNums
     * @return
     */
    public static String getCookies(String url, Map<String,String> params, Map<String,String> headerParams, RequestMethod requestMethod, Integer retryNums){
        HttpResult httpResult = requestUrl(url,params,headerParams,requestMethod,2,retryNums);
        if(httpResult==null){
            return null;
        }else {
            return httpResult.getCookies();
        }
    }

    /**
     *
     * Post请求
     * key value
     * @param url
     * @param params
     * @param headerParams
     * @param retryNums
     * @return
     */
    public static HttpResult postRequestUrl(String url,Map<String,String> params,Map<String,String> headerParams,Integer retryNums){
        HttpResult httpResult = requestUrl(url,params,headerParams,RequestMethod.POST,1,retryNums);
        return httpResult;
    }

    /**
     * post 请求
     *  body
     * @param url
     * @param requestBody
     * @param headerParams
     * @param retryNums
     * @return
     */
    public static HttpResult postRequestUrl(String url,String requestBody,Map<String,String> headerParams,Integer retryNums){
        Map<String,String> params = new HashMap<>();
        params.put("body",requestBody);
        HttpResult httpResult = requestUrl(url,params,headerParams,RequestMethod.POST_BODY,1,retryNums);
        return httpResult;
    }

    /**
     * GET请求
     * @param url
     * @param headerParams
     * @param retryNums
     * @return
     */
    public static HttpResult getRequestUrl(String url,Map<String,String> headerParams,Integer retryNums){
        HttpResult httpResult = requestUrl(url,null,headerParams,RequestMethod.GET,1,retryNums);
        return httpResult;
    }


    public static javax.net.ssl.SSLSocketFactory trustAllSsl(){
        try {
            // First create a trust manager that won't care.
            X509TrustManager trustManager = new X509TrustManager() {
                public void checkClientTrusted(X509Certificate[] chain, String authType)
                        throws CertificateException {
                    // Don't do anything.
                }

                public void checkServerTrusted(X509Certificate[] chain, String authType)
                        throws CertificateException {
                    // Don't do anything.
                }

                public X509Certificate[] getAcceptedIssuers() {
                    // Don't do anything.
                    return null;
                }
            };

            // Now put the trust manager into an SSLContext.
            SSLContext sslcontext = SSLContext.getInstance("TLS");

            sslcontext.init(null, new TrustManager[]{trustManager}, new java.security.SecureRandom());

            return sslcontext.getSocketFactory();

        } catch (KeyManagementException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return null;
    }
    /**
     *
     * @param urlPath
     * @param params
     * @param headerParams
     * @param requestMethod
     * @param mode 1 只获取返回内容 2 只获取cookies 3 获取内容和cookies
     * @param retryNums  重试次数
     * @return
     */
    public static HttpResult requestUrl(String urlPath,Map<String,String> params,Map<String,String> headerParams,RequestMethod requestMethod,Integer mode,Integer retryNums) {
        HttpURLConnection connectionHttp = null;
        HttpsURLConnection connectionHttps = null;
        InputStream inputStream=null;

        //http状态码
        HttpResult httpResult = new HttpResult();
        httpResult.setHttpCode(HttpURLConnection.HTTP_OK);
        //重试次数
        if(retryNums==null||retryNums<=0){
            retryNums = 1;
        }
        if(retryNums>10){
            retryNums = 10;
        }
        for(int i=0;i<retryNums;i++){
            try {
                URL url = new URL(urlPath);
                if(urlPath.startsWith("https")){
                    connectionHttps = (HttpsURLConnection) url.openConnection();
                    connectionHttps.setSSLSocketFactory(trustAllSsl());
                    connectionHttp = connectionHttps;
                    connectionHttps = null;
                }else {
                    connectionHttp = (HttpURLConnection) url.openConnection();
                }

                //设置请求的模式
                if(requestMethod.equals(RequestMethod.POST)||requestMethod.equals(RequestMethod.POST_BODY)){
                    connectionHttp.setRequestMethod(requestMethod.get());
                    //设置连接允许写入
                    connectionHttp.setDoOutput(true);
                    connectionHttp.setDoInput(true);
                }else if(requestMethod.equals(RequestMethod.GET)) {
                    connectionHttp.setRequestMethod(requestMethod.get());
                }
                //设置请求连接超时时间
                connectionHttp.setConnectTimeout(1000 * connect_out);
                //设置访问时的超时时间
                connectionHttp.setReadTimeout(1000 * time_out);
//				connectionHttp.setFollowRedirects(true);
                if(headerParams!=null&&headerParams.size()>0){
                    //设置header
                    for(Map.Entry<String,String> entry:headerParams.entrySet()){
                        connectionHttp.setRequestProperty(entry.getKey(), entry.getValue());
                    }
                }

                // 必须设置false，否则会自动redirect到Location的地址
                connectionHttp.setInstanceFollowRedirects(false);

                StringBuilder content = new StringBuilder();

                if(requestMethod.equals(RequestMethod.POST)){
                    if(params!=null&&params.size()>0){
                        //设置头文件内容
                        try(OutputStream outputStream = connectionHttp.getOutputStream()){
                            int index=0;
                            for(Map.Entry<String,String> entry:params.entrySet()){
                                if(index>0){
                                    content.append("&");
                                }
                                content.append(entry.getKey()).append("=").append(URLEncoder.encode(entry.getValue(),"UTF-8"));
                                index++;
                            }
                            outputStream.write(content.toString().getBytes("UTF-8"));
                            outputStream.flush();
                            outputStream.close();
                        }
                    }
                }else if(requestMethod.equals(RequestMethod.POST_BODY)){
                    if(params!=null&&params.containsKey("body")){

                        //设置头文件内容
                        try(OutputStream outputStream = connectionHttp.getOutputStream()){
                            outputStream.write(params.get("body").getBytes("UTF-8"));
                            outputStream.flush();
                            outputStream.close();
                        }
                    }

                }


                //开启连接
                connectionHttp.connect();

                //标识返回数据是否经过压缩
                boolean isGzip = false;
                String contentEncoding = connectionHttp.getHeaderField("Content-Encoding");
                if(!StringUtils.isEmpty(contentEncoding)&&contentEncoding.equalsIgnoreCase("gzip")){
                    isGzip = true;
                }
                if(connectionHttp.getResponseCode()==HttpURLConnection.HTTP_UNAUTHORIZED){
                    httpResult.setHttpCode(connectionHttp.getResponseCode());
                    httpResult.setStatus(RequestStatus.SUCCESS_STATUS);
                    return httpResult;
                }else if(connectionHttp.getResponseCode()==HttpURLConnection.HTTP_UNAVAILABLE){
                    httpResult.setHttpCode(connectionHttp.getResponseCode());
                    httpResult.setStatus(RequestStatus.SUCCESS_STATUS);
                    return httpResult;
                }else {
                    httpResult.setHttpCode(connectionHttp.getResponseCode());
                    httpResult.setStatus(RequestStatus.SUCCESS_STATUS);
                    if(mode.equals(1)){
                        //获得连接的输入流

                        if(isGzip){
                            inputStream = new GZIPInputStream(connectionHttp.getInputStream());
                        }else {
                            inputStream= connectionHttp.getInputStream();
                        }
                        String result = IOUtils.toString(inputStream, "UTF-8");
                        httpResult.setResult(result);
                        httpResult.setHttpCode(connectionHttp.getResponseCode());
                        return httpResult;
                    }else if(mode.equals(2)){
                        String cookies = getCookies(connectionHttp);
                        httpResult.setCookies(cookies);
                        return httpResult;
                    }else if(mode.equals(3)){
                        //获得连接的输入流
                        if(isGzip){
                            inputStream = new GZIPInputStream(connectionHttp.getInputStream());
                        }else {
                            inputStream= connectionHttp.getInputStream();
                        }
                        String result = IOUtils.toString(inputStream, "UTF-8");
                        String cookies = getCookies(connectionHttp);
                        httpResult.setCookies(cookies);
                        httpResult.setResult(result);
                        return httpResult;
                    }
                }



            } catch (MalformedURLException e) {
                logger.error(e.getMessage(), e);
            }catch (UnknownHostException e){
                httpResult.setStatus(RequestStatus.UNKNOWNHOST_STATUS);
                logger.error("UnknownHostException Exception url = {},httpCode={}",urlPath,httpResult.getHttpCode());
            }catch (ProtocolException e) {
                logger.error(e.getMessage(),e);
            }catch (SocketTimeoutException e){
                httpResult.setStatus(RequestStatus.SOCKET_STAUS);
                logger.error("SocketTimeOut Exception url = {},httpCode={}",urlPath,httpResult.getHttpCode());
            }catch (SocketException e){
                httpResult.setStatus(RequestStatus.SOCKET_STAUS);
                //java.net.SocketException: Software caused connection abort: socket write error
                //服务器的并发连接数超过了其承载量，服务器会将其中一些连接Down掉
                //客户关掉了浏览器，而服务器还在给客户端发送数据
                logger.error(e.getMessage(),e);
            }catch (SSLHandshakeException e){
                httpResult.setStatus(RequestStatus.SSLHANDSHAKE_STATUS);
                //javax.net.ssl.SSLHandshakeException: Remote host closed connection during handshake
                //网络超时
                logger.error(e.getMessage(),e);
            } catch (SSLException e){
                httpResult.setStatus(RequestStatus.SSLHANDSHAKE_STATUS);
                logger.error("SSLException Exception url = {},httpCode={}",urlPath,httpResult.getHttpCode());
            }catch (FileNotFoundException e){
                httpResult.setStatus(RequestStatus.FILENOTFOUND_STATUS);
                logger.error("FileNotFoundException Exception url = {},httpCode={}",urlPath,httpResult.getHttpCode());
            }catch (IOException e) {
                //java.io.IOException: Server returned HTTP response code: 503
                httpResult.setStatus(RequestStatus.UNKNOWN_STATUS);
                logger.info("httpCode={}",httpResult.getHttpCode());
                logger.error(e.getMessage(), e);
            } catch (Exception e){
                httpResult.setStatus(RequestStatus.UNKNOWN_STATUS);
                logger.info("httpCode={}",httpResult.getHttpCode());
                logger.error(e.getMessage(),e);
            }finally {
                try {
                    if(inputStream!=null){
                        //关闭流和连接
                        inputStream.close();
                    }
                } catch (IOException e) {
                    logger.error(e.getMessage(), e);
                }
                try {
                    if(connectionHttp !=null){
                        connectionHttp.disconnect();
                    }
                } catch (Exception e) {
                    logger.error(e.getMessage(),e);
                }
            }
        }
        return httpResult;
    }




    private static String getCookies(HttpURLConnection connectionHttp){
        String cookies = "";

        Map<String, List<String>> headerFields = connectionHttp.getHeaderFields();
        if(headerFields!=null){
            List<String> cookieList = headerFields.get("Set-Cookie");
            if(cookieList!=null&&cookieList.size()>0){
                for(String cookie : cookieList){
                    if(StringUtils.isEmpty(cookies)){
                        cookies = cookie;
                    }else {
                        cookies = cookies+";"+cookie;
                    }
                }
            }
        }


        return cookies;
    }

    public static Map<String,String> getCookiesMap(String cookie){

        Map<String,String> cookiesMap = new HashMap<String,String>();
        if(!StringUtils.isEmpty(cookie)){
            String[] items = cookie.split(";");
            if(items!=null&&items.length>0){
                for(String item:items){
                    String[] kv = item.split("=");
                    if(kv!=null&&kv.length==2){
                        cookiesMap.put(kv[0].toLowerCase(),kv[1]);
                    }
                }
            }
        }

        return cookiesMap;
    }


}
