package cn.quantgroup.financial.json;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.TypeMismatchException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.Ordered;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.AbstractHandlerMethodExceptionResolver;

import cn.quantgroup.financial.util.ResultCode;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 
 * @author dave
 *
 */
public class JsonExceptionResolver extends AbstractHandlerMethodExceptionResolver implements Ordered {

    /**
     * Logger for this class
     */
    private static final Logger logger = LoggerFactory.getLogger(JsonExceptionResolver.class);

    private static final String DEFAULT_VIEW_NAME = "jsonView";
    
    @Value(value = "0")
    private int order = Ordered.LOWEST_PRECEDENCE;

    private String viewName = DEFAULT_VIEW_NAME;

    @Override
    protected boolean shouldApplyTo(HttpServletRequest request, Object handler) {

        if (super.shouldApplyTo(request, handler)) {
            if (handler != null) {
                HandlerMethod handlerMethod = (HandlerMethod) handler;

                if (JsonResponse.class.isAssignableFrom(handlerMethod.getMethod().getReturnType())) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    protected ModelAndView doResolveHandlerMethodException(HttpServletRequest request, HttpServletResponse response,
            HandlerMethod handlerMethod, Exception ex) {
        ModelAndView mv = new ModelAndView(viewName);

        JsonResponse jr = null;
        if (ex instanceof MissingServletRequestParameterException) {
            jr = new JsonResponse(ResultCode.MISSING_ARGUMENT, ex.getMessage());
        } else if (ex instanceof TypeMismatchException) {
            jr = new JsonResponse(ResultCode.ARGUMENT_TYPE_ERROR, ex.getMessage());
        } else if(ex instanceof HttpRequestMethodNotSupportedException){
            jr = new JsonResponse(ResultCode.METHOD_TYPE_ERROR, ex.getMessage());
        }else {
            logger.error("occured an exception", ex);
            jr = new JsonResponse(ResultCode.UNKNOWN_ERROR, "未知异常");
        }
        mv.addObject(JsonResponseMethodReturnValueHandler.JR_MODEL_KEY, jr);
        return mv;
    }

    public void setViewName(String viewName) {
        this.viewName = viewName;
    }

}
