package cn.quantgroup.financial.dao.impl;

import cn.quantgroup.financial.dao.IPaymentDao;
import cn.quantgroup.financial.exception.ArgsInvaildException;
import cn.quantgroup.financial.mapper.PaymentDetailMapper;
import cn.quantgroup.financial.mapper.RepaymentPlanDetailMapper;
import cn.quantgroup.financial.model.PaymentDetail;
import cn.quantgroup.financial.model.RepaymentPlanDetail;
import cn.quantgroup.financial.model.RepaymentPlanStatus;
import cn.quantgroup.financial.util.DateUtil;
import com.alibaba.fastjson.JSON;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Created by WuKong on 2017/1/6.
 */
@Service
public class PaymentDaoImpl implements IPaymentDao {

    private static final Logger logger = LoggerFactory.getLogger(PaymentDaoImpl.class);

    @Autowired
    private PaymentDetailMapper paymentDetailMapper;

    @Autowired
    private RepaymentPlanDetailMapper repaymentPlanDetailMapper;


    /**
     * 查询代偿状态
     * @param loanHistoryId 借款id
     * @param repaymentPlanId 还款计划id
     * @return
     * @throws ArgsInvaildException
     */
    @Override
    public List<RepaymentPlanStatus> getRepaymentPlanStatus(Long loanHistoryId,Long repaymentPlanId) throws ArgsInvaildException {
        if(loanHistoryId==null&&repaymentPlanId==null){
            throw new ArgsInvaildException("loanHistoryId and repaymentPlanId both is null");
        }
        return repaymentPlanDetailMapper.getRepaymentPlanStatus(loanHistoryId,repaymentPlanId);
    }


    @Override
    public List<RepaymentPlanDetail> getRepaymentPlanListByLoanIds(List<Long> loanHistoryIdList){
        if(CollectionUtils.isEmpty(loanHistoryIdList)){
            return new ArrayList<RepaymentPlanDetail>();
        }
        return repaymentPlanDetailMapper.getRepaymentPlanListByLoanIds(loanHistoryIdList);
    }

    @Override
    public PaymentDetail getByLoanId(Long loanHistoryId){
        return paymentDetailMapper.getByLoanHistoryId(loanHistoryId);
    }

    @Override
    public List<RepaymentPlanDetail> getListByLoanIdsAndCompensationStatus(List<Long> loanHistoryIdList,Byte compensationStatus){
        if(org.springframework.util.CollectionUtils.isEmpty(loanHistoryIdList)){
            return new ArrayList<RepaymentPlanDetail>();
        }
        return repaymentPlanDetailMapper.getListByLoanIdsAndCompensationStatus(loanHistoryIdList,compensationStatus);
    }

    /**
     * 通过代偿日以及loanIds 来查询一组代偿的还款计划
     * @param loanHistoryIdList
     * @param compensationDate
     * @return
     */
    @Override
    public List<RepaymentPlanDetail> getListByCompensationDateAndLoanIds(List<Long> loanHistoryIdList,Date compensationDate){
        if(CollectionUtils.isEmpty(loanHistoryIdList)||compensationDate==null){
            return null;
        }
        return repaymentPlanDetailMapper.getListByCompensationDateAndLoanIds(compensationDate,loanHistoryIdList);
    }

    @Override
    public RepaymentPlanDetail getByLoanIdAndTerm(Long loanHistoryId,Integer term){
        if(loanHistoryId==null||term==null){
            return null;
        }
        return repaymentPlanDetailMapper.getByLoanIdAndTerm(loanHistoryId,term);
    }

    /**
     *
     * 保存放款信息 和 还款计划信息
     * @param paymentDetail 数据装填好再传入
     * @return
     */
    @Override
    public Integer savePaymentDetailAndRepaymentPlan(PaymentDetail paymentDetail){
        paymentDetail.setCreateTime(new Date());
        paymentDetail.setUpdateTime(new Date());
        PaymentDetail paymentDetailOld = paymentDetailMapper.getByLoanHistoryId(paymentDetail.getLoanHistoryId());
        Integer row = 0;
        if(paymentDetailOld==null){
            row = paymentDetailMapper.insertSelective(paymentDetail);
        }else {
            logger.info("paymentDetail id={} already exist ,json={}",paymentDetail.getLoanHistoryId(), JSON.toJSONString(paymentDetail));
            return row;
        }
        if(row>0){
            paymentDetail.getRepaymentPlanDetailList().forEach(repaymentPlanDetail -> {
                repaymentPlanDetailMapper.insertSelective(repaymentPlanDetail);
            });
        }
        return row;
    }

    @Override
    public List<Long> getIdListByCompensationDate(Date compensationDate){
        if(compensationDate==null){
            logger.info("compensationDate is empty");
            return null;
        }
        List<Long> idList = repaymentPlanDetailMapper.getIdListByCompensationDate(compensationDate);
        return idList;
    }

    @Override
    public List<PaymentDetail> getListByMerchantContractNo(List<String> contractNoList){
        return paymentDetailMapper.getByMerchantContractNo(contractNoList);
    }


    @Override
    public List<Long> getIdListBeforeCompensationDate(Date compensationDate){
        if(compensationDate==null){
            logger.info("compensationDate is empty");
            return null;
        }
        List<Long> idList = repaymentPlanDetailMapper.getIdListByNotYetCompensation(compensationDate);
        return idList;
    }
    @Override
    public Integer updateBatchCompensationStatusById(List<Long> idList,Byte compensationStatus){
        if(CollectionUtils.isEmpty(idList)||compensationStatus==null){
            logger.info("idList or compensationStatus is empty");
            return null;
        }
        Integer row = repaymentPlanDetailMapper.updateBatchCompensationStatusById(idList,compensationStatus);
        return row;
    }



    @Override
    public Integer updateBatchCompensationStatusBeforeDate(Date beforeDate,Byte compensationStatus){
        Integer row = repaymentPlanDetailMapper.updateBatchCompensationStatusBeforeDate(beforeDate,compensationStatus);
        return  row;
    }

    @Override
    public Integer updateMerchantContractNo(String contractNo,Long loanHistoryId){
        return paymentDetailMapper.updateMerchantContractNoByLoanHistoryId(contractNo,loanHistoryId);
    }
}
