package cn.quantgroup.financial.service.sys.impl;

import cn.quantgroup.financial.constant.EncodingConfig;
import cn.quantgroup.financial.dao.IHuBeiCFCDao;
import cn.quantgroup.financial.dao.ISystemDao;
import cn.quantgroup.financial.model.MailInfo;
import cn.quantgroup.financial.service.sys.IMailSendCallback;
import cn.quantgroup.financial.service.sys.IMailService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.mail.javamail.MimeMessagePreparator;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import javax.mail.util.ByteArrayDataSource;
import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;


@Service
public class MailServiceImpl implements IMailService {

    @Autowired
    private JavaMailSender javaMailSender;

    @Autowired
    private IHuBeiCFCDao huBeiCFCDao;

    @Autowired
    private ISystemDao systemDao;

    private static final Logger logger = LoggerFactory.getLogger(MailServiceImpl.class);

    /**
     * 异步发送邮件
     *
     * @param from
     * @param sendTo
     * @param ccTo
     * @param subject
     * @param text
     * @param attachedFileList 附件列表(绝对路径)
     * @param mailSendCallback
     */
    @Override
    @Async
    public void sendAttachmentMailAsync(String from, String sendTo, String ccTo, String subject, String text,
                                        ArrayList<String> attachedFileList, IMailSendCallback mailSendCallback) {
        sendAttachmentMailAsync(from, new String[]{sendTo}, new String[]{ccTo}, subject, text, attachedFileList, mailSendCallback);
    }

    @Override
    public void sendAttachmentMailAsync(String from, String[] sendTo, String[] ccTo, String subject, String text, ArrayList<String> attachedFileList, IMailSendCallback mailSendCallback) {

        MimeMessagePreparator mimeMessagePreparator = new MimeMessagePreparator() {
            @Override
            public void prepare(MimeMessage mimeMessage) throws Exception {
                MimeMessageHelper mimeMessageHelper;
                if (CollectionUtils.isNotEmpty(attachedFileList)) {
                    mimeMessageHelper = new MimeMessageHelper(mimeMessage, true);
                    for (String filePath : attachedFileList) {
                        File attachedFile = hasValidAttachent(filePath);
                        if (attachedFile != null) {
                            // 添加附件, 强制使用 utf-8和中文编码
                            mimeMessageHelper.addAttachment(MimeUtility.encodeWord(attachedFile.getName(), "UTF-8", "B"), new FileDataSource(attachedFile));
                        }
                    }
                } else {
                    mimeMessageHelper = new MimeMessageHelper(mimeMessage);
                }
                // 设置发件人昵称和地址
                mimeMessageHelper.setFrom("财务系统<" + from + ">");
                mimeMessageHelper.setTo(sendTo);
                if (StringUtils.isNoneBlank(ccTo)) {
                    mimeMessageHelper.setCc(ccTo);
                }
                mimeMessageHelper.setSubject(subject);
                mimeMessageHelper.setText(text);

            }
        };

        if (mailSendCallback != null) {
            sendMailAsync(mimeMessagePreparator, mailSendCallback);
        } else {
            sendMailAsync(mimeMessagePreparator);
        }
    }

    @Override
    @Async
    public void sendAttachmentMailWithBytesAsync(String from, String sendTo, String ccTo, String subject, String text, ArrayList<DataSource> attachedBytes, IMailSendCallback mailSendCallback) {
        sendAttachmentMailWithBytesAsync(from,new String[]{sendTo},new String[]{ccTo},subject,text,attachedBytes,mailSendCallback);
    }

    @Override
    public DataSource getDataSource(String content,String fileName){
        ByteArrayDataSource dataSource = new ByteArrayDataSource(content.getBytes(EncodingConfig.defaultCharset), "text/html;charset=UTF-8");
        dataSource.setName(fileName);
        return dataSource;
    }

    @Override
    public void sendAttachmentMailWithBytesAsync(String from, String[] sendTo, String[] ccTo, String subject, String text, ArrayList<DataSource> attachedBytes, IMailSendCallback mailSendCallback) {
        MimeMessagePreparator mimeMessagePreparator = new MimeMessagePreparator() {
            @Override
            public void prepare(MimeMessage mimeMessage) throws Exception {
                MimeMessageHelper mimeMessageHelper;
                if (CollectionUtils.isNotEmpty(attachedBytes)) {
                    mimeMessageHelper = new MimeMessageHelper(mimeMessage, true);
                    for (DataSource dataSource : attachedBytes) {
                        mimeMessageHelper.addAttachment(dataSource.getName(),dataSource);
                    }
                } else {
                    mimeMessageHelper = new MimeMessageHelper(mimeMessage);
                }
                mimeMessageHelper.setFrom("财务系统<" + from + ">");
                mimeMessageHelper.setTo(sendTo);
                if (StringUtils.isNoneBlank(ccTo)) {
                    mimeMessageHelper.setCc(ccTo);
                }
                mimeMessageHelper.setSubject(subject);
                mimeMessageHelper.setText(text);

            }
        };

        if (mailSendCallback != null) {
            sendMailAsync(mimeMessagePreparator, mailSendCallback);
        } else {
            sendMailAsync(mimeMessagePreparator);
        }
    }

    /**
     * 发送简单邮件
     *
     * @param from
     * @param sendTo
     * @param ccTo
     * @param subject
     * @param text
     */
    @Override
    public void sendSimpleMailAsync(String from, String sendTo, String ccTo, String subject, String text) {
        sendAttachmentMailAsync(from, sendTo, ccTo, subject, text, null, null);
    }

    @Override
    public void sendSimpleMailAsync(String from, String[] sendTo, String[] ccTo, String subject, String text) {
        sendAttachmentMailAsync(from, sendTo, ccTo, subject, text, null, null);
    }

    @Async
    @Override
    public void sendMailAsync(SimpleMailMessage[] messages, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(messages);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Override
    public void sendSimpleMail(SimpleMailMessage message){
        try {
            javaMailSender.send(message);
        }catch (Exception e){
            logger.error(e.getMessage(),e);
        }
    }
    @Async
    @Override
    public void sendMailAsync(SimpleMailMessage message, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(message);
            logger.info("send mail ok");
        }catch (Exception e){
            except = e;
        }catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Async
    @Override
    public void sendMailAsync(MimeMessage message, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(message);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Async
    @Override
    public void sendMailAsync(MimeMessage[] messages, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(messages);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Async
    @Override
    public void sendMailAsync(MimeMessagePreparator message, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(message);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Async
    @Override
    public void sendMailAsync(MimeMessagePreparator[] messages, IMailSendCallback... callbacks) {
        Throwable except = null;
        try {
            javaMailSender.send(messages);
        } catch (Throwable e) {
            except = e;
        }
        if (callbacks != null && callbacks.length != 0) {
            callbacks[0].doWhenMailSendComplete(except);
        }
    }

    @Override
    public void sendMail(SimpleMailMessage... simpleMailMessages) {
        javaMailSender.send(simpleMailMessages);
    }

    @Override
    public void sendMail(SimpleMailMessage simpleMailMessage) {
        javaMailSender.send(simpleMailMessage);
    }

    @Override
    public void sendMail(MimeMessage mimeMessage) {
        javaMailSender.send(mimeMessage);
    }

    @Override
    public void sendMail(MimeMessage... mimeMessages) {
        javaMailSender.send(mimeMessages);
    }

    @Override
    public void sendMail(MimeMessagePreparator message) {
        javaMailSender.send(message);
    }

    @Override
    public void sendMail(MimeMessagePreparator... messages) {
        javaMailSender.send(messages);
    }

    @Override
    public void sendMailQuietly(SimpleMailMessage... simpleMailMessages) {
        javaMailSender.send(simpleMailMessages);
    }

    @Override
    public void sendMailQuietly(SimpleMailMessage simpleMailMessage) {
        javaMailSender.send(simpleMailMessage);
    }

    @Override
    public void sendMailQuietly(MimeMessage mimeMessage) {
        javaMailSender.send(mimeMessage);
    }

    @Override
    public void sendMailQuietly(MimeMessage... mimeMessages) {
        javaMailSender.send(mimeMessages);
    }

    @Override
    public void sendMailQuietly(MimeMessagePreparator message) {
        javaMailSender.send(message);
    }

    @Override
    public void sendMailQuietly(MimeMessagePreparator... messages) {
        javaMailSender.send(messages);
    }

    @Override
    public MimeMessage createMimeMessage() {
        return javaMailSender.createMimeMessage();
    }

    @Override
    public MimeMessage createMimeMessage(InputStream var1) throws MailException {
        return javaMailSender.createMimeMessage(var1);
    }

    /**
     * 判断是否是有效附件
     *
     * @param attachedFileAbdPath 附件文件绝对路径
     * @return File or null
     */
    private File hasValidAttachent(String attachedFileAbdPath) {
        if (StringUtils.isBlank(attachedFileAbdPath)) {
            return null;
        }

        File file = new File(attachedFileAbdPath);
        return (file.exists() && file.isFile()) ? file : null;
    }

    /**
     * 保存邮件信息
     * @param mailInfo
     * @return
     */
    @Override
    public Long saveMailInfo(MailInfo mailInfo){
        return systemDao.saveMailInfo(mailInfo);
    }


    @Override
    public Integer deleteMailInfo(Long id){
        return systemDao.deleteMailInfo(id);
    }

}
