package cn.quantgroup.financial.config;

import cn.quantgroup.financial.aspect.data.RoutingDataSourceProxy;
import cn.quantgroup.financial.constant.DataBaseType;
import com.alibaba.druid.pool.DruidDataSourceFactory;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.annotation.MapperScan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.env.Environment;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;

import javax.sql.DataSource;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Created by WuKong on 2017/1/9.
 */
@Configuration
@MapperScan(basePackages = "cn.quantgroup.financial.mapper")
public class MyBatisConfig {

    private static final Logger logger = LoggerFactory.getLogger(MyBatisConfig.class);

    @Autowired
    private Environment env;

    /**
     * 创建数据源(数据源的名称：方法名可以取为XXXDataSource(),XXX为数据库名称,该名称也就是数据源的名称)
     */
    @Bean
    public DataSource apiDBDataSource() throws Exception {
        try {
            logger.error("finance api config-------------------api jdbc url={}",env.getProperty("api.jdbc.url"));
            Properties props = new Properties();
            props.put("driverClassName", env.getProperty("api.jdbc.driverClassName"));
            props.put("url", env.getProperty("api.jdbc.url"));
            props.put("username", env.getProperty("api.jdbc.username"));
            props.put("password", env.getProperty("api.jdbc.password"));
            props.put("maxActive",env.getProperty("api.jdbc.maxActive"));
            props.put("minIdle",env.getProperty("api.jdbc.minIdle"));
            props.put("maxWait","600");
            props.put("validationQuery","select 1 ");
            return DruidDataSourceFactory.createDataSource(props);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    @Bean
    public DataSource systemDBDataSource() throws Exception {
        try {
            logger.error("finance api config-------------------system jdbc url={}",env.getProperty("system.jdbc.url"));
            Properties props = new Properties();
            props.put("driverClassName", env.getProperty("system.jdbc.driverClassName"));
            props.put("url", env.getProperty("system.jdbc.url"));
            props.put("username", env.getProperty("system.jdbc.username"));
            props.put("password", env.getProperty("system.jdbc.password"));
            props.put("maxActive",env.getProperty("system.jdbc.maxActive"));
            props.put("minIdle",env.getProperty("system.jdbc.minIdle"));
            props.put("maxWait","600");
            props.put("validationQuery","select 1 ");
            return DruidDataSourceFactory.createDataSource(props);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    /**
     * @Primary 该注解表示在同一个接口有多个实现类可以注入的时候，默认选择哪一个，而不是让@autowire注解报错
     * @Qualifier 根据名称进行注入，通常是在具有相同的多个类型的实例的一个注入（例如有多个DataSource类型的实例）
     */
    @Bean
    @Primary
    public RoutingDataSourceProxy dataSource(@Qualifier("apiDBDataSource") DataSource apiDbDataSource,
                                        @Qualifier("systemDBDataSource") DataSource systemDb2DataSource) {
        try {
            Map<Object, Object> targetDataSources = new HashMap<>();
            targetDataSources.put(DataBaseType.Api_DB.get(), apiDbDataSource);
            targetDataSources.put(DataBaseType.System_DB.get(), systemDb2DataSource);

            RoutingDataSourceProxy dataSource = new RoutingDataSourceProxy();
            dataSource.setTargetDataSources(targetDataSources);// 该方法是AbstractRoutingDataSource的方法
            dataSource.setDefaultTargetDataSource(apiDbDataSource);// 默认的datasource设置为myTestDbDataSource

            return dataSource;
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    /**
     * 根据数据源创建SqlSessionFactory
     */
    @Bean
    public SqlSessionFactory sqlSessionFactory(RoutingDataSourceProxy ds) throws Exception {
        try {
            SqlSessionFactoryBean fb = new SqlSessionFactoryBean();
            fb.setDataSource(ds);// 指定数据源(这个必须有，否则报错)
            // 下边两句仅仅用于*.xml文件，如果整个持久层操作不需要使用到xml文件的话（只用注解就可以搞定），则不加
            fb.setTypeAliasesPackage(env.getProperty("mybatis.typeAliasesPackage"));// 指定基包
            fb.setTypeHandlersPackage(env.getProperty("mybatis.typeHandlersPackage"));//指定扫描typeHandler包
            fb.setMapperLocations(
                    new PathMatchingResourcePatternResolver().getResources(env.getProperty("mybatis.mapperLocations")));//

            return fb.getObject();
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return null;
    }

    /**
     * 配置事务管理器
     */
    @Bean
    public DataSourceTransactionManager transactionManager(RoutingDataSourceProxy dataSource) throws Exception {
        return new DataSourceTransactionManager(dataSource);
    }
}
