package cn.quantgroup.financial.util;

import cn.quantgroup.financial.constant.HubeiCFCDataType;
import cn.quantgroup.financial.constant.HubeiCFCField;
import cn.quantgroup.financial.constant.UserCardType;
import cn.quantgroup.financial.model.huibeicfc.HuBeiCFCResponse;
import cn.quantgroup.financial.model.huibeicfc.HuBeiHistory;
import cn.quantgroup.financial.model.huibeicfc.HuBeiJsonBean;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.dom4j.*;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by WuKong on 2017/1/17.
 */
public class HubeiCFCUtil {
    private static final Logger logger = LoggerFactory.getLogger(HubeiCFCUtil.class);


    public static void copy(HuBeiHistory history, HuBeiJsonBean jsonBean){
        jsonBean.setFlowNo(history.getFlowNo());
        jsonBean.setContactNo(history.getContactNo());
        jsonBean.setUserName(history.getUserName());
        jsonBean.setUserIdType(history.getUserIdType());
        jsonBean.setUserIdNo(history.getUserIdNo());
    }

    public static HuBeiHistory replicateBase(HuBeiHistory history){
        HuBeiHistory huBeiHistory = new HuBeiHistory();
        huBeiHistory.setFlowNo(history.getFlowNo());
        huBeiHistory.setContactNo(history.getContactNo());
        huBeiHistory.setCurrTermNo(history.getCurrTermNo());
        huBeiHistory.setUserIdType(history.getUserIdType());
        huBeiHistory.setUserIdNo(history.getUserIdNo());
        huBeiHistory.setUserId(history.getUserId());
        huBeiHistory.setUserName(history.getUserName());
        huBeiHistory.setHappenTime(new Date());
        return huBeiHistory;
    }

    /**
     * 计算逾期费
     * @param deadline
     * @param principal
     * @return
     */
    public static BigDecimal caculateOverDueFee(Date deadline,Date repayDate,BigDecimal principal){
        int overdueDays = caculateDays(deadline,repayDate);

        if(overdueDays<=3){
            //小于等于3天没有湖北罚息
            return BigDecimal.ZERO;
        }
        BigDecimal overDueFee = principal.multiply(BigDecimal.valueOf(overdueDays)).multiply(new BigDecimal("0.1401"));
        overDueFee = overDueFee.divide(BigDecimal.valueOf(360),6, RoundingMode.HALF_UP);
        return overDueFee;
    }

    /*
	 * 计算今天到过去某一天的总共天数
	 */
    public static int caculateDays(Date compareDay,Date repayDate) {
        long todayMilliseconds = repayDate.getTime();
        long compareMilliseconds = compareDay.getTime();
        // 获得两个日期之间的毫秒差。
        long differenceMilliseconds = todayMilliseconds - compareMilliseconds;
        // 一天的毫秒数
        long oneDayMilliseconds = 86400000;
        // 除以一天的毫秒数，就是相差的天数。
        int result = (int) (differenceMilliseconds / oneDayMilliseconds);
        return result;
    }
    /**
     * 解析证件号类型 内部用
     * @param idType
     * @return
     */
    public static Byte getIDType(String idType){
        if("20".equals(idType)){
            //身份证
            return Integer.valueOf(UserCardType.IDENTITY_CARD.get()).byteValue();
        }else {
            return NumberUtils.isNumber(idType)?Integer.valueOf(idType).byteValue():Byte.valueOf("-1");
        }
    }

    /**
     * 反解证件号 湖北用
     * @param idType
     * @return
     */
    public static String getHuBeiType(Byte idType){
        if(idType!=null&&UserCardType.IDENTITY_CARD.get()==idType.intValue()){
            return "20";
        }else {
            return idType.toString();
        }
    }
    public static Document generate() throws ParserConfigurationException {
        Document document = DocumentHelper.createDocument();
        Element rootElement = document.addElement(HubeiCFCField.root);
        Element headElement = rootElement.addElement(HubeiCFCField.head);

        Map<String,String> initalMap = new HashMap<String,String>(){
            {
                put(HubeiCFCField.channel,"7101");
                put(HubeiCFCField.flowNo,"");
                put(HubeiCFCField.reqTime,String.valueOf(DateUtil.getDateTime()));
                put(HubeiCFCField.sign,"");
                put(HubeiCFCField.signFlag,"");
            }
        };
        for(String key:initalMap.keySet()){
            Element element = headElement.addElement(key);
            element.setText(initalMap.get(key));
        }
        return document;
    }

    /**
     * 构建回盘文件请求
     * @param argsMap
     * @param currentBusinessCode
     * @return
     * @throws ParserConfigurationException
     * @throws TransformerException
     * @throws UnsupportedEncodingException
     */
    public static String buildRequest(Map<String,String> argsMap, String currentBusinessCode) throws ParserConfigurationException, TransformerException, UnsupportedEncodingException {
        Document document = generate();
        Element rootElement = document.getRootElement();
        Element headElement = rootElement.element(HubeiCFCField.head);
        Element businessElement = headElement.addElement(HubeiCFCField.currentBusinessCode);
        businessElement.setText(currentBusinessCode);

        Element reqDataElement = rootElement.addElement(HubeiCFCField.reqdata);
        for(String key:argsMap.keySet()){
            Element newElement = reqDataElement.addElement(key);
            newElement.setText(argsMap.get(key));
        }
        document.setXMLEncoding("UTF-8");
        return document.asXML();
    }

    /**
     * 每天还款信息要查询的范围
     * @return
     */
    public static Date[] getDateScope(){
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MILLISECOND,0);
        calendar.set(Calendar.SECOND,0);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.HOUR_OF_DAY,0);
        calendar.set(Calendar.DATE,calendar.get(Calendar.DATE));
        Date gtDate = calendar.getTime();
        calendar.set(Calendar.HOUR_OF_DAY,21);
        Date ltDate = calendar.getTime();
        Date[] dates = new Date[]{gtDate,ltDate};
        return dates;
    }

    public static Date[] getOfflineDateScope(){
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MILLISECOND,0);
        calendar.set(Calendar.SECOND,0);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.HOUR_OF_DAY,20);
        calendar.set(Calendar.DATE,calendar.get(Calendar.DATE)-1);
        Date gtDate = calendar.getTime();
        calendar.set(Calendar.DATE,calendar.get(Calendar.DATE)+1);
        calendar.set(Calendar.HOUR_OF_DAY,20);
        Date ltDate = calendar.getTime();
        Date[] dates = new Date[]{gtDate,ltDate};
        return dates;
    }

    /**
     * 当天还款日零点
     * @return
     */
    public static Date getTodayDeadLine(){
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MILLISECOND,0);
        calendar.set(Calendar.SECOND,0);
        calendar.set(Calendar.MINUTE,0);
        calendar.set(Calendar.HOUR_OF_DAY,0);
        return calendar.getTime();
    }

    public static String getDocName(HubeiCFCDataType hubeiCFCDataType,Byte seqNo){
        Calendar calendar = Calendar.getInstance();
//        calendar.set(Calendar.DATE,calendar.get(Calendar.DATE)-1);
        //yyyyMMdd  DateUtil.sampleDateFormatFormat(calendar.getTime())
        //20170222
        String docName = new StringBuilder(DateUtil.sampleDateFormatFormat(calendar.getTime()))
                .append("_").append(HubeiCFCField.chanel_9009)
                .append("_").append(HubeiCFCField.fileAlias.get(hubeiCFCDataType)).append(StringFormat.format0Right(seqNo,2)).append(".txt").toString();
        return docName;
    }


    /**
     * 解析返回的数据
     * @param inputStream
     * @throws IOException
     * @throws SAXException
     * @throws ParserConfigurationException
     */
    public static HuBeiCFCResponse parseDiscResponse(InputStream inputStream) throws IOException, SAXException, ParserConfigurationException, DocumentException {
        SAXReader sr = new SAXReader();
        Document document = sr.read(inputStream);
        Element rootElement = document.getRootElement();
        Element ecElement = rootElement.element(HubeiCFCField.ec);
        String errorMsg = rootElement.element(HubeiCFCField.em).getText();
        String ec = ecElement.getTextTrim();
        HuBeiCFCResponse response = new HuBeiCFCResponse();
        response.setEc(ec);
        response.setEm(errorMsg);
        if(!StringUtils.isEmpty(ec)&&HubeiCFCField.EC_SUCCESS_CODE.equals(ec.trim())){
            Element respElement = rootElement.element(HubeiCFCField.Resp);
            if(respElement!=null){
                Element docElement = respElement.element(HubeiCFCField.docName);
                String fileName = docElement==null?null:docElement.getText();
                logger.info("file name={}",fileName);
                Element picUploadFileElement = respElement.element(HubeiCFCField.picUploadFile);
                String fileContent = picUploadFileElement==null?null:picUploadFileElement.getText();
                if(logger.isDebugEnabled()){
                    logger.info("file content={} ",fileContent);
                }
                response.setFileName(fileName);
                response.setFileConent(fileContent);
            }
        }else {
            logger.warn("result is not success, ec={}, em={}",ec,errorMsg);
        }
        return response;
    }


}
