package cn.quantgroup.financial.scheduler;

import cn.quantgroup.financial.constant.HubeiCFCDataType;
import cn.quantgroup.financial.constant.HubeiCFCField;
import cn.quantgroup.financial.constant.SysConstant;
import cn.quantgroup.financial.dao.IHuBeiCFCDao;
import cn.quantgroup.financial.handler.IHuBeiDispatcher;
import cn.quantgroup.financial.model.huibeicfc.HuBeiCFCResponse;
import cn.quantgroup.financial.model.huibeicfc.HuBeiDocName;
import cn.quantgroup.financial.model.huibeicfc.HuBeiHistory;
import cn.quantgroup.financial.model.huibeicfc.HuBeiJsonBean;
import cn.quantgroup.financial.service.IHuBeiService;
import cn.quantgroup.financial.service.sys.IScheduledJudgeService;
import cn.quantgroup.financial.service.thread.IApiScheduledExecutorService;
import cn.quantgroup.financial.util.HubeiCFCUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 湖北回盘文件每日定时任务
 * Created by WuKong on 2017/2/10.
 */
@Lazy(false)
@Component
public class HuBeiReturnDiscScheduler {

    private static final Logger logger = LoggerFactory.getLogger(HuBeiReturnDiscScheduler.class);


    @Autowired
    private IHuBeiService iHuBeiService;

    @Autowired
    private IHuBeiCFCDao iHuBeiCFCDao;

    @Autowired
    private IApiScheduledExecutorService scheduledExecutorService;

    @Autowired
    private IScheduledJudgeService scheduledJudgeService;

    @Autowired
    private IHuBeiDispatcher huBeiHandlerDispatcher;


    /**
     * Seconds Minutes Hours DayofMonth Month DayofWeek
     */
    @Scheduled(cron = "0 10 20 * * ?")
    public void dayScheduler(){
        if(scheduledJudgeService.isOpenScheduled()){
            try {
                logger.info("HuBeiDayScheduler start ");
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(new Date());
                calendar.set(Calendar.DATE,calendar.get(Calendar.DATE));
                HuBeiDocName sendDebitDoc = iHuBeiCFCDao.getLastestDocByDataAndType(HubeiCFCDataType.SEND_DEBIT,calendar.getTime());
                if(sendDebitDoc==null){
                    logger.warn("sendDebitDoc is null!!! , type={}",HubeiCFCDataType.SEND_DEBIT.get());
                    return;
                }
                List<HuBeiHistory> huBeiHistoryList = iHuBeiService.getReturnDisc(sendDebitDoc);
                //保存并推送回盘文件
                saveAndPushReturnDisc(huBeiHistoryList);
            } catch (Exception e) {
                iHuBeiService.sendErrorMailNotice(e.getMessage());
                logger.error(e.getMessage(),e);
            }
        }else {
            logger.info("scheduled is closed");
        }
    }

    public void saveAndPushReturnDisc(List<HuBeiHistory> huBeiHistoryList){

        HuBeiDocName debitDoc = iHuBeiService.generateHuBeiDoc(HubeiCFCDataType.RETURN_BATCH_DEBIT,new Integer(1).byteValue());
        Long docDebitId = debitDoc.getId();
        HuBeiDocName compensationDoc = iHuBeiService.generateHuBeiDoc(HubeiCFCDataType.RETURN_BATCH_COMPENSATION,new Integer(1).byteValue());
        Long docCompensationId = compensationDoc.getId();
        HuBeiDocName advanceDoc = iHuBeiService.generateHuBeiDoc(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,new Integer(1).byteValue());
        Long advanceDocId = advanceDoc.getId();
        Map<Byte,Long> advanceIdMap = new HashMap<Byte,Long>();
        Byte maxReqNo = new Integer(1).byteValue();
        advanceIdMap.put(maxReqNo,advanceDocId);


        for(HuBeiHistory history:huBeiHistoryList){
            if(HubeiCFCDataType.RETURN_BATCH_DEBIT.get().equals(history.getDataType())){
                history.setDocNameId(docDebitId);
            }else if(HubeiCFCDataType.RETURN_BATCH_COMPENSATION.get().equals(history.getDataType())){
                history.setDocNameId(docCompensationId);
            }else if(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT.get().equals(history.getDataType())){
                if(history.getSeqNo().compareTo(maxReqNo)>0){
                    maxReqNo = history.getSeqNo();
                }
                if(advanceIdMap.containsKey(history.getSeqNo())){
                    history.setDocNameId(advanceIdMap.get(history.getSeqNo()));
                }else {
                    advanceDoc = iHuBeiService.generateHuBeiDoc(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,history.getSeqNo());
                    advanceIdMap.put(history.getSeqNo(),advanceDoc.getId());
                    history.setDocNameId(advanceDoc.getId());
                }
            }
            iHuBeiCFCDao.saveHistory(history);
        }
        Byte currentReqNo = new Integer(1).byteValue();
        //推送回盘文件
        HuBeiCFCResponse debitResponse = iHuBeiService.handleDiscData(HubeiCFCDataType.RETURN_BATCH_DEBIT,null,docDebitId,currentReqNo);
        for(;currentReqNo<=maxReqNo;currentReqNo++){
            HuBeiCFCResponse advanceResponse = iHuBeiService.handleDiscData(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,null,advanceIdMap.get(currentReqNo),currentReqNo);
        }
        HuBeiCFCResponse compensationResponse = iHuBeiService.handleDiscData(HubeiCFCDataType.RETURN_BATCH_COMPENSATION,null,docCompensationId,currentReqNo);
//        if(maxReqNo!=null&&maxReqNo.compareTo(currentReqNo)>0){
//            logger.info("start scheduleAdvanceRepay currentReqNo={},maxReqNo={},docAdvanceRepayId={}",currentReqNo,maxReqNo,advanceDocId);
//            ScheduleAdvanceRepay scheduleAdvanceRepay = new ScheduleAdvanceRepay(currentReqNo,maxReqNo,advanceIdMap);
//            scheduleAdvanceRepay.registerService(scheduledExecutorService,iHuBeiService,iHuBeiCFCDao,huBeiHandlerDispatcher);
//            scheduledExecutorService.schedule(scheduleAdvanceRepay, HubeiCFCField.thirtyMintes, TimeUnit.MINUTES);
//        }
        if(debitResponse==null||compensationResponse==null
                ||!HubeiCFCField.EC_SUCCESS_CODE.equals(compensationResponse.getEc())
                ||!HubeiCFCField.EC_SUCCESS_CODE.equals(debitResponse.getEc())){
            iHuBeiService.sendErrorMailNotice("回盘文件发送失败，请悉知 联系相关人员");
        }
    }

}

class ScheduleAdvanceRepay implements Runnable{

    private Byte currentReqNo;
    private Byte maxReqNo;
    private Map<Byte,Long> advanceIdMap;
    private IApiScheduledExecutorService scheduledExecutorService;
    private IHuBeiDispatcher huBeiHandlerDispatcher;
    private IHuBeiService iHuBeiService;
    private IHuBeiCFCDao iHuBeiCFCDao;
    //启动时间
    private Long startTime;

    private static final Logger logger = LoggerFactory.getLogger(ScheduleAdvanceRepay.class);

    public ScheduleAdvanceRepay(Byte currentReqNo,Byte maxReqNo,Map<Byte,Long> advanceIdMap){
        this.currentReqNo = currentReqNo;
        this.maxReqNo = maxReqNo;
        this.advanceIdMap = advanceIdMap;
        this.startTime = System.currentTimeMillis();
    }
    public void registerService(IApiScheduledExecutorService scheduledExecutorService,IHuBeiService iHuBeiService,
                                IHuBeiCFCDao iHuBeiCFCDao,IHuBeiDispatcher huBeiHandlerDispatcher){
        this.scheduledExecutorService = scheduledExecutorService;
        this.iHuBeiService = iHuBeiService;
        this.iHuBeiCFCDao = iHuBeiCFCDao;
        this.huBeiHandlerDispatcher = huBeiHandlerDispatcher;
    }

    public boolean isOutTime(){
        Long spendtime = System.currentTimeMillis()-startTime;
        if(spendtime/1000>60*60*6){
            //运行超时 自动停止
            return true;
        }else {
            return false;
        }
    }

    @Override
    public void run() {
        try {
            logger.info("ScheduleAdvanceRepay method start");
            logger.info("currentReqNo={},maxReqNo={}",currentReqNo,maxReqNo);
            if(isOutTime()){
                logger.info("ScheduleAdvanceRepay has run over than 23 hours, auto stop");
                return;
            }
            //先前的reqNo
            String docName = HubeiCFCUtil.getDocName(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,currentReqNo);
            HuBeiCFCResponse response = iHuBeiService.handleDiscData(HubeiCFCDataType.SEND_ADVANCE_REPAYMENT_CHECK,docName,null,currentReqNo);
            //检查提前还款送盘文件 成功
            if(response!=null&&HubeiCFCField.EC_SUCCESS_CODE.equals(response.getEc())){
                checkAdvanceRepayResponse(response.getFileId());
                if(currentReqNo.compareTo(maxReqNo)<0){
                    //下一批次提前还款文件
                    currentReqNo++;

                    iHuBeiService.handleDiscData(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,null,advanceIdMap.get(currentReqNo),currentReqNo);

                    if(currentReqNo.compareTo(maxReqNo)<=0){
                        //30分钟后执行
                        ScheduleAdvanceRepay scheduleAdvanceRepay = new ScheduleAdvanceRepay(currentReqNo,maxReqNo,advanceIdMap);
                        scheduleAdvanceRepay.registerService(scheduledExecutorService,iHuBeiService,iHuBeiCFCDao,huBeiHandlerDispatcher);
                        scheduledExecutorService.schedule(scheduleAdvanceRepay, HubeiCFCField.thirtyMintes, TimeUnit.MINUTES);
                    }else {
                        //都推送了 只要检查最后一批检验文件
                        logger.info("today all advanceRepay info finished");
                    }

                }else {
                    //今天提前还款文件都推送完毕
                    logger.info("today all check info finished");
                }
            }else if(response!=null&&response.getEc().equals(HubeiCFCField.EC_FILENOTEXIST)) {
                //未获取到提前还款检验文件 后续再轮序 10分钟
                ScheduleAdvanceRepay scheduleAdvanceRepay = new ScheduleAdvanceRepay(currentReqNo,maxReqNo,advanceIdMap);
                scheduleAdvanceRepay.registerService(scheduledExecutorService,iHuBeiService,iHuBeiCFCDao,huBeiHandlerDispatcher);
                scheduledExecutorService.schedule(scheduleAdvanceRepay, HubeiCFCField.tenMinutes, TimeUnit.MINUTES);
            }else {
                iHuBeiService.sendErrorMailNotice("回盘提前还款文件失败，请悉知 联系相关人员");
            }
            logger.info("ScheduleAdvanceRepay method end");
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }


    public void checkAdvanceRepayResponse(Long docId){
        try {
            List<HuBeiHistory> checkHuBeiHistoryList = iHuBeiCFCDao.getListByDocNameId(docId);
            StringBuilder errorStringBuilder = new StringBuilder();
            if(!CollectionUtils.isEmpty(checkHuBeiHistoryList)){
                for(HuBeiHistory history: checkHuBeiHistoryList){
                    HuBeiJsonBean jsonBean = history.getData();
                    if(!HubeiCFCField.EC_SUCCESS_CODE.equals(jsonBean.getTradeCode())){
                        errorStringBuilder.append(huBeiHandlerDispatcher.builder(HubeiCFCDataType.RETURN_ADVANCE_REPAYMENT,history)).append("\r\n");
                    }
                }
            }
            String errorString = errorStringBuilder.toString();
            iHuBeiService.sendErrorMailNotice(errorString);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
    }
}
