//
//  QGCalendarManager.m
//  CreditWallet
//
//  Created by 张博 on 16/10/19.
//  Copyright © 2016年 QuantGroup. All rights reserved.
//

#import "QGCalendarManager.h"

@interface QGCalendarManager ()

/**
 *  阳历日历信息
 */
@property (nonatomic, strong) NSCalendar * calendar;

/**
 *  阳历日历配置信息
 */
@property (nonatomic, assign) NSCalendarUnit calendarUnit;

/**
 *  阴历日历信息（农历）
 */
@property (nonatomic, strong) NSCalendar * chineseCalendar;
/**
 *  阴历日历配置信息（农历）
 */
@property (nonatomic, assign) NSCalendarUnit chineseUnit;

/**
 *  转换各种日期格式
 */
@property (nonatomic, strong) NSDateFormatter * formatter;

/**
 *  当前展示的月份信息
 */
@property (nonatomic, strong) NSDate * todayDate;

/**
 *  截止时间
 */
@property (nonatomic, strong) NSDate * endDate;

/**
 *  抢票截止时间
 */
@property (nonatomic, strong) NSDate * robTicketEndDate;
@end


@implementation QGCalendarManager


+ (instancetype)shareInstance
{
    static dispatch_once_t onceToken;
    static QGCalendarManager * shareManager = nil;
    dispatch_once(&onceToken, ^{
        shareManager = [[self alloc] init];
    });
    return shareManager;
}



- (id)init
{
    if (self = [super init]) {
        [self initNormal];
    }
    return self;
}

- (void)initNormal
{
    //初始化基本数据信息
    self.calendar = [[NSCalendar alloc]initWithCalendarIdentifier:NSCalendarIdentifierGregorian];
    self.calendarUnit =  NSCalendarUnitYear | NSCalendarUnitMonth | NSCalendarUnitDay | NSCalendarUnitWeekday;
    
    self.chineseCalendar = [[NSCalendar alloc]initWithCalendarIdentifier:NSCalendarIdentifierChinese];
    self.chineseUnit = NSCalendarUnitYear | NSCalendarUnitMonth |  NSCalendarUnitDay;
    
    //获取当前日期
    NSDateComponents *components = [self.calendar components:self.calendarUnit fromDate:[NSDate date]];
    self.todayDate = [self.calendar dateFromComponents:components];

}


- (void)setTimeLength:(NSInteger)timeLength
{
    if (timeLength <= 0) {
        return;
    }
    _timeLength = timeLength;
    //    在当前历法下，获取N天后的时间点
    //    定义一个NSDateComponents对象，设置一个时间段
    NSDateComponents *dateComponentsAsTimeQantum = [[NSDateComponents alloc] init];
    [dateComponentsAsTimeQantum setDay:timeLength];
    NSDate *dateFromDateComponentsAsTimeQantum = [self.calendar dateByAddingComponents:dateComponentsAsTimeQantum toDate:[NSDate date] options:0];
    self.endDate = dateFromDateComponentsAsTimeQantum;
}


- (void)setRobTicketLength:(NSInteger)robTicketLength
{
    if (robTicketLength <= 0) {
        return;
    }
    _robTicketLength = robTicketLength;
    //    在当前历法下，获取N天后的时间点
    //    定义一个NSDateComponents对象，设置一个时间段
    NSDateComponents *dateComponentsAsTimeQantum = [[NSDateComponents alloc] init];
    [dateComponentsAsTimeQantum setDay:robTicketLength];
    NSDate *dateFromDateComponentsAsTimeQantum = [self.calendar dateByAddingComponents:dateComponentsAsTimeQantum toDate:[NSDate date] options:0];
    self.robTicketEndDate = dateFromDateComponentsAsTimeQantum;
    
}


/**
 *  获取今天的日期
 *
 *  @return  今天的日期
 */
- (NSDate *)getToday
{
    return self.todayDate;
}

/**
 *  获取截止日期
 *
 *  @return 截止日期
 */
- (NSDate *)getEndDate
{
    return self.endDate;
}

/**
 *  抢票截止日期
 *
 *  @return 截止日期
 */
- (NSDate *)getRobTicketEndDate
{
    return self.robTicketEndDate;
}

/**
 *  比较两个时间是否是同一个月
 *
 *  @param theDate
 *  @param anotherDate
 *
 *  @return 比较结果
 */
- (BOOL)date:(NSDate *)theDate isEqualMonthAnother:(NSDate *)anotherDate
{
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:@"yyyyMM"];
    NSString * theStr = [self.formatter stringFromDate:theDate];
    NSString * anotherStr = [self.formatter stringFromDate:anotherDate];
    
    if ([theStr isEqualToString:anotherStr]) {
        return YES;
    }
    return NO;
}



/**
 *  获取传入日期的下个月的时间  传入2016-10-11 得到 2016-11-11
 *
 *  @param date 参考日期
 *
 *  @return 参考日期的下个月日期
 */
- (NSDate *)nextMonthDateWithDate:(NSDate *)date
{
    NSDateComponents *components = [self.calendar components:self.calendarUnit fromDate:date];
    components.day=1;
    components.month++;
    NSDate *nextMonthDate = [self.calendar dateFromComponents:components];
//    NSLog(@"______%@",nextMonthDate);
    return nextMonthDate;
}

/**
 *  获取格式为 20161111的string时间
 */
- (NSString *)getStringFromDate:(NSDate *)date
{
  return [self getStringFromDate:date formatter:@"yyyyMMdd"];
}


/**
 *  根据字符串 和 formatter格式获取NSDate
 *
 *  @param dateStr   时间字符串
 *  @param formatter 时间格式
 *
 *  @return date
 */
- (NSDate *)getDateFromString:(NSString *)dateStr formatter:(NSString *)formatter
{
    if (dateStr.length == 0 || formatter.length == 0) {
        return nil;
    }
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:formatter];
    
    NSDate * date = [self.formatter dateFromString:dateStr];
    return date;
}


/**
 *  根据时间戳 和 formatter日期格式获取字符串
 *
 *  @param dateStr   时间戳
 *  @param formatter 日期格式
 *
 *  @return 字符串
 */
- (NSString *)getDateStrFromSince1970:(NSString *)dateStr formatter:(NSString *)formatter
{
    if (dateStr.length == 0 || formatter.length == 0) {
        return nil;
    }
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:formatter];
    NSDate* date = [NSDate dateWithTimeIntervalSince1970:[dateStr doubleValue]/ 1000.0];
    NSString *destDateString = [_formatter stringFromDate:date];
    return destDateString;
}


/**
 *  根据日期 和 formatter日期格式获取字符串
 *
 *  @param date      日期
 *  @param formatter 日期格式
 *
 *  @return 字符串
 */
- (NSString *)getStringFromDate:(NSDate *)date formatter:(NSString *)formatter
{
    if (!date || formatter.length == 0) {
        return nil;
    }
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:formatter];
    NSString *destDateString = [_formatter stringFromDate:date];
    return destDateString;
}


/**
 *  获取从fromTime 一段时间后的日期
 *
 *  @return 日期
 */
- (NSDate *)getSomeTimeLaterFromTime:(NSDate *)fromTime timeInterval:(NSTimeInterval)secsToBeAdded;
{
    if (!fromTime) {
        return nil;
    }
    NSDate * anHourLaterDate = [NSDate dateWithTimeInterval:secsToBeAdded sinceDate:fromTime];
    return anHourLaterDate;
}


/**
 *  获取 dateStr 的 someTime 前的时间
 *
 *  @param dateStr   指定时间字符串
 *  @param formatter 时间格式
 *
 *  @return 返回结果时间
 */
- (NSDate *)getSomeTimeAgoFromDateStr:(NSString *)dateStr timeInterval:(NSTimeInterval)someTime formatter:(NSString *)formatter;
{
    if (dateStr.length == 0 || formatter.length == 0) {
        return nil;
    }
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:formatter];
    
    NSDate * date = [self.formatter dateFromString:dateStr];
    date = [self getSomeTimeLaterFromTime:date timeInterval:someTime];
    return date;
}

/**
 *  获取开始时间到结束时间的日期数组 (火车票抢票时间选择使用)
 *  日期格式 ：MM：dd
 *  时间格式 ：HH：00
 *  @param fromDate 开始时间
 *  @param toDate   结束时间
 *
 *  @return 结果数组
 */
- (NSArray *)getDateArrayFromDate:(NSDate *)fromDate toDate:(NSDate *)toDate
{
    NSMutableArray * dateArray = [NSMutableArray arrayWithCapacity:0];

    NSDateComponents *fromDateComponent = [self.calendar components:NSCalendarUnitHour fromDate:fromDate];
    NSInteger fromHour = [fromDateComponent hour];
    
    NSDateComponents *toDateComponent = [self.calendar components:NSCalendarUnitHour fromDate:toDate];
    NSInteger toHour = [toDateComponent hour];

    /**
     *  获取月份信息 计算相差几个月
     */
    NSString * fromMonthStr = [self getStringFromDate:fromDate formatter:@"yyyyMM"];
    NSString * toMonthStr = [self getStringFromDate:toDate formatter:@"yyyyMM"];
    //月份的数量
    NSInteger monthCount = toMonthStr.integerValue - fromMonthStr.integerValue;
    
    /**
     *  获取日信息 用于取值范围
     */
    NSString * fromDayStr = [self getStringFromDate:fromDate formatter:@"yyyyMMdd"];
    NSString * toDayStr = [self getStringFromDate:toDate formatter:@"yyyyMMdd"];
    
    //标记日期 用于拿取当月日期
    NSDate * tagDate = fromDate;
    for (int i = 0; i <= monthCount; i++) {
        NSDateComponents *components = [self.calendar components:self.calendarUnit fromDate:tagDate];
        components.day = 1;
        //计算总天数
        NSRange days = [self.calendar rangeOfUnit:NSCalendarUnitDay inUnit:NSCalendarUnitMonth forDate:tagDate];
        
        for (int j = 0; j < days.length; j ++) {
            components.day = j + 1;
            NSDate * date = [self.calendar dateFromComponents:components];
            NSString * dateStr = [self getStringFromDate:date formatter:@"yyyyMMdd"];
            
            //当前的日期 大于等于开始时间  &&  小于等于结束时间
            if (dateStr.integerValue >= fromDayStr.integerValue && dateStr.integerValue <= toDayStr.integerValue) {
                
                NSMutableDictionary * dayDic = [NSMutableDictionary dictionaryWithCapacity:0];
                [dayDic setObject:date forKey:@"Date"];
                
                NSMutableArray * hoursArray = [NSMutableArray arrayWithArray:@[@"00:00",@"01:00",@"02:00",@"03:00",@"04:00",@"05:00",@"06:00",@"07:00",@"08:00",@"09:00",@"10:00",@"11:00",@"12:00",@"13:00",@"14:00",@"15:00",@"16:00",@"17:00",@"18:00",@"19:00",@"20:00",@"21:00",@"22:00",@"23:00",]];
                
                
                //今天的日期，去掉已经过得小时
                if (dateStr.integerValue == fromDayStr.integerValue) {
                    [hoursArray removeObjectsInRange:NSMakeRange(0, fromHour)];
                }
                
                //去掉截止日期之后的小时
                if (dateStr.integerValue == toDayStr.integerValue) {
                    
                    NSInteger endHour = toHour;
                    //如果开始的日期和结束的日期是同一天,则需要考虑之前去除的时
                    if (fromDayStr.integerValue == toDayStr.integerValue) {
                        endHour = endHour - fromHour;
                    }
                    
                    hoursArray = [NSMutableArray arrayWithArray:[hoursArray subarrayWithRange:NSMakeRange(0, endHour + 1)]];
                    
                }
                
                [dayDic setObject:hoursArray forKey:@"hours"];
                //如果小时的数据还有，则会被添加到最终数据源中
                if (hoursArray.count != 0) {
                    [dateArray addObject:dayDic];
                }
                
                
            }
            
        }
        
        //如果到了结束日期月份 就结束了
        if ([self date:tagDate isEqualMonthAnother:toDate]) {
            break;
        }
        
        //获取下个月日期
        tagDate = [self nextMonthDateWithDate:tagDate];
    }
    
    
    
    
    return dateArray;
}



/// 获取从开始日期到结束日期之间的年和月
/// 格式如下：
/// 2030 = (01,02,03,04,05,06);
/// 取值范围 大于等于开始日期，小于结束日期（月）
/// @param fromDate 开始日期
/// @param toDate 结束日期
- (NSDictionary *)getYearsAndMonthsFromDate:(NSDate *)fromDate toDate:(NSDate *)toDate
{
    //每年对应的月数组
    NSMutableDictionary * monthDic = [NSMutableDictionary dictionaryWithCapacity:0];
    //获取月份信息 计算相差几个月
    NSDateComponents * comp = [self.calendar components:NSCalendarUnitMonth fromDate:fromDate toDate:toDate options:NSCalendarWrapComponents];
    //月份的数量
    NSInteger monthCount = comp.month;
    //标记日期 用于拿取当月日期
    NSDate * tagDate = fromDate;
    //开始日期和结束日期，用于比对
    NSString * formDateStr = [self getStringFromDate:fromDate formatter:@"yyyyMM"];
    NSString * toDateStr = [self getStringFromDate:toDate formatter:@"yyyyMM"];
    //标记日期 用于拿取当月日期
    for (int i = 0; i < monthCount; i++) {
        //转换成年月
        NSString * tagDateStr = [self getStringFromDate:tagDate formatter:@"yyyyMM"];
        //异常处理（一般不会出现）
        if (tagDateStr.length != 6) {
            continue;
        }
        
        //安全判断，只取中间值
        if (tagDateStr.integerValue >= formDateStr.integerValue && tagDateStr.integerValue < toDateStr.integerValue) {
            NSString * yearStr = [tagDateStr substringWithRange:NSMakeRange(0, 4)];
            NSString * monthStr = [tagDateStr substringWithRange:NSMakeRange(4, 2)];
            
            //获取月数组
            NSMutableArray * monthArray = monthDic[yearStr];
            if (!monthArray) {
                monthArray = [NSMutableArray arrayWithCapacity:0];
                [monthDic setObject:monthArray forKey:yearStr];
            }
            [monthArray addObject:monthStr];
            
            //获取下个月日期
            tagDate = [self nextMonthDateWithDate:tagDate];
        }
        //出现不在目标范围内的日期，直接结束
        else
        {
            break;
        }
    }
    return monthDic;
}



/// 获取折线图所传日期当月的数据
/// @param date 传输的月份信息
/// @param today 今天的日期
- (QGBrokenLineMonthModel *)getBrokenLineMonthDataSourceWithDate:(NSDate *)date today:(NSDate *)today
{
    QGBrokenLineMonthModel * monthModel = [[QGBrokenLineMonthModel alloc] init];
    monthModel.dayArray = [NSMutableArray arrayWithCapacity:0];
    
    //计算总天数
    NSRange days = [self.calendar rangeOfUnit:NSCalendarUnitDay inUnit:NSCalendarUnitMonth forDate:date];
    monthModel.daysNum = days.length;
    
    //获取年月
    NSDateComponents *components = [self.calendar components:self.calendarUnit fromDate:date];
    NSString * year = [NSString stringWithFormat:@"%ld",(long)components.year];
    NSString * month = [NSString stringWithFormat:@"%ld",(long)components.month];
    monthModel.year = year;
    monthModel.month = month;
    
    //获取日信息
    for (NSInteger i = 0; i < days.length; i++) {
        components.day = i + 1;
        NSDate * date = [self.calendar dateFromComponents:components];
        QGBrokenLineDayModel * dayModel = [[QGBrokenLineDayModel alloc] init];
        dayModel.date = date;
        dayModel.year = year;
        dayModel.month = month;
        dayModel.day = [NSString stringWithFormat:@"%ld",(long)components.day];
        NSDateComponents* todayComp = [self.calendar components:self.calendarUnit fromDate:today];
        //今天的日期
        BOOL isSameDay = [components day] == [todayComp day] && [components month] == [todayComp month] && [components year]  == [todayComp year];
        if (isSameDay) {
            //今天的日期
            dayModel.isToday = YES;
        }
        [monthModel.dayArray addObject:dayModel];
    }
    return monthModel;
}




/**
 *  获取火车票所传日期的当月数据
 *
 *  @param date 传参
 */
- (QGCalendarMonthModel *)getTrainMonthDataSourceWithData:(NSDate *)date
{
    QGCalendarMonthModel * monthModel = [[QGCalendarMonthModel alloc]init];
    monthModel.dayArray = [NSMutableArray arrayWithCapacity:0];
    
    //计算当月第一天是周几
    NSDateComponents *components = [self.calendar components:self.calendarUnit fromDate:date];
    components.day = 1;
    NSDate *firstDayOfMonth = [self.calendar dateFromComponents:components];
    NSDateComponents *comps = [self.calendar components:NSCalendarUnitWeekday fromDate:firstDayOfMonth];
    NSInteger weekdayBegining = [comps weekday];
    weekdayBegining -=1;
    
    monthModel.beginIndex = weekdayBegining;
    //计算总天数
    NSRange days = [self.calendar rangeOfUnit:NSCalendarUnitDay inUnit:NSCalendarUnitMonth forDate:date];
    monthModel.daysNum = days.length;
    
    //减去第一周的天数，剩余天数除以7，得到倍数和余数, 获得一个月有几周
    monthModel.weeksNum = 1;
    NSInteger otherDays = (days.length - (7 - weekdayBegining));
    monthModel.weeksNum = monthModel.weeksNum + otherDays / 7;
    if (otherDays % 7 > 0) {
        monthModel.weeksNum = monthModel.weeksNum + 1;
    }
    
    NSString * year = [NSString stringWithFormat:@"%ld",(long)components.year];
    NSString * month = [NSString stringWithFormat:@"%ld",(long)components.month];
    monthModel.year = year;
    monthModel.month = month;
    
    for (int i = 0; i < days.length; i++) {
        components.day = i + 1;
        NSDate * date = [self.calendar dateFromComponents:components];
        QGCalendarDayModel * dayModel = [[QGCalendarDayModel alloc]init];
        dayModel.date = date;
        dayModel.year = year;
        dayModel.month = month;
        dayModel.day = [NSString stringWithFormat:@"%ld",(long)components.day];
        dayModel.lunarDay = [self getChineseCalendarWithDate:components.day month:month.integerValue year:year.integerValue];
        
        NSTimeInterval timeInterval = [date timeIntervalSinceDate:self.todayDate];
        //小于今天的日期
        if (timeInterval < 0) {
            dayModel.dayStyle = CalendarDayStyleUnusable;
        }
        else
        {
            //今天的日期
            if (timeInterval < 24 * 60 * 60) {
                dayModel.dayStyle = CalendarDayStyleToday;
                dayModel.lunarDay = @"今天";
            }
            
            NSTimeInterval endTimeInterval = 0;
            //抢票的截止日期  抢票逻辑
            if (self.robTicketEndDate) {
                endTimeInterval = [date timeIntervalSinceDate:self.robTicketEndDate];
                
                //如果当前日期大于截止日期 并且小于抢票截止日期，则为预售阶段
                NSTimeInterval reservationTime = [date timeIntervalSinceDate:self.endDate];
                if (reservationTime > 0 && endTimeInterval < 0) {
                    dayModel.dayStyle = CalendarDayStyleReservation;
                }
                
                
            }
            //买票的截止日期
            else
            {
                endTimeInterval = [date timeIntervalSinceDate:self.endDate];
            }
            
            
            
            //超过预售期 || 超过截止时间
            if (endTimeInterval > 0) {
                dayModel.dayStyle = CalendarDayStyleUnusable;
            }
        }
        
        
        
        [monthModel.dayArray addObject:dayModel];
    }
    
    
    
    
    
    return monthModel;
}



-(NSString*)getChineseCalendarWithDate:(NSInteger)day month:(NSInteger)month year:(NSInteger)year{
    NSArray *chineseDays=[NSArray arrayWithObjects:
                          @"初一", @"初二", @"初三", @"初四", @"初五", @"初六", @"初七", @"初八", @"初九", @"初十",
                          @"十一", @"十二", @"十三", @"十四", @"十五", @"十六", @"十七", @"十八", @"十九", @"二十",
                          @"廿一", @"廿二", @"廿三", @"廿四", @"廿五", @"廿六", @"廿七", @"廿八", @"廿九", @"三十",  nil];
    NSString* string;
    if(month<10)
    {
        if (day < 10) {
            string = [NSString stringWithFormat:@"%ld0%ld0%ld23",(long)year,month,day];
        }
        else{
            string = [NSString stringWithFormat:@"%ld0%ld%ld23",(long)year,month,day];
        }
    }
    else
    {
        if (day < 10) {
            string = [NSString stringWithFormat:@"%ld%ld0%ld23",(long)year,month,day];
        }
        else{
            string = [NSString stringWithFormat:@"%ld%ld%ld23",(long)year,month,day];
        }
    }
    
    if (!self.formatter) {
        self.formatter = [[NSDateFormatter alloc]init];
    }
    [self.formatter setDateFormat:@"yyyyMMddHH"];
    NSDate* inputDate = [self.formatter dateFromString:string];
    NSDateComponents *localeComp = [self.chineseCalendar components:self.chineseUnit fromDate:inputDate];
    NSString *d_str = [chineseDays objectAtIndex:localeComp.day-1];
    return d_str;
}




- (NSString *)getDateStringWithTimestamp:(NSInteger)timestamp andFormatter:(NSString *)format{
    
    NSDateFormatter *formatter = [[NSDateFormatter alloc] init];
    
    [formatter setDateStyle:NSDateFormatterMediumStyle];
    
    [formatter setTimeStyle:NSDateFormatterShortStyle];
    
    [formatter setDateFormat:format];
    
    NSTimeZone *timeZone = [NSTimeZone timeZoneWithName:@"Asia/Beijing"];
    
    [formatter setTimeZone:timeZone];
    
    NSDate *confromTimesp = [NSDate dateWithTimeIntervalSince1970:(timestamp/1000)];
    
    NSString *confromTimespStr = [formatter stringFromDate:confromTimesp];
    
    return confromTimespStr;
}

- (NSString *)getDateStingFromTimestamp:(NSString *)timeStampString withFormatter:(NSString *)format{
    NSTimeInterval _interval=[timeStampString doubleValue] / 1000.0;
    NSDate *date = [NSDate dateWithTimeIntervalSince1970:_interval];
    NSDateFormatter *objDateformat = [[NSDateFormatter alloc] init];
    [objDateformat setDateFormat:format];
    return [objDateformat stringFromDate: date];
}


- (NSString *)getDateStingFromOriginTimestamp:(NSString *)timeStampString withFormatter:(NSString *)format{
    NSTimeInterval _interval=[timeStampString doubleValue];
    NSDate *date = [NSDate dateWithTimeIntervalSince1970:_interval];
    NSDateFormatter *objDateformat = [[NSDateFormatter alloc] init];
    [objDateformat setDateFormat:format];
    return [objDateformat stringFromDate: date];
}

- (NSDate *)getDateWithTimestamp:(NSInteger)timestamp andFormatter:(NSString *)format {
    NSDateFormatter *formatter = [[NSDateFormatter alloc] init];
    
    [formatter setDateStyle:NSDateFormatterMediumStyle];
    
    [formatter setTimeStyle:NSDateFormatterShortStyle];
    
    [formatter setDateFormat:format];
    
    NSTimeZone *timeZone = [NSTimeZone timeZoneWithName:@"Asia/Beijing"];
    
    [formatter setTimeZone:timeZone];
    
    NSDate *confromTimesp = [NSDate dateWithTimeIntervalSince1970:(timestamp/1000)];
    
    return confromTimesp;
}


- (NSString *)formatDateWithDateString:(NSString *)dateStr {
    NSDateFormatter *dateFormatter2 = [[NSDateFormatter alloc] init];
    [dateFormatter2 setDateFormat:@"yyyy-MM-dd"];
    NSDateFormatter *dateFormatter1 = [[NSDateFormatter alloc] init];
    //设定时间格式,这里可以设置成自己需要的格式
    [dateFormatter1 setDateFormat:@"M月d日"];
    //用[NSDate date]可以获取系统当前时间
    NSString *currentDateStr = [dateFormatter1 stringFromDate:[dateFormatter2 dateFromString:dateStr]];
    return currentDateStr;
}

- (NSString *)formatWeek:(NSNumber *)week {
    if (week) {
        switch (week.intValue) {
            case 1:
                return @"周一";
                break;
            case 2:
                return @"周二";
                break;
            case 3:
                return @"周三";
                break;
            case 4:
                return @"周四";
                break;
            case 5:
                return @"周五";
                break;
            case 6:
                return @"周六";
                break;
            case 7:
                return @"周日";
                break;
            default:
                return @"";
                break;
        }
    }
    return @"";
}

- (NSString *)getDateStringWithSecondTimestamp:(NSInteger)timestamp andFormatter:(NSString *)format {
    
    NSDateFormatter *formatter = [[NSDateFormatter alloc] init];
    
    [formatter setDateStyle:NSDateFormatterMediumStyle];
    
    [formatter setTimeStyle:NSDateFormatterShortStyle];
    
    [formatter setDateFormat:format];
    
    NSTimeZone *timeZone = [NSTimeZone timeZoneWithName:@"Asia/Beijing"];
    
    [formatter setTimeZone:timeZone];
    
    NSDate *confromTimesp = [NSDate dateWithTimeIntervalSince1970:(timestamp)];
    
    NSString *confromTimespStr = [formatter stringFromDate:confromTimesp];
    
    return confromTimespStr;
}

@end
