//
//  QGCustomPageControl.m
//  VirtualPayment
//
//  Created by silicn on 2019/11/22.
//  Copyright © 2019 bo.zhang. All rights reserved.
//

#import "QGCustomPageControl.h"
 
#import "QGBasicUITool.h"

@interface QGCustomPageControl ()

///内部dot承载视图
@property (nonatomic, strong)UIView *innerView;

/// 上个选中的page
@property (nonatomic, strong)UIImageView *lastPageView;

/// currentPage设置锁
@property (nonatomic, strong)NSLock *lock;



@end

@implementation QGCustomPageControl

- (id)init{
    self = [super init];
    if (self) {
        // 默认设置
        self.numberOfPages = 0;
        self.currentPage =  0;
        self.currentPageIndicatorTintColor = [UIColor whiteColor];
        self.pageIndicatorTintColor = [UIColor lightGrayColor];
        self.indictorSize = CGSizeMake(8, 8);
        self.indictorSpace = 8;
    }
    return self;
}
- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        self.innerView = [[UIView alloc]init];
        [self addSubview:_innerView];
        [self.innerView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.centerX.equalTo(self);
            make.centerY.equalTo(self);
        }];
    }
    return self;
}


/// 设置圆点个数
/// @param numberOfPages NSInteger
- (void)setNumberOfPages:(NSInteger)numberOfPages
{
    if (_numberOfPages != numberOfPages) {
        _numberOfPages = numberOfPages;
        [self initSubViewsWithNum:numberOfPages];
    }
}

/// 初始化圆点
/// @param pages 个数
- (void)initSubViewsWithNum:(NSInteger)pages
{
    /// 移除之前的所有圆点
    [self.innerView.subviews enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(__kindof UIView * _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if ([obj isMemberOfClass:[UIImageView class]]) {
            [obj removeFromSuperview];
        }
    }];
    /// 初始化圆点
    if (pages <= 0 ) {
        return ;
    }else if (pages == 1 && self.hidesForSinglePage){
        self.hidden = YES;
        [self makeDotWithPages:pages];
    }else{
        self.hidden = NO;
        [self makeDotWithPages:pages];
    }
}

/// 设置单个是否隐藏
- (void)setHidesForSinglePage:(BOOL)hidesForSinglePage
{
    if (_hidesForSinglePage != hidesForSinglePage) {
        _hidesForSinglePage = hidesForSinglePage;
        if (self.numberOfPages == 1 && hidesForSinglePage) {
            self.hidden = hidesForSinglePage;
        }else{
            self.hidden = NO;
        }
    }
}


/// 设置圆点横向间距
/// @param indictorSpace 间距大小
- (void)setIndictorSpace:(CGFloat)indictorSpace
{
    if (_indictorSpace != indictorSpace) {
        _indictorSpace = indictorSpace;
        [self reloadDotSpace:indictorSpace];
    }
}

/// 重新根据间距重新布局
/// @param space 间距大小
- (void)reloadDotSpace:(CGFloat)space
{
    if (space < 0.0f) return;
   __block UIView *left = nil;
    [self.innerView.subviews enumerateObjectsUsingBlock:^(__kindof UIView * _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if (left == nil) {
            left = obj;
        }else{
            [obj mas_updateConstraints:^(MASConstraintMaker *make) {
                make.left.equalTo(left.mas_right).offset(space);
            }];
            left = obj;
        }
    }];
}


/// 设置圆点大小
/// @param indictorSize 圆点大小
- (void)setIndictorSize:(CGSize)indictorSize
{
    if (!CGSizeEqualToSize(_indictorSize, indictorSize)) {
        _indictorSize = indictorSize;
        [self reloadDotSize:indictorSize];
    }
}


/// 重新布局圆点大小
/// @param indictorSize 圆点大小
- (void)reloadDotSize:(CGSize)indictorSize
{
    if (indictorSize.width < 0.0f) return;
    
    [self.innerView.subviews enumerateObjectsUsingBlock:^(__kindof UIView * _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        [obj mas_updateConstraints:^(MASConstraintMaker *make) {
            make.size.equalTo(@(self.indictorSize));
        }];
        obj.layer.cornerRadius = self.indictorSize.width/2;
        obj.clipsToBounds = YES;
    }];
}


/// 设置当前页面
/// @param currentPage 当前页面
- (void)setCurrentPage:(NSInteger)currentPage
{
    [self.lock lock];
    if (_currentPage != currentPage) {
        _currentPage = currentPage;
        [self reloadCurrentPage:currentPage];
    }
    [self.lock unlock];
}


/// 切换当前Page
/// @param page 当前page
- (void)reloadCurrentPage:(NSInteger)page
{
    UIImageView *current = [self.innerView viewWithTag:1000 + page];
    if (current) {
        /// 移除上次选中的状态
        self.lastPageView.backgroundColor = self.pageIndicatorTintColor;
        if (self.pageImageName) {
            self.lastPageView.image = [UIImage imageNamed:self.pageImageName];
        }
        /// 设置本次的选中状态
        current.backgroundColor = self.currentPageIndicatorTintColor;
        if (self.currentPageImageName) {
            current.image = [UIImage imageNamed:self.currentPageImageName];
        }
        /// 重新赋值lastPageView
        self.lastPageView = current;
    }
}

- (void)setPageImageName:(NSString *)pageImageName
{
    if (_pageImageName != pageImageName) {
        _pageImageName = pageImageName;
        if (pageImageName) {
            [self reloadDotPageImage];
        }
    }
}


/// 刷新Dot图片,如果有图片
- (void)reloadDotPageImage
{
    [self.innerView.subviews enumerateObjectsUsingBlock:^(__kindof UIView * _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if ([obj isMemberOfClass:[UIImageView class]]) {
            UIImageView *iv = (UIImageView *)obj;
            if (idx == self.currentPage) {
                iv.image = [UIImage imageNamed:_currentPageImageName];
            }else{
                iv.image = [UIImage imageNamed:_pageImageName];
            }
        }
    }];
}

/// 创建圆点
- (void)makeDotWithPages:(NSInteger)pages
{
    UIImageView *lastIV = nil;
    for (int i = 0; i < pages; i++) {
        UIImageView *iv = [[UIImageView alloc]init];
        // 是否是当前选中的
        if (i == self.currentPage) {
            // 选中设置
            iv.backgroundColor = self.currentPageIndicatorTintColor;
            if (self.currentPageImageName) {
                iv.image = [UIImage imageNamed:self.currentPageImageName];
            }
            self.lastPageView = iv;
        }else{
            // 未选中设置
            iv.backgroundColor = self.pageIndicatorTintColor;
            if (self.pageImageName) {
                iv.image = [UIImage imageNamed:self.pageImageName];
            }
        }
        iv.tag = 1000 + i;
        iv.layer.cornerRadius = self.indictorSize.width/2;
        iv.clipsToBounds = YES;
        [self.innerView addSubview:iv];
        
        /// 横向并排布局一行
        [iv mas_makeConstraints:^(MASConstraintMaker *make) {
            make.top.bottom.inset(8);
            make.size.equalTo(@(self.indictorSize));
            if (lastIV) {
                make.left.equalTo(lastIV.mas_right).offset(self.indictorSpace);
            }else{
                make.left.inset(self.indictorSpace);
            }
        }];
        lastIV = iv;
    }
    
    [lastIV mas_updateConstraints:^(MASConstraintMaker *make) {
        make.right.inset(self.indictorSpace);
    }];
}

- (NSLock *)lock
{
    if (_lock == nil) {
        _lock = [[NSLock alloc]init];
    }
    return _lock;
}

/*
// Only override drawRect: if you perform custom drawing.
// An empty implementation adversely affects performance during animation.
- (void)drawRect:(CGRect)rect {
    // Drawing code
}
*/

@end
