//
//  QGDashPatternView.m
//  VirtualPayment
//
//  Created by 张博 on 2019/8/7.
//  Copyright © 2019 bo.zhang. All rights reserved.
//

#import "QGDashPatternView.h"

@interface QGDashPatternView ()

/**
 虚线图案样式
 */
@property (nonatomic, assign) QGDashPatternViewType dashPatternType;

/**
 虚线的颜色
 */
@property (nonatomic, strong) UIColor * lineColor;

/**
 线的宽度
 */
@property (nonatomic, assign) CGFloat lineWidth;

/**
 @[@a,@b] a:虚线长度，b:两虚线间隔长度
 */
@property (nonatomic, strong) NSArray * lineDashPattern;

/**
 圆角的弧度
 */
@property (nonatomic, assign) CGFloat cornerRadius;

/**
 虚线layer
 */
@property (nonatomic, strong) CAShapeLayer * dashLayer;

@end



@implementation QGDashPatternView



/**
 初始化虚线实例

 @param dashPatternType 虚线的类型 上、下、左、右、四周
 @param lineColor 虚线的颜色
 @param lineWidth 虚线的宽度
 @param lineDashPattern lineDashPatter @[@a,@b] a:虚线长度，b:两虚线间隔长度
 @return 实例对象
 */
- (instancetype)initWithDashPatternType:(QGDashPatternViewType)dashPatternType lineColor:(UIColor *)lineColor lineWidth:(CGFloat)lineWidth lineDashPattern:(NSArray *)lineDashPattern
{
    if (self = [super init]) {
        self.dashPatternType = dashPatternType;
        //不设置会强烈提示
        self.lineColor = lineColor ? lineColor : [UIColor redColor];
        self.lineWidth = lineWidth;
        self.lineDashPattern = lineDashPattern;
    }
    return self;
}


/**
 初始化虚线实例 - 带圆角
 
 @param dashPatternType 虚线的类型 上、下、左、右、四周
 @param cornerRadius 圆角的弧度
 @param lineColor 虚线的颜色
 @param lineWidth 虚线的宽度
 @param lineDashPattern lineDashPatter @[@a,@b] a:虚线长度，b:两虚线间隔长度
 @return 实例对象
 */
- (instancetype)initWithDashPatternType:(QGDashPatternViewType)dashPatternType cornerRadius:(CGFloat)cornerRadius lineColor:(UIColor *)lineColor lineWidth:(CGFloat)lineWidth lineDashPattern:(NSArray *)lineDashPattern
{
    if (self = [super init]) {
        self.dashPatternType = dashPatternType;
        self.lineColor = lineColor ? lineColor : [UIColor redColor];
        self.lineWidth = lineWidth;
        self.lineDashPattern = lineDashPattern;
        self.cornerRadius = cornerRadius;
    }
    return self;
}

- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //四周
    if (self.dashPatternType == QGDashPatternViewTypeRound) {
        [self setRoundedShapeLayerWithStrokeColor:self.lineColor fillColor:[UIColor clearColor] patternViewType:self.dashPatternType bezierPathWithRoundedRect:self.bounds cornerRadius:self.cornerRadius lineWidth:self.lineWidth lineDashPattern:self.lineDashPattern borderFrame:self.bounds];
    }
    //上下左右
    else
    {
        CGRect bezierPathRect = CGRectZero;
        switch (_dashPatternType) {
                //上方
            case QGDashPatternViewTypeTop:
            {
                bezierPathRect = CGRectMake(0, 0, CGRectGetWidth(self.frame), self.lineWidth);
            }
                break;
                //左方
            case QGDashPatternViewTypeLeft:
            {
                bezierPathRect = CGRectMake(0, 0, self.lineWidth, CGRectGetHeight(self.frame));
            }
                break;
                //下方
            case QGDashPatternViewTypeBottom:
            {
                bezierPathRect = CGRectMake(0, CGRectGetHeight(self.frame) - self.lineWidth, CGRectGetWidth(self.frame), self.lineWidth);
            }
                break;
                //右方
            case QGDashPatternViewTypeRight:
            {
                bezierPathRect = CGRectMake(CGRectGetWidth(self.frame) - self.lineWidth, 0, self.lineWidth, CGRectGetHeight(self.frame));
            }
                break;
                
            default:
                break;
        }
        
        [self setRoundedShapeLayerWithStrokeColor:self.lineColor fillColor:[UIColor clearColor]  patternViewType:self.dashPatternType bezierPathWithRoundedRect:bezierPathRect cornerRadius:0 lineWidth:self.lineWidth lineDashPattern:self.lineDashPattern borderFrame:self.bounds];
    }
    
}


/**
 控件四周都是虚线

 @param strokeColor 虚线的颜色
 @param fillColor 虚线空缺的颜色
 @param bezierRect 虚线的路径
 @param cornerRadius 圆角大小
 @param lineWidth 虚线的宽度
 @param lineDashPatter @[@a,@b] a:虚线长度，b:两虚线间隔长度
 @param frame shapeLayer对象的frame
 */
- (void)setRoundedShapeLayerWithStrokeColor:(UIColor *)strokeColor fillColor:(UIColor *)fillColor patternViewType:(QGDashPatternViewType)patternViewType bezierPathWithRoundedRect:(CGRect)bezierRect cornerRadius:(CGFloat)cornerRadius lineWidth:(CGFloat)lineWidth lineDashPattern:(NSArray*)lineDashPatter borderFrame:(CGRect)frame{
    if (!self.dashLayer) {
        self.dashLayer = [CAShapeLayer layer];
        self.dashLayer.strokeColor = strokeColor.CGColor;
        self.dashLayer.fillColor = fillColor.CGColor;
        self.dashLayer.lineWidth = lineWidth;
        self.dashLayer.lineDashPattern = lineDashPatter;
        [self.layer addSublayer:self.dashLayer];
    }
    
    //不重复赋值
    if (!CGRectEqualToRect(self.dashLayer.frame, frame)) {
        self.dashLayer.frame = frame;
    }
    
    
    if (!CGPathIsRect(self.dashLayer.path, &bezierRect)) {
        //生成曲线图
        CGPathRef bezierPath;
        //四周环绕的
        if (patternViewType == QGDashPatternViewTypeRound) {
            bezierPath = [UIBezierPath bezierPathWithRoundedRect:bezierRect cornerRadius:cornerRadius].CGPath;
        }
        else
        {
            UIBezierPath * bezier = [UIBezierPath bezierPath];
            [bezier moveToPoint:CGPointMake(bezierRect.origin.x, bezierRect.origin.y)];
            //竖线
            if (lineWidth == bezierRect.size.width) {
                [bezier addLineToPoint:CGPointMake(bezierRect.origin.x, bezierRect.origin.y + bezierRect.size.height)];
            }
            //横线
            else
            {
                [bezier addLineToPoint:CGPointMake(bezierRect.origin.x + bezierRect.size.width, bezierRect.origin.y)];
            }
            bezierPath = bezier.CGPath;
        }
        
        //避免重复赋值
        if (!CGPathEqualToPath(self.dashLayer.path, bezierPath)) {
            self.dashLayer.path = bezierPath;
        }
    }
}


@end
