//
//  QGShareView.m
//  VirtualPayment
//
//  Created by silicn on 2019/12/5.
//  Copyright © 2019 bo.zhang. All rights reserved.
//





#import "QGShareView.h"
#import "QGDesignatedCornerView.h"
#import "QGBasicUITool.h"
#import "ZHWeChatTool.h"
#import "QGTencentTool.h"


#pragma mark - QGShareItemView

@class QGShareItemView;
typedef void(^ShareItemTapBlock)(QGShareItemView *view);

@interface QGShareItemView : UIView

@property (nonatomic, assign)QGShareItemType type;

@property (nonatomic, copy)NSString *ImageName;

@property (nonatomic, copy)NSString *name;

@property (nonatomic, copy)ShareItemTapBlock itemTapBlock;


/// 指定初始化方法
/// @param name 平台名字
/// @param imageName 平台图标
- (id)initWithType:(QGShareItemType)type Title:(NSString *)name imageName:(NSString *)imageName;

@end

@implementation QGShareItemView

- (id)initWithType:(QGShareItemType)type Title:(NSString *)name imageName:(NSString *)imageName
{
    self = [super init];
    if (self) {
        self.type = type;
        self.name = name;
        self.ImageName = imageName;
        
        // 平台图标
        UIImageView *imageView = [UIImageView creatImageViewWithSuperView:self imageName:imageName];
        [imageView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.width.height.equalTo(@50);
            make.top.left.right.inset(5);
        }];
        // 平台名字
        UILabel *nameLab = [UILabel creatLabelWithSuperView:self title:name titleFontSize:[UIFont qg_regularFontOfSize:12.0f] textColor:UIColor.textColor3333 alignment:NSTextAlignmentCenter];
        [nameLab mas_makeConstraints:^(MASConstraintMaker *make) {
            make.top.equalTo(imageView.mas_bottom).offset(5);
            make.centerX.equalTo(imageView);
            make.bottom.inset(5);
        }];
        
        UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];
        [self addSubview:btn];
        
        [btn mas_makeConstraints:^(MASConstraintMaker *make) {
            make.edges.equalTo(self);
        }];
        
        [btn addTarget:self action:@selector(btnAction:) forControlEvents:UIControlEventTouchUpInside];

    }
    return self;
}

- (void)btnAction:(UIButton *)btn
{
    if (self.itemTapBlock) {
        self.itemTapBlock(self);
    }
}

@end

#pragma mark - QGShareView


#define kScreen_Width ([UIScreen mainScreen].bounds.size.width)

//参考QGShareItemView的约束
#define kItem_Width 60

@interface QGShareView ()

/// 可用分享项
@property (nonatomic, copy)NSArray *ableItems;
/// 背景视图
@property (nonatomic, strong)UIView *shareBackView;
///分享的模型
@property (nonatomic, strong)QGShareModel *shareModel;
/// 分享平台的间隔
@property (nonatomic, assign)CGFloat itemSpace;
/// 上一个布局的平台,用于布局
@property (nonatomic, strong)QGShareItemView *lastItem;

@end


@implementation QGShareView

- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        self.popType = PopViewTypeActionSheet;
        // 默认全平台可分享
        self.ableItems = @[@(QGShareItemTypeWeChat),
                           @(QGShareItemTypeTimeLine),
                           @(QGShareItemTypeQQ),
                           @(QGShareItemTypeQQZone)];
    }
    return self;
}

- (instancetype)initWithItems:(NSArray<NSNumber *> *)items shareModel:(QGShareModel *)model
{
    self = [super init];
    if (self) {
        /// 如果items是空的,则默认全平台.
        /// 不为空,则是items包含的平台
        if (items) {
            self.ableItems = items;
        }
        
        /*
        // 如果微信平台没有安装,则移除微信和朋友圈
        if (![[ZHWeChatTool share] isWXInstalled]) {
           self.ableItems = [self removeWeChatPlatformWithItems:self.ableItems];
        }
        // 如果QQ没有安装,则移除QQ和QQ空间
        if (![[QGTencentTool share] isQQInstalled]) {
            self.ableItems = [self removeWeChatPlatformWithItems:self.ableItems];
        }
         */
        
        self.shareModel = model;
        [self setups];
    }
    return self;
}

- (void)setups
{
    
    /// 设置父视图的背景色为透明色,因为样式略有不同
    self.contentView.backgroundColor = [UIColor clearColor];
    
    ///重新创建圆角白色背景视图
    QGDesignatedCornerView * shareBackView = [[QGDesignatedCornerView alloc] initWithRadius:20 corner:UIRectCornerTopLeft | UIRectCornerTopRight];
    shareBackView.backgroundColor = [UIColor whiteColor];
    [self.contentView addSubview:shareBackView];
    /// 新的背景视图完全覆盖父视图,所有的子视图在
    [shareBackView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.edges.equalTo(self.contentView);
    }];
    
    /// 标题
    UILabel *titleLab = [UILabel creatLabelWithSuperView:shareBackView title:@"分享至" titleFontSize:[UIFont qg_regularFontOfSize:13.0f] textColor:UIColor.textColor6666 alignment:NSTextAlignmentCenter];
    [titleLab mas_makeConstraints:^(MASConstraintMaker *make) {
        make.centerX.equalTo(shareBackView);
        make.top.inset(20);
    }];
    
    /// 关闭按钮
    UIButton *closeBtn = [UIButton creatButtonWithSuperView:shareBackView title:nil titleFont:[UIFont qg_lightFontOfSize:14.0] textColor:UIColor.whiteColor];
    [closeBtn setImage:[UIImage imageNamed:@"share_close"] forState:UIControlStateNormal];
    
    [closeBtn mas_makeConstraints:^(MASConstraintMaker *make) {
        make.centerY.equalTo(titleLab);
        make.right.equalTo(shareBackView).inset(20);
    }];
    /// 关联关闭方法
    [closeBtn addTarget:self action:@selector(closeAction) forControlEvents:UIControlEventTouchUpInside];
    
    // 标题下划线
    UIView *line = [UIView creatViewWithSuperView:shareBackView backgroundColor:UIColor.lineSeparatorColorDCDC];
    [line mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.equalTo(titleLab.mas_bottom).offset(13);
        make.left.right.equalTo(shareBackView);
        make.height.mas_equalTo(0.5);
    }];
    
    /// 没有可分享的平台
    if (self.ableItems.count == 0) {
        UILabel *noPlatformLab = [UILabel creatLabelWithSuperView:shareBackView title:@"暂无可分享平台" titleFontSize:[UIFont qg_mediumFontOfSize:16.0f] textColor:UIColor.textColor3333 alignment:NSTextAlignmentCenter];
        [noPlatformLab mas_makeConstraints:^(MASConstraintMaker *make) {
            make.centerX.equalTo(shareBackView);
            make.top.equalTo(line.mas_bottom).offset(20);
            make.bottom.equalTo(shareBackView).inset(20);
            make.height.mas_equalTo(30);
        }];
        return ;
    };
    
    // 获取平台之间的间距
    self.itemSpace = (kScreen_Width - self.ableItems.count * kItem_Width)/(self.ableItems.count + 1);
    @weakify(self);
    //布局各个平台
    for (NSNumber *type in self.ableItems) {
        QGShareItemView *view = [self viewByType:type];
        [shareBackView addSubview:view];
        // 点击分享平台操作
        view.itemTapBlock = ^(QGShareItemView *view) {
            @strongify(self);
            [self shareToPlatForm:view.type];
        };
        [view mas_makeConstraints:^(MASConstraintMaker *make) {
            if (self.lastItem) {
                make.left.equalTo(self.lastItem.mas_right).offset(self.itemSpace);
            }else{
                make.left.equalTo(shareBackView).inset(self.itemSpace);
            }
            make.top.equalTo(line.mas_bottom).offset(20);
            make.bottom.inset(20);
        }];
        self.lastItem = view;
    }
    //最后一个增加右边距约束
    [self.lastItem mas_updateConstraints:^(MASConstraintMaker *make) {
        make.right.inset(self.itemSpace);
    }];
}

/// 各平台具体的分享方法
- (void)shareToPlatForm:(QGShareItemType)platform
{
    if (self.shareModel == nil) return;
    
    //关闭此页面
    [self closeAction];
        
    //如果外部接管了分享事件，则外部自己处理所有事件
    if (self.takeOverShareBlock) {
        self.takeOverShareBlock(platform);
        return;
    }
    
    //分享中回调
    if (self.sharingBlock) {
        self.sharingBlock(platform);
    }
    
    //复制链接单独处理
    if (platform == QGShareItemTypeCopyLink) {
        UIPasteboard *pasteboard = [UIPasteboard generalPasteboard];
        NSString *shareUrl = self.shareModel.link;
        if (shareUrl.length > 0) {
            pasteboard.string = shareUrl;
            [QGHUDManager showHud:@"已复制到粘贴板"];
        }
        return;
    }
    
    //平台分享
    NSDictionary * shareDic = [self.shareModel convertToShareDic];
    switch (self.shareModel.shareObjType) {
            //多媒体分享
        case QGShareModelObjTypeNormal:
        {
            @weakify(self);
            //触发分享
            [[QGThirdPlatformService shareInstance] shareWithType:platform dictionary:shareDic resultBlock:^(BOOL isSuccess) {
                @strongify(self);
                if (self.resultBlock) {
                    self.resultBlock(isSuccess);
                }
            }];
        }
            break;
            //图片分享
        case QGShareModelObjTypeImage:
        {
            @weakify(self);
            //触发分享
            [[QGThirdPlatformService shareInstance] shareImageWithType:platform dictionary:shareDic resultBlock:^(BOOL isSuccess) {
                @strongify(self);
                if (self.resultBlock) {
                    self.resultBlock(isSuccess);
                }
            }];
        }
            break;
            
        default:
            break;
    }
    
    
    
}

- (void)closeAction
{
    [self dismissView:NULL];
}

/// 移除微信和朋友圈
- (NSArray *)removeWeChatPlatformWithItems:(NSArray *)platforms
{
    NSMutableArray *items = [NSMutableArray arrayWithArray:platforms];
    [items enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if ([obj isKindOfClass:[NSNumber class]]) {
            NSNumber *objNum = (NSNumber *)obj;
            if (objNum.integerValue == QGShareItemTypeWeChat || objNum.integerValue == QGShareItemTypeTimeLine) {
                [items removeObject:obj];
            }
        }
    }];
    return items;
}
/// 移除QQ和QQ空间
- (NSArray *)removeQQPlatformWithItems:(NSArray *)platforms
{
    NSMutableArray *items = [NSMutableArray arrayWithArray:platforms];
    [items enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if ([obj isKindOfClass:[NSNumber class]]) {
            NSNumber *objNum = (NSNumber *)obj;
            if (objNum.integerValue == QGShareItemTypeQQ || objNum.integerValue == QGShareItemTypeQQZone) {
                [items removeObject:obj];
            }
        }
    }];
    return items;
}
/// 根据类型创建视图
- (QGShareItemView *)viewByType:(NSNumber *)typeNum
{
    NSInteger type = typeNum.integerValue;
    if (type == 0) return nil;
    switch (type) {
        case QGShareItemTypeWeChat:
            return [[QGShareItemView alloc]initWithType:QGShareItemTypeWeChat Title:@"微信" imageName:@"share_wechat"];
        case QGShareItemTypeTimeLine:
            return [[QGShareItemView alloc]initWithType:QGShareItemTypeTimeLine Title:@"朋友圈" imageName:@"share_timeline"];
        case QGShareItemTypeQQ:
            return [[QGShareItemView alloc]initWithType:QGShareItemTypeQQ Title:@"QQ" imageName:@"share_qq"];
        case QGShareItemTypeQQZone:
            return [[QGShareItemView alloc]initWithType:QGShareItemTypeQQZone Title:@"QQ空间" imageName:@"share_qqzone"];
        case QGShareItemTypeCopyLink:
            return [[QGShareItemView alloc]initWithType:QGShareItemTypeCopyLink Title:@"复制链接" imageName:@"share_linkcopy"];
        default:
            break;
    }
    return nil;
}

/*
// Only override drawRect: if you perform custom drawing.
// An empty implementation adversely affects performance during animation.
- (void)drawRect:(CGRect)rect {
    // Drawing code
}
*/

@end

@implementation QGShareModel


- (NSDictionary *)convertToShareDic
{
    NSMutableDictionary * shareDic = [NSMutableDictionary dictionaryWithCapacity:0];
    if (self.title) {
        [shareDic setObject:self.title forKey:@"title"];
    }
    if (self.imgUrl) {
        [shareDic setObject:self.imgUrl forKey:@"imgUrl"];
    }
    if (self.image) {
        [shareDic setObject:self.image forKey:@"image"];
    }
    if (self.desc) {
        [shareDic setObject:self.desc forKey:@"desc"];
    }
    if (self.link) {
        [shareDic setObject:self.link forKey:@"link"];
    }
    return shareDic;
}

///// 分享的标题
//@property (nonatomic, copy)NSString * _Nullable title;
///// 分享的图标
//@property (nonatomic, copy)NSString * _Nullable imgUrl;
///// 图片
//@property (nonatomic, copy)UIImage * _Nullable image;
///// 分享的描述文字
//@property (nonatomic, copy)NSString * _Nullable desc;
///// 分享的链接
//@property (nonatomic, copy)NSString * _Nullable link;

@end
